<?php

/* Appora server
 * Copyright (C) 2011 Appora contributors
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

require_once(dirname(__FILE__) . "/../../env.php");
require_once(dirname(__FILE__) . "/MysqlData.php");
require_once(dirname(__FILE__) . "/MysqlConnector.php");
require_once($_DESC_MODULES_DIR . "/IAppDescGetter.php");
require_once($_MODELS_DIR . "/ApplicationDescriptor.php");
require_once($_MODELS_DIR . "/AttributesMask.php");
require_once($_MODELS_DIR . "/Category.php");
require_once($_MODELS_DIR . "/Branch.php");
require_once($_MODELS_DIR . "/Version.php");
require_once($_MODELS_DIR . "/dto/CategoryJSONDTO.php");
require_once($_TOOLS_DIR . "/TypeChecker.php");

/** Factory to retreive application descriptors from a database. */
class MysqlAppDescGetter implements IAppDescGetter {
	const TAG = "MysqlAppDescGetter";

	private $connector;
	private $sDefaultLocale;

	public function __construct($connector, $sDefaultLocale) {
		$this->connector = $connector;
		$this->sDefaultLocale = $sDefaultLocale;
	}

	/** Get an application descriptor from its identifier.
	 * @param undefined $appIdentifier The application identifier.
	 * Type is server dependant.
	 * @param int $iAttributesMask Unused.
	 * @param array $aLocales Array of prefered locales.
	 * @return Application The application or NULL  if not found.
	 * @throws Exception If the connection cannot be opened.
	 */
	public function getApplication($appIdentifier, $iAttributesMask,
			$aLocales = NULL) {
		// Check parameters
		TypeChecker::checkNotNull($appIdentifier, "appIdentifier");
		TypeChecker::checkString($appIdentifier, true, "appIdentifier");
		TypeChecker::checkInt($iAttributesMask, "iAttributesMask");
		try {
			$this->connector->open();
		} catch (Exception $e) {
			throw $e;
		}
		// Use default locale if no locale is given
		if ($aLocales == NULL) {
			$aLocales = array($this->sDefaultLocale);
		}
		// Prepare query
		if (is_string($appIdentifier)) {
			$sEscapedId = mysql_real_escape_string($appIdentifier,
							$this->connector->getConnection());
		} else {
			$sEscapedId = strval($appIdentifier);
		}
		if ($sEscapedId !== FALSE && strlen($sEscapedId) > 0) {
			$application = NULL;
			// Try to get data for each locale and stop on the first found
			for ($i = 0; $i < count($aLocales); $i++) {
				$sLocale = strtolower($aLocales[$i]);
				$sEscapedLocale = mysql_real_escape_string($sLocale);
				if ($sEscapedLocale === FALSE) {
					// Skip invalid locale
					continue;
				}
				$sQuery = "SELECT * FROM applications WHERE "
						. "application_id = \"" . $sEscapedId . "\""
						. " AND locale = \"" . $sEscapedLocale . "\";";
				$data = mysql_query($sQuery,
								$this->connector->getConnection());
				if ($data === FALSE) {
					$error = mysql_error();
					$this->connector->close();
					throw new Exception("Unable to query " . $sQuery . " ("
							. $error . ")");
				}
				// Fetch query
				$row = mysql_fetch_array($data);
				if ($row === FALSE) {
					// No data, try with next locale
					continue;
				} else {
					// Get branches
					$branches = $this->getBranches($appIdentifier, $aLocales);
					// Get result and return it
					$application = new ApplicationDescriptor(
									$row['application_id'],
									$row['name'], $row['description'],
									$row['website'], $branches);
					mysql_free_result($data);
					$this->connector->close();
					return $application;
				}
			}
			// All locales didn't get any result
			$this->connector->close();
			return NULL;
		} else {
			throw new Exception("Unable to escape identifier" .$appIdentifier);
		}
	}

	/** Get an unfilled category.
	 * @param undefined $categoryIdentifier Category identifier. Use NULL
	 * to get default root category.
	 * @param int $iAppAttrMask Mask for application descriptors attibutes.
	 * See AttributesMask,
	 * @param array $aLocales Array of prefered locales.
	 * @param boolean $bAddDefault Wether to append default locale to
	 * given ones.
	 * @return Category Unfilled category or NULL if no data are found.
	 * @throws Exception If connection cannot be opened or id not escapable.
	 */
	public function getCategory($categoryIdentifier, $iAppAttrMask,
			$aLocales = NULL) {
		try {
			$this->connector->open();
		} catch (Exception $e) {
			throw $e;
		}
		// Set root default if required
		if ($categoryIdentifier === NULL) {
			$categoryIdentifier = MysqlData::CATEGORY_ROOT_ID;
		}
		// First, get subcategories
		$aSubcategories = array();
		// Prepare query
		$sEscapedId = NULL;
		if (is_string($categoryIdentifier)) {
			$sEscapedId = mysql_real_escape_string($categoryIdentifier,
							$this->connector->getConnection());
		} else {
			$sEscapedId = strval($categoryIdentifier);
		}
		if ($sEscapedId !== FALSE && strlen($sEscapedId) > 0) {
			$sQuery = "SELECT category_id FROM categories_tree WHERE "
					. "parent_id = " . $sEscapedId . ";";
			$data = mysql_query($sQuery, $this->connector->getConnection());
			if ($data === FALSE) {
				$error = mysql_error();
				$this->connector->close();
				throw new Exception("Unable to query " . $sQuery . " ("
						. $error . ")");
			}
			// Fetch query and get subcategories in $aSubcategories
			while ($row = mysql_fetch_array($data)) {
				$subcategory = $this->getSingleCategory($row['category_id'],
						$aLocales);
				if ($subcategory !== NULL) {
					array_push($aSubcategories, $subcategory);
				}
			}
			// Next, get applications in $aApplications
			$aApplications = array();
			$sQuery = "SELECT application_id FROM applications_categories "
					. "WHERE category_id=\"" . $sEscapedId . "\";";
			$data = mysql_query($sQuery, $this->connector->getConnection());
			if ($data === FALSE) {
				$error = mysql_error();
				$this->connector->close();
				throw new Exception("Unable to query " . $sQuery . " ("
						. $error . ")");
			}
			// Fetch query and get application descriptors
			while ($row = mysql_fetch_array($data)) {
				$descriptor = $this->getApplication($row['application_id'],
								$iAppAttrMask, $aLocales);
				if ($descriptor !== NULL) {
					array_push($aApplications, $descriptor);
				}
			}
			mysql_free_result($data);
			// Finally, get targetted category description in $category
			$category = $this->getSingleCategory($categoryIdentifier,
					$aLocales, $aSubcategories, $aApplications);
			if ($category === NULL) {
				// Generate default category
				$category = new Category(MysqlData::CATEGORY_ROOT_ID,
						NULL, $aSubcategories, $aApplications);
			}
			$this->connector->close();
			return $category;
		} else {
			throw new Exception("Unable to escape identifier "
					. $categoryIdentifier);
		}
	}

	/** Get description of a category.
	 * @param undefined $categoryId Category idendifier
	 * @param array $aLocales Locales to use (default already included if
	 * needed).
	 * @param array $aSubcategories Array of Category to attach to
	 * the category. Default is NULL.
	 * @param array $aApplications Array of ApplicationDescriptor to attach
	 * the category. Default is NULL.
	 * @return A built category or NULL if not found.
	 * @throws Exception If connection cannot be opened or id not escapable.
	 */
	private function getSingleCategory($categoryId, $aLocales,
			$aSubcategories = NULL, $aApplications = NULL) {
		try {
			$this->connector->open();
		} catch (Exception $e) {
			throw $e;
		}
		if (is_string($categoryId)) {
			$sEscapedId = mysql_real_escape_string($categoryId,
							$this->connector->getConnection());
		} else {
			$sEscapedId = strval($categoryId);
		}
		if ($sEscapedId !== FALSE && strlen($sEscapedId) > 0) {
			// Try to get data for each locale and stop on the first found
			for ($i = 0; $i < count($aLocales); $i++) {
				$sLocale = strtolower($aLocales[$i]);
				$sEscapedLocale = mysql_real_escape_string($sLocale,
						$this->connector->getConnection());
				if ($sEscapedLocale === FALSE) {
					// Skip invalid locale
					continue;
				}
				$sQuery = "SELECT * FROM categories_descriptions WHERE "
						. "category_id = \"" . $sEscapedId . "\""
						. " AND locale = \"". $sEscapedLocale . "\";";
				$data = mysql_query($sQuery, $this->connector->getConnection());
				if ($data === FALSE) {
					// No data, try with next locale
					continue;
				}
				$row = mysql_fetch_array($data);
				if ($row !== FALSE) {
					// Create category
					$category = new Category($row['category_id'], $row['name'],
									$aSubcategories, $aApplications);
					mysql_free_result($data);
					$this->connector->close();
					return $category;
				}
			}
			// All locales didn't get any result
			return NULL;
		} else {
			throw new Exception("Unable to escape identifier "
					. $cateqoryIdentifier);
		}
	}

	/** Get branches assotiated to an application
	 * @param string $sApplicationId The application identifier.
	 * @param array $aLocales Prefered locales.
	 * @return array An array of Branch.
	 */
	private function getBranches($sApplicationId, $aLocales) {
		try {
			$this->connector->open();
		} catch (Exception $e) {
			throw $e;
		}
		// Prepare query
		$sEscapedId = mysql_real_escape_string($sApplicationId,
							$this->connector->getConnection());
		$sQuery = "SELECT branch_id FROM branches WHERE "
						. "application_id = \"" . $sEscapedId
						. "\" ORDER BY branches.order ASC;";
		$data = mysql_query($sQuery,
					$this->connector->getConnection());
		if ($data === FALSE) {
			$error = mysql_error();
					$this->connector->close();
					throw new Exception("Unable to query " . $sQuery . " ("
							. $error . ")");
		}
		// Get branches
		$aBranches = array();
		while ($row = mysql_fetch_array($data)) {
			// Get versions and set branch
			$versions = $this->getVersions($sApplicationId, $row['branch_id'],
					$aLocales);
			$branch = $this->getBranch($sApplicationId, $row['branch_id'],
					$aLocales, $versions);
			array_push($aBranches, $branch);
		}
		mysql_free_result($data);
		$this->connector->close();
		return $aBranches;
	}

	/** Get a sigle branch. */
	private function getBranch($sApplicationId, $iBranchId, $aLocales,
			$aVersions) {
		try {
			$this->connector->open();
		} catch (Exception $e) {
			throw $e;
		}
		$sEscapedId = mysql_real_escape_string($sApplicationId,
							$this->connector->getConnection());
		// Try to get data for each locale and stop on the first found
		for ($i = 0; $i < count($aLocales); $i++) {
			$sLocale = strtolower($aLocales[$i]);
			$sEscapedLocale = mysql_real_escape_string($sLocale);
			if ($sEscapedLocale === FALSE) {
				// Skip invalid locale
				continue;
			}
			$sQuery = "SELECT name FROM branches_descriptions WHERE "
					. "application_id = \"" . $sEscapedId . "\""
					. " AND branch_id = " . $iBranchId
					. " AND locale = \"" . $sEscapedLocale . "\";";
			$data = mysql_query($sQuery,
							$this->connector->getConnection());
			if ($data === FALSE) {
				$error = mysql_error();
				$this->connector->close();
				throw new Exception("Unable to query " . $sQuery . " ("
						. $error . ")");
			}
			// Fetch query
			$row = mysql_fetch_array($data);
			if ($row === FALSE) {
				// No data, try with next locale
				continue;
			} else {
				// Get result and return it
				$branch = new Branch($iBranchId, $row['name'], $aVersions);
				mysql_free_result($data);
				$this->connector->close();
				return $branch;
			}
		}
		// All locales didn't get any result
		mysql_free_result($data);
		$this->connector->close();
		return new Branch($iBranchId, "", $aVersions);
	}

	/** Get all versions for a branch. */
	private function getVersions($sApplicationId, $iBranchId, $aLocales) {
		try {
			$this->connector->open();
		} catch (Exception $e) {
			throw $e;
		}
		// Prepare query
		$sEscapedId = mysql_real_escape_string($sApplicationId,
							$this->connector->getConnection());
		$sQuery = "SELECT code FROM versions WHERE application_id = \""
				. $sEscapedId . "\" AND branch_id = " . $iBranchId
				. " ORDER BY code asc;";
		$data = mysql_query($sQuery,
							$this->connector->getConnection());
		if ($data === FALSE) {
			$error = mysql_error();
			$this->connector->close();
			throw new Exception("Unable to query " . $sQuery . " ("
					. $error . ")");
		}
		// Fetch query
		$versions = array();
		// Get versions
		while ($row = mysql_fetch_array($data)) {
			$version = $this->getVersion($sApplicationId, $iBranchId,
					$row['code'], $aLocales);
			array_push($versions, $version);
		}
		mysql_free_result($data);
		$this->connector->close();
		return $versions;
	}

	/** Get a version.
	 * @param string $sApplicationId The application identifier.
	 * @param int $iBranchId The branch identifier.
	 * @param int $iCode The version code.
	 * @param array $aLocales Array of prefered locales.
	 * @return Version The version from database, NULL if not found.
	 * @throws Exception If connection cannot be opened.
	 */
	private function getVersion($sApplicationId, $iBranchId, $iCode,
			$aLocales) {
		try {
			$this->connector->open();
		} catch (Exception $e) {
			throw $e;
		}
		$sEscapedId = mysql_real_escape_string($sApplicationId,
							$this->connector->getConnection());
		// Try to get data for each locale and stop on the first found
		for ($i = 0; $i < count($aLocales); $i++) {
			$sLocale = strtolower($aLocales[$i]);
			$sEscapedLocale = mysql_real_escape_string($sLocale);
			if ($sEscapedLocale === FALSE) {
				// Skip invalid locale
				continue;
			}
			$sQuery = "SELECT * FROM versions_descriptions WHERE "
					. "application_id = \"" . $sEscapedId . "\""
					. "	AND version_code = " . $iCode
					. " AND locale = \"" . $sEscapedLocale . "\";";
			$data = mysql_query($sQuery,
							$this->connector->getConnection());
			if ($data === FALSE) {
				$error = mysql_error();
				$this->connector->close();
				throw new Exception("Unable to query " . $sQuery . " ("
						. $error . ")");
			}
			// Fetch query
			$row = mysql_fetch_array($data);
			if ($row === FALSE) {
				// No data, try with next locale
				continue;
			} else {
				// Get result and return it
				$version = new Version($iCode, $row['name']);
				mysql_free_result($data);
				$this->connector->close();
				return $version;
			}
		}
		// All locales didn't get any result
		$this->connector->close();
		return new Version($iCode, "");
	}
}

?>
