<?php

/* Appora server
 * Copyright (C) 2011 Appora contributors
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/** Representation of a category, grouping applications descriptors and
 * subcategories. A category is a node of the serveur tree.
 * A Category is said to be "filled" if it is fully generated (i.e. all
 * direct children are present, even if not filled). To avoid
 * recursion on subcategories and generating only a subpart of the tree
 * a partial representation can be instanciated.
 */
class Category {

	/** Category unique identifier. Type is left to server implementation. */
	private $identifier;
	/** Human readable name. */
	private $sName;
	/** Array of subcategories.
	 * These are not necessary full (i.e. just name).
	 */
	private $aSubcategoriesIds;
	/** Array of application descriptors in this category. */
	private $aApplicationIds;

	/** Construct a new Category.
	 * @param undefined $identifier Category server unique identifier.
	 * @param String $sName Category human readable name.
	 * @param Array $aSubcategories Array of Category with direct subcategories.
	 * Set to NULL if the category is not filled, use empty array to indicate
	 * that there's no subcategories.
	 * @param Array $aAppDescs Array of ApplicationDescriptor with direct
	 * application children. Set to NULL if the category is not filled,
	 * use empty array to indicate that there's no applications.
	 */
	public function __construct($identifier, $sName,
			$aSubcategories, $aAppDescs) {
		$this->identifier = $identifier;
		if ($sName !== NULL) {
			$this->sName = $sName;
		} else {
			$this->sName = "";
		}
		$this->aSubcategoriesIds = $aSubcategories;
		$this->aApplicationIds = $aAppDescs;
	}

	public function getIdentifier() {
		return $this->identifier;
	}

	/**
	 * Get category name. The name is never NULL.
	 * @return string The category name. May be empty string but not NULL.
	 */
	public function getName() {
		return $this->sName;
	}

	/** Get subcategories. May be NULL if category is not filled. */
	public function getSubcategories() {
		return $this->aSubcategoriesIds;
	}

	/** Get descriptors children applications. May be NULL if category is
	 * not filled.
	 */
	public function getApplications() {
		return $this->aApplicationIds;
	}

	/** Check if the category is filled. If true, you can safely get
	 * subcategories and applications. If false, one of them can be NULL.
	 */
	public function isFilled() {
		return $this->aSubCategories === NULL || $this->aApplicationIds;
	}
}
?>
