;;;; "optiplot.scm" graphs and drives the calculations in "opticompute.scm" -*-scheme-*-
;;; Copyright (C) 2003, 2004, 2005 Aubrey Jaffer

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation, either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

;; http://swiss.csail.mit.edu/~jaffer/FreeSnell

(require 'fresnel-equations)
(require 'video-processing)
(require 'array)
(require 'eps-graph)
(require 'printf)			; for DESCRIBE-PLT

(if (not (defined? real-log10))
    (define real-log10 $log10))

;;; Defaults from graph parameters
(define graph:dimensions '(600 400))
(define graph:font "Times")
(define graph:font-size 13)
(define graph:left-template #f)
(define graph:right-template #f)

(define (ps:viewer path)
  (system (string-append (case (software-type)
			   ((ms-dos windows atarist os/2) "gsview \"")
			   (else "gv \""))
			 path
			 "\"")))

;;; View using ps:viewer or convert using ImageMagick
(define *output-format* #t)
(define (convert-output path suffix)
  (define (replace.suf path suf)
    (string-append (substring path 0 (+ -3 (string-length path)))
		   (if (symbol? suf) (symbol->string suf) suf)))
  (case suffix
    ((#t) (and ps:viewer (ps:viewer path)) path)
    ((eps) path)
    ((ps)
     (let ((psname (replace.suf path "ps")))
       (if (file-exists? psname) (delete-file psname))
       (rename-file path psname)
       psname))
    ((pdf)
     (and (zero? (system (string-append "epstopdf \"" path "\"")))
	  (delete-file path)
	  (replace.suf path "pdf")))
    ((png jpg jpeg)
     (let ((sufname (replace.suf path suffix)))
       (and (zero? (system (string-append "convert \"" path "\" \"" sufname "\"")))
	    (delete-file path)
	    sufname)))
    ((#f) path)
    (else (slib:error 'convert-output 'unknown 'suffix suffix))))

;;; Angle conversions
(define /180pi (/ 180 pi))
(define /pi180 (/ pi 180))
(define (rad->deg rad) (* /180pi rad))
(define (deg->rad rad) (* /pi180 rad))
				 ; Energy	     E =         h*c/L
(define eV<->L			 ; Photon wavelength L =         h*c/E
  (let ((c 299792458)		 ; speed of light    c =           m/s
	(h 6.62606876e-34)	 ; Plank's constant  h =           J.s
	(J/eV 1.602176462e-19))
    (lambda (eV) (/ (* h c) eV J/eV))))

(define (recip-cm x) (/ .01 x))

(define (trace->dash tok)
  (case tok
    ((T_s T_p T) (set-linedash))	;transmission
    ((R_s R_p R) (set-linedash 2 2))	;reflection from top
    ((B_s B_p B) (set-linedash 3 3))	;reflection from bottom
    ((A_s A_p A) (set-linedash 6 2))	;absorption from top
    ((L_s L_P L) (set-linedash 9 3))	;absorption from bottom
    ((n real real-part) (set-linedash))
    ((k imag imag-part ec extinction-coefficient) (set-linedash 4 8))
    ((k/n) (set-linedash 8 4))
    ((r0 reflect reflectance) (set-linedash 2 6))
    (else (slib:warn 'trace->dash 'unknown 'token tok)
	  (set-linedash 5 2))))

(define (count-plots incidents)
  (apply + (map length (map cdr incidents))))

(define (plot-response . args)
  (define db? #f)
  (define rmin #f)
  (define rmax #f)
  (define x-scale-type #f)
  (define incidents '())
  (define fixed-wavelengths '())
  (define IRs '())
  (define markers '())
  (define samples #f)
  (define title #f)
  (define file #f)
  (define dat #f)
  (define wmin #f)
  (define wmax #f)
  (define gmin #f)
  (define gmax #f)
  (define thmin #f)
  (define thmax #f)
  (define optics '())
  (define printf1 #f)
  (define printf* #f)
  (define smooth #f)
  (define swatches '())
  (define stack-colors '())
  (define outtype *output-format*)
  (fluid-let ((graph:right-template graph:right-template)
	      (graph:left-template graph:left-template)
	      (graph:dimensions graph:dimensions)
	      (graph:font-size graph:font-size)
	      (graph:font graph:font))
    (for-each
     (lambda (arg)
       (cond ((not (and (list? arg) (pair? arg)))
	      (slib:warn 'plot-response 'mystery arg))
	     ((and (pair? arg)
		   (pair? (car arg))
		   (number? (caar arg)))
	      (set! optics (cons arg optics)))
	     (else (case (car arg)
		     ((title)
		      (set! title (cadr arg))
		      (set! file (caddr arg)))
		     ((range)
		      (set! rmin (cadr arg))
		      (set! rmax (caddr arg)))
		     ((db)
		      (set! db? #t)
		      (cond ((null? (cdr arg)))
			    (else (set! rmin (cadr arg))
				  (set! rmax (caddr arg)))))
		     ((smooth)
		      (set! smooth (cadr arg)))
		     ((samples)
		      (set! samples (cadr arg)))
		     ((color-swatch)
		      (set! swatches (cons (cdr arg) swatches)))
		     ((stack-colors)
		      (set! stack-colors (append stack-colors (cdr arg))))
		     ((wavelengths)
		      (set! x-scale-type 'wavelengths)
		      (set! wmin (cadr arg))
		      (set! wmax (caddr arg))
		      (set! gmin (cadddr arg))
		      (set! gmax (car (cddddr arg))))
		     ((angles)
		      (set! x-scale-type 'angle)
		      (set! thmin (cadr arg))
		      (set! thmax (caddr arg)))
		     ((logarithmic)
		      (set! x-scale-type 'logarithmic)
		      (set! wmin (cadr arg))
		      (set! wmax (caddr arg))
		      (set! gmin (cadddr arg))
		      (set! gmax (car (cddddr arg))))
		     ((wavenumbers)
		      (set! x-scale-type 'wavenumbers)
		      (set! wmin (cadr arg))
		      (set! wmax (caddr arg))
		      (set! gmin (cadddr arg))
		      (set! gmax (car (cddddr arg))))
		     ((eVs)
		      (set! x-scale-type 'eVs)
		      (set! wmin (caddr arg))
		      (set! wmax (cadr arg))
		      (set! gmin (cadddr arg))
		      (set! gmax (car (cddddr arg))))
		     ((marker)
		      (set! markers (append (cdr arg) markers)))
		     ((incident)
		      (set! incidents (cons (cdr arg) incidents)))
		     ((wavelength)
		      (set! fixed-wavelengths (cons (cdr arg) fixed-wavelengths)))
		     ((IR)
		      (set! IRs (cons (cdr arg) IRs)))
		     ((output-format)
		      (case (length (cdr arg))
			((0) #t)
			((1) (set! outtype (cadr arg)))
			((2) (set! graph:dimensions (cdr arg)))
			((3)
			 (set! outtype (cadr arg))
			 (set! graph:dimensions (cddr arg)))
			(else (slib:error 'split 'output-format arg))))
		     ((output-data)
		      (case (length (cdr arg))
			((1)
			 (set! dat (cadr arg)))
			((2)
			 (set! dat (cadr arg))
			 (set! printf* (cadndr arg)))
			((3)
			 (set! dat (cadr arg))
			 (set! printf1 (caddr arg))
			 (set! printf* (cadddr arg)))
			(else (slib:error 'write-response 'output-format arg))))
		     ((font)
		      (case (length (cdr arg))
			((0) #t)
			((1) (cond ((number? (cadr arg))
				    (set! graph:font-size (cadr arg)))
				   (else (set! graph:font (cadr arg)))))
			((2)
			 (set! graph:font (cadr arg))
			 (set! graph:font-size (caddr arg)))
			((3)
			 (cond ((number? (cadr arg))
				(set! graph:font-size (cadr arg)))
			       (else (set! graph:font (cadr arg))))
			 (set! graph:left-template (caddr arg))
			 (set! graph:right-template (cadddr arg)))
			((4)
			 (set! graph:font (cadr arg))
			 (set! graph:font-size (caddr arg))
			 (set! graph:left-template (cadddr arg))
			 (set! graph:right-template (car (cddddr arg))))
			(else (slib:error 'bad 'font arg))))
		     (else
		      (slib:warn 'plot-response 'unknown arg))))))
     args)
    (set! fixed-wavelengths (reverse fixed-wavelengths))
    (set! incidents (reverse incidents))
    (set! swatches (reverse swatches))
    (set! optics (reverse optics))
    (set! IRs (reverse IRs))
    (cond ((= (length stack-colors) (length optics)))
	  ((< (length stack-colors) (length optics))
	   (if (pair? stack-colors)
	       (slib:warn 'not 'enough 'stack-colors stack-colors))
	   (set! stack-colors
		 (append stack-colors (make-list (length optics) 0))))
	  (else (slib:warn 'ignoring 'extra 'stack-colors stack-colors)))
    (if (not samples) (set! samples 200))
    (if (not (null? swatches))
	(make-colors-swatch file swatches optics))
    (let ((w-transform
	   (case x-scale-type
	     ((wavelengths) identity)
	     ((logarithmic) identity)
	     ((wavenumbers) recip-cm)
	     ((eVs) eV<->L)
	     ((angle) identity)
	     ;;(else (slib:error 'plot-response 'unknown 'x-scale-type x-scale-type))
	     ))
	  (l-transform
	   (case x-scale-type
	     ((wavelengths) identity)
	     ((logarithmic) real-log10)
	     ((wavenumbers) recip-cm)
	     ((eVs) eV<->L)
	     ((angle) identity)
	     ;;(else (slib:error 'plot-response 'unknown 'x-scale-type x-scale-type))
	     ))
	  (path (and file (string-append file ".eps"))))
      (define l-transform-xs!
	(lambda (data)
	  (define xs (subarray data #f 0))
	  (define xmax (+ -1 (car (array-dimensions data))))
	  (array-map! xs l-transform xs)
	  (if (<= (array-ref xs 0) (array-ref xs xmax))
	      data
	      (apply make-shared-array data (lambda (idx jdx) (list (- xmax idx) jdx))
		     (array-dimensions data)))))
      (let ((*trp-spectra*
	     (and x-scale-type
		  (map (lambda (optic)
			 ;;(write file) (display ": ") (display title)
			 (newline)
			 (if (pair? optic) (describe-stack optic))
			 ((if db? dbize-columns! identity)
			  (cond ((null? fixed-wavelengths)
				 (describe-plt wmin wmax x-scale-type samples markers w-transform
					       incidents IRs path title)
				 ((if smooth
				      (lambda (pts) (smooth-columns pts smooth))
				      identity)
				  (l-transform-xs!
				   (compute/wavelength (w-transform wmin) (w-transform wmax)
						       samples incidents optic))))
				(else
				 (compute/angle thmin thmax rad->deg
						samples fixed-wavelengths optic)))))
		       optics)))
	    (ir-spectra
	     (and x-scale-type
		  (null? fixed-wavelengths)
		  (l-transform-xs!
		   (compute-IRs/wavelength (w-transform wmin) (w-transform wmax)
					   samples IRs)))))
	(and dat (write-data-to-file *trp-spectra* dat x-scale-type printf1 printf*))
	(and path x-scale-type *trp-spectra*
	     (if (null? fixed-wavelengths)
		 (plot/wavelength *trp-spectra*
				  stack-colors
				  (l-transform (w-transform gmin))
				  (l-transform (w-transform gmax))
				  x-scale-type
				  rmin rmax db? incidents ir-spectra IRs
				  (map l-transform (map w-transform markers))
				  path title)
		 (plot/angle *trp-spectra* stack-colors
			     (rad->deg thmin) (rad->deg thmax)
			     rmin rmax db? fixed-wavelengths path title)))
	(and path x-scale-type (convert-output path outtype))))))

(define (write-data-to-file Data* path x-scale-type printf1 printf*)
  (define dims (array-dimensions (car Data*)))
  (define (write-data-to-port port)
    (define rlen (+ -1 (car dims)))
    (do ((rdx 0 (+ 1 rdx)))
	((>= rdx rlen) (force-output port))
      (let ((col0 (array-ref (car Data*) rdx 0)))
	(if (eq? 'logarithmic x-scale-type)
	    (set! col0 (real-expt 10 col0)))
	(if printf1
	    (fprintf port printf1 col0)
	    (display col0 port)))
      (for-each (lambda (Data)
		  (do ((cdx 1 (+ 1 cdx)))
		      ((>= cdx (cadr dims)))
		    (cond (printf*
			   (fprintf port printf* (array-ref Data rdx cdx)))
			  (else
			   (display slib:tab port)
			   (display (array-ref Data rdx cdx) port)))))
		Data*)
      (newline port)))
  (if (eqv? #\| (string-ref path 0))
      (let ()
	(require 'posix)
	(write-data-to-port
	 (open-output-pipe (substring path 1 (string-length path)))))
      (call-with-output-file path write-data-to-port)))

(define (x-scale-ruler x-scale-type)
  (case x-scale-type
    ((logarithmic)
     (rule-horizontal bottomedge "log(wavelength/m)" 10))
    ((wavelengths)
     (rule-horizontal bottomedge "wavelength in m" 10))
    ((wavenumbers)
     (rule-horizontal bottomedge "wavenumber" 10))
    ((eVs)
     (rule-horizontal bottomedge "energy in eV" 10))
    ((angle)
     (rule-horizontal bottomedge "angle in degrees" 10))
    (else (slib:error 'x-scale-ruler 'unknown x-scale-type))))

(define (find-encompassing-range* Data* start end)
  (do ((idx (+ -1 end) (+ -1 idx))
       (lst '() (append (map (lambda (dra) (column-range dra idx)) Data*)
			lst)))
      ((< idx start) (apply combine-ranges lst))))

(define (find-encompassing-range Data start end)
  (do ((idx (+ -1 end) (+ -1 idx))
       (lst '() (cons (column-range Data idx) lst)))
      ((< idx start) (apply combine-ranges lst))))

(define (plot-columns Data idx incidents dashizer glyphizer)
  (set! idx (+ -1 idx))
  (map (lambda (incident)
	 (map (lambda (tok)
		(set! idx (+ 1 idx))
		(in-graphic-context
		 (dashizer tok)
		 (plot-column Data 0 idx (glyphizer tok))))
	      (cdr incident)))
       incidents))

(define (plot/angle Data* stack-colors thmin thmax rmin rmax db? fixed-wavelengths path title)
  (define cpi (count-plots fixed-wavelengths))
  (create-postscript-graph
   path graph:dimensions
   (set-font graph:font graph:font-size)
   (if graph:left-template
       (set-margin-templates graph:left-template graph:right-template)
       "")
   (whole-page)
   (setup-plot
    (list thmin thmax)
    (if rmin
	(list rmin rmax)
	(snap-range (find-encompassing-range* Data* 1 (+ 1 cpi)))))
   (title-top "" title)
   (outline-rect plotrect)
   (in-graphic-context (set-linewidth 0) (grid-verticals) (grid-horizontals))
   (x-scale-ruler 'angle)
   (rule-vertical leftedge (if db? "dB" "power ratio") 10)
   (in-graphic-context
    (map (lambda (Data color)
	   (cons (set-color color)
		 (plot-columns Data 1 fixed-wavelengths
			       trace->dash (lambda (tok) 'line))))
	 Data*
	 stack-colors))))

(define (compute/markers markers)
  (if (null? markers)
      #f
      (let ((Data (make-array (A:floR64b 50) (length markers) 2))
	    (idx 0))
	(for-each (lambda (mark)
		    (array-set! Data mark idx 0)
		    (set! idx (+ 1 idx)))
		  markers)
	Data)))

(define (plot/markers Data wmin wmax)
  (if Data
      (in-graphic-context
       (setup-plot (list wmin wmax) '(0 100) 'graphrect)
       (plot-column Data 0 1 'impulse)
       (plot-column Data 0 1 'triup))
      ""))

(define (comma-append strs)
  (cond ((null? strs) "")
	((null? (cdr strs)) (car strs))
	(else
	 (string-append (car strs) ", "
			(comma-append (cdr strs))))))

(define (plot/wavelength Data* stack-colors xmin xmax x-scale-type rmin rmax
			 db? incidents IRData IRs markers file title)
  (define (collect-edges edg)
    (if db?
	(string-append "20*log(" (symbol->string edg) ")")
	(symbol->string edg)))
  (define mdata (compute/markers markers))
  (define ledges '())
  (define redges '())
  (let ((cpi (count-plots incidents)))
    (define tra-plot
      (if (null? incidents)
	  ""
	  (in-graphic-context
	   (setup-plot
	    (list xmin xmax)
	    (if rmin
		(list rmin rmax)
		(snap-range (find-encompassing-range* Data* 1 (+ 1 cpi)))))
	   (title-top "" title)
	   (outline-rect plotrect)
	   (in-graphic-context (set-linewidth 0) (grid-verticals) (grid-horizontals))
	   (x-scale-ruler x-scale-type)
	   (rule-vertical leftedge (if db? "dB" "power ratio") 10)
	   (in-graphic-context
	    (map (lambda (Data color)
		   (cons (set-color color)
			 (plot-columns Data 1 incidents
				       trace->dash (lambda (tok) 'line))))
		 Data*
		 stack-colors))
	   (plot/markers mdata xmin xmax))))
    (define ir-plot
      (if (null? IRs)
	  ""
	  (let ((columns
		 (plot-columns
		  IRData 1 IRs
		  (lambda (tok)
		    (case tok
		      ((n real real-part)
		       (if (not (memq 'n ledges))
			   (set! ledges (cons 'n ledges))))
		      ((k imag imag-part ec extinction-coefficient)
		       (if (not (memq 'k redges))
			   (set! redges (cons 'k redges))))
		      ((k/n)
		       (if (not (memq 'k/n redges))
			   (set! redges (cons 'k/n redges))))
		      ((r0 reflect reflectance)
		       (if (not (memq 'r0 ledges))
			   (set! ledges (cons 'r0 ledges))))
		      (else (slib:warn 'plot/wavelength 'unknown 'token tok)))
		    (trace->dash tok))
		  (lambda (tok) 'line))))
	    (in-graphic-context
	     (setup-plot (list xmin xmax)
			 (snap-range (find-encompassing-range
				      IRData 1 (+ 1 (count-plots IRs)))))
	     (title-top "" (if (null? incidents) title ""))
	     (outline-rect plotrect)
	     (in-graphic-context (set-linewidth 0) (grid-verticals) (grid-horizontals))
	     (x-scale-ruler x-scale-type)
	     (if (null? ledges)
		 ""
		 (in-graphic-context
		  ;;(set-color 'AntiqueBrass)
		  (rule-vertical leftedge
				 (comma-append (map collect-edges ledges))
				 10)))
	     (if (null? redges)
		 ""
		 (in-graphic-context
		  ;;(set-color 'seagreen)
		  (rule-vertical rightedge
				 (comma-append (map collect-edges redges))
				 -10)))
	     (in-graphic-context columns)
	     (plot/markers mdata xmin xmax)))))
    (if (null? IRs)
	(create-postscript-graph file graph:dimensions
				 (set-font graph:font graph:font-size)
				 (if graph:left-template
				     (set-margin-templates graph:left-template graph:right-template)
				     "")
				 (whole-page) tra-plot)
	(create-postscript-graph
	 file (if (null? incidents)
		  graph:dimensions
		  (list (car graph:dimensions)
			(* 2 (cadr graph:dimensions))))
	 (set-font graph:font graph:font-size)
	 (if graph:left-template
	     (set-margin-templates graph:left-template graph:right-template)
	     "")
	 (whole-page)
	 (if (null? incidents) "" (partition-page 1 2))
	 tra-plot
	 ir-plot))))

(define (describe-plt wmin wmax x-scale-type samples markers w-transform
		      incidents IRs path title)
  (and title (printf "\"%s\"\\n" title))
  (printf "Plotting %d samples from %6.3Km :: %6.3Km (%s)\\n"
	  samples (w-transform wmin) (w-transform wmax) x-scale-type)
  (cond ((not (null? markers))
	 (printf " with markers at")
	 (for-each (lambda (marker) (printf " %6.3Km" marker)) markers)
	 (newline)))
  (and path (printf " to \"%s\"\\n" path)))

(define (title str . file)
  (list 'title str (if (null? file) str (car file))))

(define (incident zdegrees . plot-list)
  (cons 'incident (cons (if (number? zdegrees)
			    (list (deg->rad zdegrees))
			    (map deg->rad zdegrees))
			plot-list)))

;;; Each angle th is weighted by 2*pi*sin(th)
;;; The average angle over the ZR cone is (sin(zr)-(zr*cos(zr)))/(1-cos(zr));
;;; very close to 0.65*zr
(define (cone zdegrees . plot-list)
  (define zr (deg->rad zdegrees))
  (cons 'incident (cons (list (/ (- (sin zr) (* zr (cos zr)))
				 (- 1 (cos zr))))
			plot-list)))

(define (wavelength wl . plot-list)
  (cons 'wavelength (cons wl plot-list)))

(define (IR substance . plot-list)
  (cons 'IR (cons substance plot-list)))

(define (samples count)
  (list 'samples count))

(define (marker . ws)
  (cons 'marker ws))

(define (angles ang . args)
  (define ang/4 (/ (deg->rad ang) 4))
  (set! ang (deg->rad ang))
  (set! args (map deg->rad args))
  (case (length args)
    ((0) (list 'angles (- ang ang/4) (+ ang ang/4)))
    ((1) (list 'angles ang (car args)))
    (else (slib:error 'angles ang args))))

(define (wavelengths wl . args)
  (define wl/4 (/ wl 4))
  (case (length args)
    ((0) (let ((rng (list 'wavelengths (- wl wl/4) (+ wl wl/4))))
	   (append rng (cdr rng))))
    ((1) (let ((rng (cons 'wavelengths (cons wl args))))
	   (append rng (cdr rng))))
    ((3) (cons 'wavelengths (cons wl args)))
    (else (slib:error 'wavelengths wl args))))
(define linear wavelengths)		;legacy alias

(define (logarithmic wl . args)
  (case (length args)
    ((0) (let ((rng (list 'logarithmic (/ wl 1.5) (* wl 1.5))))
	   (append rng (cdr rng))))
    ((1) (let ((rng (cons 'logarithmic (cons wl args))))
	   (append rng (cdr rng))))
    ((3) (cons 'logarithmic (cons wl args)))
    (else (slib:error 'logarithmic wl args))))

(define (wavenumbers x . args)
  (case (length args)
    ((0) (let ((rng (list 'wavenumbers (/ x 1.5) (* x 1.5))))
	   (append rng (cdr rng))))
    ((1) (let ((rng (cons 'wavenumbers (cons x args))))
	   (append rng (cdr rng))))
    ((3) (cons 'wavenumbers (cons x args)))
    (else (slib:error 'wavenumbers x args))))
(define wavenumber wavenumbers)		;legacy alias

(define (eVs x . args)
  (case (length args)
    ((0) (let ((rng (list 'eVs (/ x 1.5) (* x 1.5))))
	   (append rng (cdr rng))))
    ((1) (let ((rng (cons 'eVs (cons x args))))
	   (append rng (cdr rng))))
    ((3) (cons 'eVs (cons x args)))
    (else (slib:error 'eVs x args))))
(define eV eVs)				;legacy alias

(define (color-swatch angle tok)
  (list 'color-swatch (deg->rad angle) tok))

(define (stack-colors . args)
  (cons 'stack-colors args))

(define (range low high . units)
  (list (if (null? units) 'range (car units)) low high))

(define (output-format . args)
  (cons 'output-format args))

(define (output-data . args)
  (cons 'output-data args))

(define (font . args)
  (cons 'font args))

(define (db . args)
  (cons 'db args))

(define (smooth sigma)
  (list 'smooth sigma))
;;(trace-all "../FreeSnell/optiplot.scm") (set! *qp-width* 333)
