<?php
/* LARUS BOARD ========================================================
 * Encoded in UTF-8 (micro symbol: µ)
 * Copyright © 2008,2009 by "The Larus Board Team"
 * This file is part of "Larus Board".
 *
 * "Larus Board" is free software: you can redistribute it and/or modify
 * it under the terms of the modified BSD license.
 *
 * "Larus Board" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * You should have received a copy of the modified BSD License
 * along with this package. If not, see
 * <http://download.savannah.gnu.org/releases/larusboard/COPYING.BSD>.
 */
  if ( !defined('__XF_INCLUDE') )
  die('File "'.basename(__FILE__).'" cannot be executed directly!');
  if ( !class_exists('XF') )
  die('Root class is not loaded, yet!');

/**
* XFParser takes care of text processing, tag management and some more
* @package lbbackend
*/
class XFParser {
/**
* @var integer what is the maximum bonus for rating a posting
*/
const MAX_RATE_BONUS = 10;
/**
* @var boolean should the text parser for forum postings allow external uri?
*/
const PARSER_ALLOW_URI = true;
/**
* @var boolean should the text parser for forum postings allow external images?
*/
const PARSER_ALLOW_IMG = true;
/**
* @var boolean should the text parser for forum postings allow external objects?
*/
const PARSER_ALLOW_EMBED = true;
/**
* @var boolean should the text parser for forum postings allow e-mail addresses?
*/
const PARSER_ALLOW_MAIL = true;
/**
* @var string which protocols should be allowed to link to in forum postings?
*/
const PARSER_ALLOW_SCHEME_URI = 'http,https,ftp,ftps';
/**
* @var string which media types can be linked to in embed tags?
*/
const PARSER_ALLOW_SCHEME_EMBED = 'avi,mp3,mpg,swf,svg,xxx,other';
/**
* @var array amount of open tags in message to close them later during parsing
*/
static protected $bbcode_stack = array();

  /**
  * parse a text by several routines for output (usually used on posting text)
  * @param string $a input stream
  * @return string
  * @since 1.0.0
  */
  public static function message($a){
  $o = $a;

  // slice out codeblocks, put in placeholders instead and parse later...
  $codeblocks = array();
  preg_match_all('/\[code(=[^\]]+)*\](.+?)\[\/code\]/siu',$o,$codeblocks,PREG_SET_ORDER);
    if ( sizeof($codeblocks) > 0 ){
      foreach ( $codeblocks as $k=>$v )
      $o = str_replace($v[0],chr(19).'@@CODE_BLOCK:'.$k.'@@'.chr(7),$o);
    }

  // main parser: remove html, parse 'bbcode', sanitize it and replace emoticons...
  $o = str_replace(array('<','>'),array('&lt;','&gt;'),$o);
  $o = preg_replace_callback('/\[([\/]{0,1}[a-z]+)\]/i',array('self','parser_simple_format'),$o);
  $o = preg_replace_callback('/\[(url|img|embed|mail|link|anchor|format)=(.+?)\]/iu',array('self','parser_references'),$o);
  self::parser_tidy($o);
  self::parser_emoticon($o);
  $o = nl2br($o);

  // replace placeholders by codeblocks, if neccessary...
    if ( sizeof($codeblocks) > 0 ){
      foreach ( $codeblocks as $k=>$v )
      $o = str_replace(chr(19).'@@CODE_BLOCK:'.$k.'@@'.chr(7),self::parser_code($v[2],substr($v[1],1)),$o);
    }

  // done ;)
  self::$bbcode_stack = array();
  return $o;
  }

  /**
  * parse simple tags like bold, italic etc.
  * @param string $a input stream
  * @return string
  * @since 1.0.0
  */
  protected static function parser_simple_format($a){
  $isopen = true;
  $a[1] = strtolower($a[1]);
    switch ( $a[1] ){
    case 'b': $o = '<b>'; break;
    case '/b': $o = '</b>'; break;
    case 'i': $o = '<i>'; break;
    case '/i': $o = '</i>'; break;
    case 's': $o = '<s>'; break;
    case '/s': $o = '</s>'; break;
    case 'u': $o = '<u>'; break;
    case '/u': $o = '</u>'; break;
    case 'quote': $o = '<div class="xf_tp_quote">'; break;
    case '/quote': $o = '</div>'; break;
    case 'ul': $o = '<ul>'; break;
    case '/ul': $o = '</ul>'; break;
    case 'li': $o = '<li>'; break;
    case '/li': $o = '</li>'; break;
    case 'notice': $o = '<div class="xf_box_message xf_bm_message xf_bm_warning">'; break;
    case '/notice': $o = '</div>'; break;
    case '/format': $o = '</div>'; break;
    case 'hr': $o = '<hr />'; break;
    default: $o = $a[0]; break;
    }
    if ( substr($a[1],0,1) === '/' ){
    $a[1] = substr($a[1],1);
    $isopen = false;
    }
    if ( !isset(self::$bbcode_stack[$a[1]]) )
    self::$bbcode_stack[$a[1]] = 0;
  ( $isopen ) ? self::$bbcode_stack[$a[1]]++ : self::$bbcode_stack[$a[1]]--;
  return $o;
  }

  /**
  * parse special formatting, e.g. '[format=b,i,bgcolor:#ff0000]'
  * @param string $a input stream
  * @return string
  * @since 1.0.0
  */
  protected static function parser_enhanced_format($a){
  $o = '<div ';
  $a = explode(',',$a);
    if ( sizeof($a) === 1 && substr($a[0],0,5) === 'class' )
    $o .= 'class="xf_tp_usercss__'.preg_replace('/[^a-z0-9_]+/i','',substr($a[0],5,128));
    else{
    $o .= 'style="';
      foreach ( $a as $v ){
      $v = explode(':',$v);
        if ( isset($v[1]) )
        $v[1] = preg_replace('/[^a-z0-9#%]+/u','',$v[1]);
        switch ( strtolower($v[0]) ){
        case 'b': $o .= 'font-weight:bold;'; break;
        case 'i': $o .= 'font-style:italic;'; break;
        case 's': $o .= 'text-decoration:line-through;'; break;
        case 'u': $o .= 'text-decoration:underline;'; break;
        case 'color': $o .= 'color:'.$v[1].';'; break;
        case 'bgcolor': $o .= 'background-color:'.$v[1].';'; break;
        case 'size': $o .= 'font-size:'.$v[1].';'; break;
        case 'width': $o .= 'width:'.$v[1].';'; break;
        case 'inline': $o .= 'display:inline;'; break;
        case 'block': $o .= 'display:inline-block;'; break;
        case 'left': $o .= 'float:left;margin-right:10px;'; break;
        case 'right': $o .= 'float:right;margin-left:10px;'; break;
        }
      }
    }
  $o .= '">';
  return $o;
  }

  /**
  * parse references like images and links
  * @param string $input input stream
  * @return string
  * @since 1.0.0
  */
  protected static function parser_references($input){
  static $scheme = null;
  static $subject = array();
    if ( is_null($scheme) )
    $scheme = explode(',',self::PARSER_ALLOW_SCHEME_URI);
    if ( sizeof($input) !== 3 )
    return '';
  $embed_info = '';
  $input[1] = strtolower($input[1]);
  $input[2] = htmlspecialchars_decode($input[2]);
    if ( $input[1] === 'embed' && substr($input[2],3,1) === ',' ){
    $embed_info = strtolower(substr($input[2],0,3));
    $input[2] = substr($input[2],4);
    }
  $parsed = parse_url($input[2]);
  //D($parsed,$input[2]);
    if ( $input[1] !== 'format' && isset($parsed['scheme']) && !in_array($parsed['scheme'],$scheme,true) )
    return '<span class="xf_tp_denied">#scheme_not_allowed</span>';

    if ( $input[1] === 'url' ){
      if ( !self::PARSER_ALLOW_URI )
      return '<span class="xf_tp_denied">#uri_not_allowed</span>';
    $c = ( isset($parsed['query']) ) ? str_replace('?'.$parsed['query'],'?'.XF::clear_query_string($parsed['query'],'force_decode,strip_html'),$input[2]) : $input[2];
    $c = XF::sanitize_var($c,'uri');
      if ( !$c )
      return '<span class="xf_tp_denied">#uri_miss_check</span>';
      else
      return '<a href="'.$c.'" target="_blank" class="xf_tp_external_link" title="'.XFUI::i18n('be_careful_on_external_content').'"> '.htmlspecialchars($parsed['host']).'</a>';
    } // ==================================================
    elseif ( in_array($input[1],array('img','embed'),true) ){
      if ( !self::PARSER_ALLOW_IMG && $input[1] === 'img' )
      return '<span class="xf_tp_denied">#img_not_allowed</span>';
      if ( !self::PARSER_ALLOW_EMBED && $input[1] === 'embed' )
      return '<span class="xf_tp_denied">#embed_not_allowed</span>';
    $at = ( isset($parsed['path']) ) ? htmlspecialchars(basename($parsed['path'])) : 'unknown_filename';
    //$c = ( isset($parsed['query']) ) ? str_replace('?'.$parsed['query'],'',$input[2]) : $input[2]; // always remove query_string here!
    $c = ( isset($parsed['query']) ) ? str_replace('?'.$parsed['query'],'?'.XF::clear_query_string($parsed['query'],'force_decode,strip_html'),$input[2]) : $input[2];
    $c = XF::sanitize_var($c,'uri');
    $z = ( $input[1] === 'img' )
    ? '<img src="'.$c.'" alt="[IMG] '.$at.'" class="xf_tp_external_image" />'
    : self::parser_embedded_object($embed_info,$c);
      if ( !$c )
      return '<span class="xf_tp_denied">#uri_miss_check</span>';
      else
      return $z;
    } // ==================================================
    elseif ( $input[1] === 'mail' ){
      if ( !self::PARSER_ALLOW_MAIL )
      return '<span class="xf_tp_denied">#mail_not_allowed</span>';
    $c = XF::sanitize_var($parsed['path'],'mail');
      if ( !$c )
      return '<span class="xf_tp_denied">#address_miss_check</span>';
      else
      return '<a href="mailto:'.$c.'" class="xf_tp_external_link" title="[MAIL]">'.htmlspecialchars($c).'</a>';
    } // ==================================================
    elseif ( $input[1] === 'link' ){
      if ( is_numeric($input[2]) ){
        if ( !isset($subject[(int)$input[2]]) ){
        $ssbj = XF::sql_query("SELECT p_subject FROM ".XF::tbl('post_meta')." WHERE p_id = :postid",
        array('postid'=>array($input[2],'int')),__METHOD__,__LINE__);
        $r = $ssbj->fetchObject();
        $ssbj->closeCursor();
        $s = ( is_object($r) )
        ? htmlspecialchars($r->p_subject)
        : intval($input[2]);
        $subject[(int)$input[2]] = $s;
        }
        else
        $s = $subject[(int)$input[2]];
      return '<a href="'.XF::link('topic',array('post'=>intval($input[2]),'single'=>1)).'" class="xf_tp_post_link"> '.$s.'</a>';
      }
      else
      return '<a href="#'.preg_replace('/[^a-z0-9_]+/u','',$input[2]).'" class="xf_tp_anchor_link"> '.htmlspecialchars($input[2]).'</a>';
    } // ==================================================
    elseif ( $input[1] === 'anchor' )
    return '<a name="'.preg_replace('/[^a-z0-9_]+/u','',$input[2]).'"></a>';
    elseif ( $input[1] === 'format' )
    return self::parser_enhanced_format($input[2]);
    else
    return $input[2];
  }

  /**
  * parse code blocks
  * @param string $a input stream
  * @param string $b additional parameters
  * @return string
  * @since 1.0.0
  */
  protected static function parser_code($a,$b = ''){
  $a = trim($a);
    if ( substr($a,0,5) === '<?php' || strtolower($b) === 'php' ){
    $m = 'php';
    $o = str_replace('<br />','',highlight_string($a,true));
    }
    elseif ( strpos($a,'<html>') || strtolower($b) === 'html' ){
    $m = 'html';
    $pattern = array('/&lt;(\w+)/','/&lt;(\/.+?)&gt;/','/([\w\-]+)=(&quot;.+?&quot;)/','/&lt;!\-\-(.+?)\-\-&gt;/su');
    $replace = array(
    '&lt;<span class="xf_tp_code_tag">\1</span>',
    '&lt;<span class="xf_tp_code_tag">\1</span>&gt;',
    '<span class="xf_tp_code_property">\1</span>=<span class="xf_tp_code_value">\2</span>',
    '<span class="xf_tp_code_comment">&lt;!--\1--&gt;</span>',
    );
    $o = preg_replace($pattern,$replace,htmlspecialchars($a));
    }
    elseif ( strtolower($b) === 'css' ){
    $pattern = array('/^(.+?)\{/mu','/^([^:]+):(.+?;)$/mu','/(\/\*.+?\*\/)/su');
    $replace = array(
    '<span class="xf_tp_code_tag">\1</span>{',
    '<span class="xf_tp_code_property">\1</span>:<span class="xf_tp_code_value">\2</span>',
    '<span class="xf_tp_code_comment">\1</span>');
    $o = preg_replace($pattern,$replace,htmlspecialchars($a));
    }
    elseif ( strtolower($b) === 'udiff' ){
    $pattern = array('/^([\+\-]{3}.+?)$/mu','/^@@([0-9, \+\-]+)@@$/mu','/^\+{1}(.*)/mu','/^\-{1}(.*)/mu');
    $replace = array(
    '<span class="xf_tp_code_property">\1</span>',
    '<span class="xf_tp_code_tag">@@ \1 @@</span>',
    '<span class="xf_tp_code_add">+\1</span>',
    '<span class="xf_tp_code_del">-\1</span>');
    $o = preg_replace($pattern,$replace,htmlspecialchars($a));
    }
    else{
    $m = 'plain';
    $o = htmlspecialchars($a);
    }
  $b = ( empty($b) ) ? ':: ['.$m.']' : htmlspecialchars($b);
  return '<div class="xf_tp_code"><div class="xf_tp_code_head">Sourcecode '.$b.'</div>'.$o.'</div>';
  }

  /**
  * clean up message by closing open tags
  * @param string $a input stream
  * @return true
  * @since 1.0.0
  */
  protected static function parser_tidy(&$a){
  arsort(self::$bbcode_stack);
    foreach ( self::$bbcode_stack as $k=>$v ){
      if ( $v <= 0 )
      break;
      switch ( $k ){
      case 'b': $a .= str_repeat('</b>',$v); break;
      case 'i': $a .= str_repeat('</i>',$v); break;
      case 's': $a .= str_repeat('</s>',$v); break;
      case 'u': $a .= str_repeat('</u>',$v); break;
      case 'quote': $a .= str_repeat('</div>',$v); break;
      case 'ul': $a .= str_repeat('</ul>',$v); break;
      case 'li': $a .= str_repeat('</li>',$v); break;
      case 'notice': $a .= str_repeat('</notice>',$v); break;
      case 'format': $a .= str_repeat('</div>',$v); break;
      }
    }
  return true;
  }

  /**
  * parse emoticon strings to images
  * @param string $a input stream
  * @return true
  * @since 1.0.0
  */
  protected static function parser_emoticon(&$a){
  $b = XFCache::get('simple','smilies');
    if ( is_array($b) && sizeof($b) > 0 ){
      foreach ( $b as $v )
      $a = str_replace($v[0],'<img src="'.XF::get_cfg('uri_gfx').'/'.preg_replace('/[\x00-\x1f]+/u','',$v[1]).'" alt="smilie: '.htmlspecialchars($v[1]).'" />',$a);
    }
  return true;
  }

  /**
  * parse embedded objects
  * @param string $a type (can be empty)
  * @param string $b uri
  * @return string
  * @since 1.0.0
  */
  protected static function parser_embedded_object($a,$b){
  static $scheme = null;
    if ( is_null($scheme) )
    $scheme = explode(',',self::PARSER_ALLOW_SCHEME_EMBED);
    if ( $a === '' )
    $a = 'other';
    if ( !in_array($a,$scheme,true) )
    return '<span class="xf_tp_denied">#embed__'.$a.'__not_allowed</span>';
    switch ( $a ){
    case 'avi': $c = 'video/avi'; $d = 1; break;
    case 'mp3': $c = 'audio/mp3'; $d = 1; break;
    case 'mpg': $c = 'video/mpg'; $d = 1; break;
    case 'swf': $c = 'application/x-shockwave-flash'; $d = 2; break;
    case 'svg': $c = 'image/svg+xml'; $d = 0; break;
    case 'xxx': $c = '???'; $d = -1; break;
    default: $c = 'application/octetstream'; $d = 0; break;
    }
  $o = '<div class="xf_tp_external_object"><div class="xf_tp_eo_head">';
  $o .= '&nbsp;&nbsp;<span style="position:relative;top:-10px;">'.htmlspecialchars(urldecode(basename($b))).' ['.$c.']</span></div>';
    if ( $d !== -1 ){
    $o .= '<object data="'.$b.'" type="'.$c.'" width="700" height="400">';
      if ( $d === 2 ){
      $o .= '<param name="movie" value="'.$b.'"></param>';
      $o .= '<param name="allowfullscreen" value="true"></param>';
      $o .= '<param name="allowscriptaccess" value="always"></param>';
      }
      elseif ( $d === 1 ){
      $o .= '<param name="controller" value="true"></param>';
      $o .= '<param name="autoplay" value="false"></param>';
      $o .= '<param name="autoStart" value="0"></param>';
      $o .= '<param name="loop" value="0"></param>';
      }
    }
  $o .= '<a href="'.$b.'" target="_blank">'.htmlspecialchars($b).'</a></object></div>';
  return $o;
  }

  /**
  * query tags and put them to local cache for further operations
  * @param array $a tag id or a name, multiple allowed
  * @return true
  * @since 1.0.0
  */
  public static function tag_query($a = ''){
    try {
      if ( !is_array($a) )
      throw new XFE('tags must be passed as array');
      if ( sizeof($a) === 0 || !$a[0] )
      return false;
      //throw new XFE('no valid tags found');
    } catch ( XFE $E ){ $E->handle(); }
  $a = array_unique($a);
  $asint = array();
  $asstr = array();

  // resolve only values not cached before
    foreach ( $a as $v ){
      if ( is_numeric($v) ){
        if ( !isset(XF::$tags[$v]) )
        $asint[] = intval($v);
      }
      else{
        if ( !array_search($v,XF::$tags) )
        $asstr[] = $v;
      }
    }
  $asstr = self::resolve_tags($asstr);
    if ( is_array($asstr) ){
      foreach ( $asstr as $k=>$v ){
        if ( !isset(XF::$tags[$k]) ){
        XF::$tags[$k] = $v;
        $k = (int)array_search($k,$asint);
          if ( $k > 0 )
          unset($asint[$k]);
        }
      }
    }
  $a = $asint;
  unset($asint,$asstr);

    if ( sizeof($a) > 0 ){
    $stag = XF::sql_query("SELECT t_id,t_name FROM ".XF::tbl('tag_meta')." WHERE t_id IN (".implode(',',$a).")",'',__METHOD__,__LINE__);
      while ( $r = $stag->fetchObject() )
      XF::$tags[(int)$r->t_id] = $r->t_name;
    $stag->closeCursor();
    }
  return true;
  }

  /**
  * resolve any tags to their id and return them
  * @param array $a tag name, multiple allowed
  * @return mixed
  * @since 1.0.0
  */
  protected static function resolve_tags($a){ // resolve tags literal=>numeric
    if ( !is_array($a) || sizeof($a) === 0 )
    return false;
  $b = array();
    foreach ( $a as $k=>$v ){
      if ( strlen($v) === 0 || is_numeric($v) )
      continue;
    $b[$k] = strip_tags(trim($v));
    }
  $b = array_unique($b);
  $q = array();
  $o = array();
    if ( is_null(XF::$sql) ) // quote() needs an PDO-object, therefore initiate connection here
    XF::sql_query("SELECT 1",'-',__METHOD__,__LINE__);
    foreach ( $b as $k=>$v )
    $q[] = XF::$sql->quote($v);
  $stag = XF::sql_query("SELECT t_id,t_name FROM ".XF::tbl('tag_meta')." WHERE t_name IN (".implode(',',$q).")",'',__METHOD__,__LINE__);
    while ( $r = $stag->fetchObject() )
    $o[(int)$r->t_id] = strval($r->t_name);
  $stag->closeCursor();
  return $o;
  }

  /**
  * manage the tags of a posting
  * @param string $postid post id
  * @param string $tags tags (names separated by comma)
  * @param string $flags 'skip_check' does not check for posting existence
  * @return boolean
  * @since 1.0.0
  */
  public static function tag_management($postid,$tags,$flags = ''){
    if ( !is_string($tags) || empty($tags) )
    return true;
  $flags = explode(',',$flags);
    if ( !in_array('skip_check',$flags,true) ){
    $schk = XF::sql_query("SELECT p_id FROM ".XF::tbl('post_meta')." WHERE p_id = :postid",
    array('postid'=>array($postid,'int')),__METHOD__,__LINE__);
      if ( $schk->rowCount() !== 1 )
      return false;
    $schk->closeCursor();
    }
  $tags = explode(',',$tags);
  $map = array(); // mapping of t[key] -> tag_meta.t_id
  $q = array();
    foreach ( $tags as $k=>$v )
    $tags[$k] = preg_replace('/[ ]{2,}/u',' ',strtolower(strip_tags(trim($v))));
  $tags = array_unique($tags);
    foreach ( $tags as $k=>$v ){
      if ( is_numeric($v) || strlen($v) === 0 || strlen($v) > 63 ){
      unset($tags[$k]);
      continue;
      }
    $q[] = XF::$sql->quote($v);
    $map[$k] = 0;
    }

  // resolve tag names to their ids
  $stag = XF::sql_query("SELECT t_id,t_name FROM ".XF::tbl('tag_meta')." WHERE t_name IN (".implode(',',$q).")",'',__METHOD__,__LINE__);
    while ( $r = $stag->fetchObject() ){
    $k = array_search($r->t_name,$tags);
      if ( is_integer($k) )
      $map[$k] = intval($r->t_id);
    }
  $stag->closeCursor();

  // now we have the mapping...
  // TODO: [idle] tag_management() replace this 'dumb' by better 'delta' solution - look for changes instead of purge data and replace by fresh one
    try {
    XF::sql_query("DELETE FROM ".XF::tbl('tag_data')." WHERE td_p_id = :postid",
    array('postid'=>array($postid,'int')),__METHOD__,__LINE__);
      foreach ( $map as $k=>$v ){
        if ( $v === 0 ){ // insert missing tags
        XF::sql_query("INSERT INTO ".XF::tbl('tag_meta')." (t_name) VALUES (:name)",
        array('name'=>array($tags[$k],'str')),__METHOD__,__LINE__);
        $newtagid = XF::sql_lastId('tag_meta.t_id');
          if ( !$newtagid )
          throw new XFE('new tag could not be inserted');
        $map[$k] = $v = $newtagid;
        }
      $r = XF::sql_query("INSERT INTO ".XF::tbl('tag_data')." (td_t_id,td_p_id) VALUES (:tagid,:postid)",
      array('tagid'=>array($v,'int'),'postid'=>array($postid,'int')),__METHOD__,__LINE__);
        if ( !$r )
        throw new XFE('tag "'.htmlspecialchars($tags[$k]).'" could not be attached to posting');
      }
    } catch ( XFE $E ){ $E->handle(); }

  return true;
  }

  /**
  * calculate reply count
  * @param array $a 'tree' from XFCache::topic()
  * @param string $b fetch unapproved '-' or approved '+' posts
  * @return integer
  * @since 1.0.0
  */
  public static function get_reply_count($a,$b){
  $a = array_count_values($a);
    if ( !isset($a['+']) )
    $a['+'] = 0;
    if ( !isset($a['-']) )
    $a['-'] = 0;
  return ( $a[$b] > 1 ) ? $a[$b]-1 : 0;
  }

  /**
  * limit any rating values to multiples of ten
  * @param string $a rating value
  * @return integer
  * @since 1.0.0
  */
  public static function limit_rating($a){
  $b = floor($a/10)*10;
    if ( $b > 100 )
    $b = 100;
    if ( $b < -100 )
    $b = -100;
  return intval($b);
  }

  /**
  * calculate the age of a timestamp
  * @param integer $a input timestamp
  * @param boolean $astext return it as text or array?
  * @return mixed
  * @since 1.0.0
  */
  public static function calculate_age($a,$astext = true){
  $b = intval(XF::vault_query('uts')-$a);
  $o = array();
  $c = array(
  'y'=>31536000,
  'M'=>2678400,
  'd'=>86400,
  'h'=>3600,
  'm'=>60,
  's'=>false
  );
    foreach ( $c as $k=>$v ){
    $x = ( $v ) ? floor($b/$v) : $b;
      if ( $v )
      $b -= $x*$v;
    $o[$k] = $x;
    }
    if ( $astext ){
    $z = '';
      if ( $o['y'] > 0 )
      $z .= $o['y'].' '.XFUI::i18n('year').' ';
      if ( $o['M'] > 0 )
      $z .= $o['M'].' '.XFUI::i18n('month').' ';
      if ( $o['d'] > 0 )
      $z .= $o['d'].' '.XFUI::i18n('day').' ';
      if ( $o['h'] > 0 )
      $z .= $o['h'].' '.XFUI::i18n('hour').' ';
      if ( $o['m'] > 0 )
      $z .= $o['m'].' '.XFUI::i18n('minute').' ';
      if ( $o['s'] > 0 )
      $z .= $o['s'].' '.XFUI::i18n('second');
    }
    else
    $z = array('sec'=>$o['s'],'min'=>$o['m'],'hr'=>$o['h'],'day'=>$o['d'],'month'=>$o['M'],'year'=>$o['y']);
  return $z;
  }

  /**
  * fetch last postid from topic
  * @param array $a resource array from XFCache::topic()
  * @param boolean $b if false, skip unapproved postings
  * @return integer
  * @since 1.0.0
  */
  public static function lastpost(&$a,$b = false){
    if ( !isset($a['tree']) )
    return 0;
  $c = array_reverse($a['tree'],true);
    foreach ( $c as $k=>$v ){
      if ( $v === '-' && !$b )
      continue;
    break;
    }
  return $k;
  }

  /**
  * fetch last postid from topic which is unread
  * @param array $a resource array from XFCache::topic()
  * @return integer
  * @since 1.0.0
  */
  public static function lastunread(&$a){
  $ts = self::fetch_read_tracker_timestamp($a['topicid']);
    if ( $ts === end($a['time']) )
    return 0;
    if ( $ts !== false /*&& $ts !== 0*/ ){
      foreach ( $a['time'] as $k=>$v ){
        if ( $v === $ts && $a['tree'][$k] === '+' )
        return $k;
      }
    }
  return 0;
  }

  /**
  * check whether a posting has been read before during session
  * @param integer $topicid topic id
  * @param integer $last timestamp of last posting in topic
  * @return boolean
  * @since 1.0.0
  */
  public static function is_posting_read($topicid,$last){
  $ts = self::fetch_read_tracker_timestamp($topicid);
    if ( $ts === false /*|| $ts === 0*/ )
    $ts = $last+1;
  return ( $last > $ts ) ? true : false;
  }

  /**
  * create a calendar
  * output: [year][month] => [0]=timestamp_of_month_begin,[1]=composed_date ("yearZmonth")
  * @param integer $min beginning timestamp
  * @param integer $max ending timestamp
  * @return array
  * @since 1.0.0
  */
  public static function calendar($min,$max){
  $min = intval($min);
  $max = intval($max);
  $o = array();
    for ( $y = gmdate('Y',$min) ; $y <= gmdate('Y',$max) ; $y++ ){
    $o[(int)$y] = array();
    $a = ( $y == gmdate('Y',$min) ) ? gmdate('m',$min) : 1;
    $b = ( $y == gmdate('Y',$max) ) ? gmdate('m',$max) : 12;
      for ( $m = $a ; $m <= $b ; $m++ )
      $o[(int)$y][(int)$m] = array(gmmktime(0,0,0,$m,1,$y),(int)$y.'Z'.(int)$m);
    }
  return $o;
  }

  /**
  * do some calulations on an user account. currently it computes 'rating_bonus'.
  * @param integer $a user id
  * @return boolean
  * @since 1.0.0
  */
  public static function user_calculations($a = 0){
    if ( !$a )
    $a = XF::vault_query('current_user_id');
    if ( $a === XF::get_cfg('main_guest_uid') )
    return false;
  $srat = XF::sql_query("SELECT SUM(p_rating) AS sum,COUNT(p_id) AS count FROM ".XF::tbl('post_meta')."
  WHERE p_u_id = :userid AND p_rating != 0 GROUP BY p_u_id",
  array('userid'=>array($a,'int')),__METHOD__,__LINE__);
  $r = $srat->fetchObject();
  $srat->closeCursor();
    if ( is_object($r) ){
    $bonus = round((int)$r->sum/(int)$r->count,0);
    $bonus = round($bonus/100,1)*10;
      if ( $bonus > self::MAX_RATE_BONUS )
      $bonus = self::MAX_RATE_BONUS;
      if ( $bonus < intval(0-self::MAX_RATE_BONUS) )
      $bonus = intval(0-self::MAX_RATE_BONUS);
    }
    else
    $bonus = 0;
  XF::sql_query("UPDATE ".XF::tbl('user')." SET u_rating_bonus = :bonus WHERE u_id = :userid",
  array('userid'=>array($a,'int'),
  'bonus'=>array($bonus,'int')),__METHOD__,__LINE__);
  XF::update_local_cache('user',$a,'u_rating_bonus',$bonus);
  XFCache::purge('user',$a);
  return true;
  }

  /**
  * get the latest timestamp of topic from read tracker
  * @param int $topicid topic id
  * @return mixed
  * @since 1.0.0
  */
  protected static function fetch_read_tracker_timestamp(&$topicid){
  static $cookiedata = null;
  static $isloggedin = null;
    if ( $topicid === 0 )
    return 0;
    if ( isset($_SESSION['xf_read_tracker']) ){
    $ts = ( isset($_SESSION['xf_read_tracker']['local'][$topicid]) )
    ? $_SESSION['xf_read_tracker']['local'][$topicid]
    : $_SESSION['xf_read_tracker']['global'];
    }
    else{
      if ( is_null($isloggedin) )
      $isloggedin = XF::vault_query('is_logged_in');
    $ck = ( is_null($cookiedata) )
    ? XF::sanitize_var(XF::scramble(XF::ifset('cookie',XF::get_cfg('cookie_read_tracker'),0),'decrypt'),'int')
    : $cookiedata;
      if ( !$isloggedin && $ck > 0 )
      $ts = $ck;
      else
      $ts = false; // never mark as 'new', if whether tracker nor cookie is available...
    }
  return $ts;
  }

  /**
  * strip common non-word from strings and compare them
  * @param object $a first input string
  * @param object $b second input string
  * @return boolean
  * @since 1.1.0
  */
  public static function normalized_string_compare($a,$b){
  $a = preg_replace('/[\x00-\x20\.,:;\-]+/','',$a);
  $b = preg_replace('/[\x00-\x20\.,:;\-]+/','',$b);
  return ( sha1($a) === sha1($b) ) ? true : false;
  }

}
?>