/*
 * @(#)SecondPassAnalysis.java     1.5        13 September 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib;

import java.util.Vector;
import net.percederberg.mib.asn1.analysis.DepthFirstAdapter;
import net.percederberg.mib.asn1.node.Node;
import net.percederberg.mib.symbol.Symbol;
import net.percederberg.mib.symbol.TypeSymbol;
import net.percederberg.mib.symbol.ValueSymbol;
import net.percederberg.mib.type.ArrayType;
import net.percederberg.mib.type.CompoundType;
import net.percederberg.mib.type.Constraint;
import net.percederberg.mib.type.EnumerationType;
import net.percederberg.mib.type.IntegerType;
import net.percederberg.mib.type.NamedType;
import net.percederberg.mib.type.ObjectIdentifierType;
import net.percederberg.mib.type.SizeConstraint;
import net.percederberg.mib.type.SnmpObjectType;
import net.percederberg.mib.type.SnmpTrapType;
import net.percederberg.mib.type.StringType;
import net.percederberg.mib.type.Type;
import net.percederberg.mib.type.ValueConstraint;
import net.percederberg.mib.type.ValueRangeConstraint;

/**
 * The second pass analysis of the parse tree. Checks all symbol
 * types and adds the type and object identifier values to the
 * defined symbols.<p>
 *
 * This class is used internally during the MIB creation.
 *
 * @version  1.5
 * @author   Per Cederberg, per@percederberg.net
 */
class SecondPassAnalysis extends DepthFirstAdapter {

   /**
    * The MIB to work on.
    */
   private MIB     mib;

   /**
    * Creates a new second pass analyzer.
    *
    * @param   mib       the mib where the symbols are present
    */
   public SecondPassAnalysis(MIB mib) {
      this.mib = mib;
   }

   /**
    * Sets the type of a type symbol (i.e. a type definition).
    *
    * @param node    the parse tree node
    */
   protected void outTypeAssignment(Node node) {
      TypeSymbol sym = (TypeSymbol)getOut(node.childOfType(Node.TYPEREFERENCE));
      Type       type = (Type)getOut(node.childOfType(Node.TYPE));

      if (sym != null && type != null) {
         sym.setType(type);
      }
   }

   /**
    * Sets the output value to the type object.
    *
    * @param node    the parse tree node
    */
   protected void outType(Node node) {
      setOut(node, getOut(node.childAt(0)));
   }

   /**
    * Sets the output value to the type object.
    *
    * @param node    the parse tree node
    */
   protected void outBuiltinType(Node node) {
      if (node.childrenOfType(Node.TOBJECT) == 1) {
         setOut(node, ObjectIdentifierType.getInstance());
      } else if (node.childrenOfType(Node.INTEGERTYPE) == 1) {
         setOut(node, getOut(node.childAt(0)));
      } else if (node.childrenOfType(Node.STRINGTYPE) == 1) {
         setOut(node, getOut(node.childAt(0)));
      } else if (node.childrenOfType(Node.SEQUENCETYPE) == 1) {
         setOut(node, getOut(node.childAt(0)));
      } else if (node.childrenOfType(Node.SEQUENCEOFTYPE) == 1) {
         setOut(node, getOut(node.childAt(0)));
      } else if (node.childrenOfType(Node.SETTYPE) == 1) {
         setOut(node, getOut(node.childAt(0)));
      } else if (node.childrenOfType(Node.SETOFTYPE) == 1) {
         setOut(node, getOut(node.childAt(0)));
      } else {
         // This case should have been handled already in first pass analysis
         // Here we simply ignore unknown types
      }
   }

   /**
    * Sets the input value for NamedNumberList nodes to an new EnumerationType
    * object. Also sets this nodes output value to that object.
    *
    * @param node    the parse tree node
    */
   protected void inIntegerType(Node node) {
      if (node.childrenOfType(Node.NAMEDNUMBERLIST) > 0) {
         EnumerationType type = new EnumerationType();
         setIn(node.childOfType(Node.NAMEDNUMBERLIST), type);
         setOut(node, type);
      }
   }

   /**
    * Sets the output value to an IntegerType object.
    *
    * @param node    the parse tree node
    */
   protected void outIntegerType(Node node) {
      if (node.childrenOfType(Node.NAMEDNUMBERLIST) == 1) {
         // Nothing to be done
      } else if (node.childrenOfType(Node.CONSTRAINTLIST) == 1) {
         Object obj = getOut(node.childOfType(Node.CONSTRAINTLIST));
         if (obj instanceof ValueRangeConstraint) {
            setOut(node, new IntegerType((ValueRangeConstraint)obj));
         } else {
            mib.addError("integer used with non-integer constraint",
                         node.firstLine(),
                         node.lastLine());
            setOut(node, new IntegerType());
         }
      } else {
         setOut(node, new IntegerType());
      }
   }

   /**
    * Sets the input value for all the children to the same as for
    * this node.
    *
    * @param node    the parse tree node
    */
   protected void inNamedNumberList(Node node) {
      Object obj = getIn(node);
      for (int i = 0; i < node.children(); i++) {
         setIn(node.childAt(i), obj);
      }
   }

   /**
    * Adds the name and number to the input EnumerationType.
    *
    * @param node    the parse tree node
    */
   protected void outNamedNumber(Node node) {
      String           label;
      Number           value;
      EnumerationType  type = (EnumerationType)getIn(node);

      if (node.childrenOfType(Node.SIGNEDNUMBER) > 0) {
         label = (String)getOut(node.childOfType(Node.IDENTIFIER));
         value = (Number)getOut(node.childOfType(Node.SIGNEDNUMBER));
         if (label != null && value != null) {
            type.addValue(label, value);
         }
      } else {
         mib.addWarning("defined number values not supported",
                        node.firstLine(),
                        node.lastLine());
      }
   }

   /**
    * Sets the output value to the number represented by the underlying
    * token. The actual number object may be either an Integer or a Long.
    *
    * @param node    the parse tree node
    */
   protected void outSignedNumber(Node node) {
      Number num = (Number)getOut(node.childOfType(Node.NUMBER));

      if (node.childrenOfType(Node.TMINUS) > 0) {
         if (num instanceof Integer) {
            setOut(node, new Integer(-num.intValue()));
         } else {
            setOut(node, new Long(-num.longValue()));
         }
      } else {
         setOut(node, num);
      }
   }

   /**
    * Sets the output value to a StringType object.
    *
    * @param node    the parse tree node
    */
   protected void outStringType(Node node) {
      Type type = null;
      if (node.childrenOfType(Node.CONSTRAINTLIST) == 1) {
         Object obj = getOut(node.childOfType(Node.CONSTRAINTLIST));
         if (obj instanceof SizeConstraint) {
            type = new StringType((SizeConstraint)obj);
         } else {
            mib.addError("strings can only have size constraints",
                         node.firstLine(),
                         node.lastLine());
         }
      }
      if (type == null) {
         setOut(node, new StringType());
      } else {
         setOut(node, type);
      }
   }

   /**
    * Sets the output value to a CompoundType object, and also sends
    * this object along to the ElementTypeList child.
    *
    * @param node    the parse tree node
    */
   protected void inSequenceType(Node node) {
      Type t = new CompoundType();

      if (node.childrenOfType(Node.ELEMENTTYPELIST) == 1) {
         setIn(node.childOfType(Node.ELEMENTTYPELIST), t);
      }
      setOut(node, t);
   }

   /**
    * Sets the output value to an ArrayType object.
    *
    * @param node    the parse tree node
    */
   protected void outSequenceOfType(Node node) {
      Type t = (Type)getOut(node.childOfType(Node.TYPE));
      if (node.childrenOfType(Node.SIZECONSTRAINT) == 1) {
         SizeConstraint size =
            (SizeConstraint)getOut(node.childOfType(Node.SIZECONSTRAINT));
         setOut(node, new ArrayType(t, size));
      } else {
         setOut(node, new ArrayType(t));
      }
   }

   /**
    * Sets the output value to a CompoundType object, and also sends
    * this object along to the ElementTypeList child.
    *
    * @param node    the parse tree node
    */
   protected void inSetType(Node node) {
      inSequenceType(node);
   }

   /**
    * Sets the output value to an ArrayType object.
    *
    * @param node    the parse tree node
    */
   protected void outSetOfType(Node node) {
      outSequenceOfType(node);
   }

   /**
    * Sets the input value of this node to all it's child nodes.
    *
    * @param node    the parse tree node
    */
   protected void inElementTypeList(Node node) {
      Object obj = getIn(node);
      for (int i = 0; i < node.children(); i++) {
         setIn(node.childAt(i), obj);
      }
   }

   /**
    * Adds a ValueSymbol to the given compound type.
    *
    * @param node    the parse tree node
    */
   protected void outElementType(Node node) {
      if (node.children() == 1) {
         CompoundType compound = (CompoundType)getIn(node);
         Object obj = getOut(node.childAt(0));
         if (obj instanceof ValueSymbol) {
            compound.addComponent((ValueSymbol)obj);
         } else {
            mib.addError("set or sequence element not declared " +
                         "separately as a variable",
                         node.firstLine(),
                         node.lastLine());
         }
      } else {
         mib.addWarning("only set or sequence elements that are simple " +
                        "variables are supported",
                        node.firstLine(),
                        node.lastLine());
      }
   }

   /**
    * Sets the output value to the identified Symbol.
    *
    * @param node    the parse tree node
    */
   protected void outNamedType(Node node) {
      if (node.childrenOfType(Node.IDENTIFIER) == 1) {
         String str = (String)getOut(node.childOfType(Node.IDENTIFIER));
         setOut(node, mib.findSymbol(str));
      }
   }

   /**
    * Sets the output value to a NamedType object.
    *
    * @param node    the parse tree node
    */
   protected void outDefinedType(Node node) {
      Symbol     sym;
      Constraint c  = null;

      if (node.childrenOfType(Node.MODULEREFERENCE) > 0) {
         mib.addWarning("module references unsupported",
                        node.firstLine(),
                        node.lastLine());
      }
      sym = (Symbol)getOut(node.childOfType(Node.TYPEREFERENCE));
      if (node.childrenOfType(Node.CONSTRAINTLIST) > 0) {
         c = (Constraint)getOut(node.childOfType(Node.CONSTRAINTLIST));
      }
      if (sym instanceof TypeSymbol) {
         setOut(node, new NamedType((TypeSymbol)sym, c));
      }
   }

   /**
    * Sets the output value to that of the first child node.
    *
    * @param node    the parse tree node
    */
   protected void outConstraintList(Node node) {
      setOut(node, getOut(node.childAt(0)));
      if (node.children() != 1) {
         mib.addWarning("multiple constraints not supported",
                        node.firstLine(),
                        node.lastLine());
      }
   }

   /**
    * Sets the output value to that of the child node.
    *
    * @param node    the parse tree node
    */
   protected void outConstraint(Node node) {
      setOut(node, getOut(node.childAt(0)));
   }

   /**
    * Sets the output value to a ValueRange, or null.
    *
    * @param node    the parse tree node
    */
   protected void outValueConstraint(Node node) {
      Object obj = null;
      if (node.childrenOfType(Node.VALUERANGE) > 0) {
         obj = getOut(node.childOfType(Node.VALUERANGE));
      } else {
         obj = getOut(node.childOfType(Node.VALUE));
         if (obj instanceof Number) {
            obj = new ValueConstraint((Number)obj);
         } else {
            mib.addWarning("only numeric value constraints are supported",
                           node.firstLine(),
                           node.lastLine());
         }
      }
      setOut(node, obj);
   }

   /**
    * Sets the output value to a ValueRangeConstraint object.
    *
    * @param node    the parse tree node
    */
   protected void outValueRange(Node node) {
      Number lower;
      Number upper;

      lower = (Number)getOut(node.childOfType(Node.LOWERENDPOINT));
      upper = (Number)getOut(node.childOfType(Node.UPPERENDPOINT));
      setOut(node, new ValueRangeConstraint(lower, upper));
   }

   /**
    * Sets the output value to a Number object.
    *
    * @param node    the parse tree node
    */
   protected void outLowerEndPoint(Node node) {
      if (node.childrenOfType(Node.TLESSTHAN) > 0) {
         mib.addWarning("unsupported constraint limit modifier: <",
                        node.firstLine(),
                        node.lastLine());
      }
      if (node.childrenOfType(Node.VALUE) == 1) {
         Object obj = getOut(node.childOfType(Node.VALUE));
         if (obj instanceof Number) {
            setOut(node, obj);
         } else {
            mib.addError("constraint limit not a number",
                         node.firstLine(),
                         node.lastLine());
         }
      }
   }

   /**
    * Sets the output value to a Number object.
    *
    * @param node    the parse tree node
    */
   protected void outUpperEndPoint(Node node) {
      if (node.childrenOfType(Node.TLESSTHAN) > 0) {
         mib.addWarning("unsupported constraint limit modifier: <",
                        node.firstLine(),
                        node.lastLine());
      }
      if (node.childrenOfType(Node.VALUE) == 1) {
         Object obj = getOut(node.childOfType(Node.VALUE));
         if (obj instanceof Number) {
            setOut(node, obj);
         } else {
            mib.addError("constraint limit not a number",
                         node.firstLine(),
                         node.lastLine());
         }
      }
   }

   /**
    * Sets the output value to a SizeConstraint object.
    *
    * @param node    the parse tree node
    */
   protected void outSizeConstraint(Node node) {
      Object obj = getOut(node.childOfType(Node.VALUECONSTRAINT));
      if (obj instanceof ValueRangeConstraint) {
         setOut(node, new SizeConstraint((ValueRangeConstraint)obj));
      } else if (obj instanceof ValueConstraint) {
         setOut(node, new SizeConstraint((ValueConstraint)obj));
      } else {
         mib.addError("size constraint must have value range",
                      node.firstLine(),
                      node.lastLine());
      }
   }

   /**
    * Sets value symbols type information and object id.
    *
    * @param node    the parse tree node
    */
   protected void outValueAssignment(Node node) {
      String       id;
      Type         type;
      Symbol       sym;
      ValueSymbol  value = null;
      Object       obj;
      ObjectId     oid = null;

      id = (String)getOut(node.childOfType(Node.IDENTIFIER));
      type = (Type)getOut(node.childOfType(Node.TYPE));
      obj = getOut(node.childOfType(Node.VALUE));
      sym = mib.findSymbol(id);
      if (sym instanceof ValueSymbol) {
         value = (ValueSymbol)sym;
      }
      if (obj instanceof ObjectId) {
         oid = (ObjectId)obj;
      }
      if (value != null && type != null) {
         value.setType(type);
      }
      if (value != null && oid != null) {
         if (oid.isSimple()) {
            value.setParent(oid.getParentSymbol());
            value.setOID(oid.getValue());
         } else {
            mib.addWarning("cannot handle complex object identifiers: " + oid,
                           node.firstLine(),
                           node.lastLine());
         }
      }

      // Check for hyphenation character ('-')
      for (int i = 0; i < id.length(); i++) {
         if (id.charAt(i) == '-') {
            mib.addWarning("hyphen characters ('-') in identifiers " +
                           "cause trouble in code generation",
                           node.firstLine(),
                           node.lastLine());
            break;
         }
      }
   }

   /**
    * Sets the output value to that of the child node.
    *
    * @param node    the parse tree node
    */
   protected void outValue(Node node) {
      setOut(node, getOut(node.childAt(0)));
   }

   /**
    * Sets the output value to that of the symbol referred.
    *
    * @param node    the parse tree node
    */
   protected void outDefinedValue(Node node) {
      String id = (String)getOut(node.childOfType(Node.IDENTIFIER));
      setOut(node, mib.findSymbol(id));
   }

   /**
    * Sets the output value to that of the child node.
    *
    * @param node    the parse tree node
    */
   protected void outBuiltinValue(Node node) {
      setOut(node, getOut(node.childAt(0)));
   }

   /**
    * Sets the output value to that of the child node.
    *
    * @param node    the parse tree node
    */
   protected void outObjectIdentifierValue(Node node) {
      setOut(node, getOut(node.childAt(0)));
   }

   /**
    * Set the output value to that of the last ObjIdComponent.
    *
    * @param node    the parse tree node
    */
   protected void outObjIdComponentList(Node node) {
      setOut(node, getOut(node.childAt(node.children() - 1)));
   }

   /**
    * Sets the output value to an ObjectId. This value is created with
    * the given input value, and the new value is also passed on to
    * the next sibling.
    *
    * @param node    the parse tree node
    */
   protected void outObjIdComponent(Node node) {
      Node      parent = node.getParent();
      ObjectId  oid = (ObjectId)getIn(node);

      // Find the oid
      if (node.childrenOfType(Node.NUMBER) == 1) {
         Number value = (Number)getOut(node.childOfType(Node.NUMBER));
         oid = new ObjectId(oid, value.intValue());
      } else if (node.childrenOfType(Node.IDENTIFIER) == 1) {
         String id = (String)getOut(node.childOfType(Node.IDENTIFIER));
         Symbol sym = mib.findSymbol(id);
         if (sym instanceof ValueSymbol) {
            if (oid != null) {
               mib.addError("overruled object identifier before " + sym,
                            node.firstLine(),
                            node.lastLine());
            }
            oid = new ObjectId((ValueSymbol)sym);
         } else {
            mib.addError("no object identifier for symbol: " + sym,
                         node.firstLine(),
                         node.lastLine());
         }
      } else {
         mib.addWarning("only object identifiers in either name or number " +
                        "form are handled",
                        node.firstLine(),
                        node.lastLine());
      }

      // Pass the oid on to next child
      setOut(node, oid);
      if (parent.childAfter(node) != null) {
         setIn(parent.childAfter(node), oid);
      }
   }

   /**
    * Sets the output value to the identifer string.
    *
    * @param node    the parse tree node
    */
   protected void outIdentifier(Node node) {
      setOut(node, getOut(node.childOfType(Node.TLCASEFIRST_IDENT)));
   }

   /**
    * Sets the output value to the TypeSymbol represented, or null if
    * not present. If the symbol isn't found an error message will be
    * generated.
    *
    * @param node    the parse tree node
    */
   protected void outTypeReference(Node node) {
      String  id;
      Symbol  sym;

      id = (String)getOut(node.childOfType(Node.TUCASEFIRST_IDENT));
      sym = mib.findSymbol(id);
      if (sym instanceof TypeSymbol) {
         setOut(node, sym);
      } else {
         mib.addError("type reference to non-type symbol: " + sym,
                      node.firstLine(),
                      node.lastLine());
      }
   }

   /**
    * Sets the output value to that of the node child.
    *
    * @param node    the parse tree node
    */
   protected void outNumber(Node node) {
      setOut(node, getOut(node.childOfType(Node.TNUMBER)));
   }

   /**
    * Sets the output value to that of the child node.
    *
    * @param node    the parse tree node
    */
   protected void outDefinedMacroType(Node node) {
      setOut(node, getOut(node.childAt(0)));
   }

   /**
    * Sets the output value to a SnmpObjectType object.
    *
    * @param node    the parse tree node
    */
   protected void outSnmpObjectTypeMacroType(Node node) {
      Type     subtype;
      Integer  access;
      Integer  status;
      String   descr = null;
      Vector   index = null;
      Object   defval = null;

      // Get mandatory values
      subtype = (Type)getOut(node.childOfType(Node.TYPE));
      access = (Integer)getOut(node.childOfType(Node.SNMPACCESSPART));
      status = (Integer)getOut(node.childOfType(Node.SNMPSTATUSPART));
      if (access == null) {
         access = new Integer(SnmpObjectType.NO_ACCESS);
      }
      if (status == null) {
         status = new Integer(SnmpObjectType.OBSOLETE_STATUS);
      }

      // Get optional values
      if (node.childrenOfType(Node.SNMPDESCRPART) == 1) {
         descr = (String)getOut(node.childOfType(Node.SNMPDESCRPART));
      }
      if (node.childrenOfType(Node.SNMPINDEXPART) == 1) {
         index = (Vector)getOut(node.childOfType(Node.SNMPINDEXPART));
      }
      if (node.childrenOfType(Node.SNMPDEFVALPART) == 1) {
         defval = getOut(node.childOfType(Node.SNMPDEFVALPART));
      }

      // Return result
      setOut(node, new SnmpObjectType(subtype,
                                      access.intValue(),
                                      status.intValue(),
                                      descr,
                                      index,
                                      defval));
   }

   /**
    * Sets the output value to a SnmpTrapType object.
    *
    * @param node    the parse tree node
    */
   protected void outSnmpTrapTypeMacroType(Node node) {
      SnmpTrapType type;

      if (node.childrenOfType(Node.SNMPDESCRPART) == 1) {
         String descr = (String)getOut(node.childOfType(Node.SNMPDESCRPART));
         type = new SnmpTrapType(descr);
      } else {
         type = new SnmpTrapType();
      }
      setOut(node, type);
   }

   /**
    * Sets the output value to an Integer containing the access constant
    * value.
    *
    * @param node    the parse tree node
    */
   protected void outSnmpAccessPart(Node node) {
      String str = (String)getOut(node.childOfType(Node.IDENTIFIER));
      if (str.equals("read-only")) {
         setOut(node, new Integer(SnmpObjectType.READ_ACCESS));
      } else if (str.equals("read-write")) {
         setOut(node, new Integer(SnmpObjectType.READ_WRITE_ACCESS));
      } else if (str.equals("write-only")) {
         setOut(node, new Integer(SnmpObjectType.WRITE_ACCESS));
      } else if (str.equals("not-accessible")) {
         setOut(node, new Integer(SnmpObjectType.NO_ACCESS));
      } else {
         mib.addError("invalid snmp access: " + str,
                      node.firstLine(),
                      node.lastLine());
      }
   }

   /**
    * Sets the output value to an Integer containing the status constant
    * value.
    *
    * @param node    the parse tree node
    */
   protected void outSnmpStatusPart(Node node) {
      String str = (String)getOut(node.childOfType(Node.IDENTIFIER));
      if (str.equals("mandatory")) {
         setOut(node, new Integer(SnmpObjectType.MANDATORY_STATUS));
      } else if (str.equals("optional")) {
         setOut(node, new Integer(SnmpObjectType.OPTIONAL_STATUS));
      } else if (str.equals("obsolete")) {
         setOut(node, new Integer(SnmpObjectType.OBSOLETE_STATUS));
      } else {
         mib.addError("invalid snmp status: " + str,
                      node.firstLine(),
                      node.lastLine());
      }
   }

   /**
    * Sets the output value to the character string value.
    * Also removes whitespace before and after newlines.
    *
    * @param node    the parse tree node
    */
   protected void outSnmpDescrPart(Node node) {
      String text = (String)getOut(node.childOfType(Node.TCSTRING));
      String result = "";

      // Removes whitespace before and after newlines.
      int pos = 0;
      while ((pos = text.indexOf('\n')) >= 0) {
         result = result + text.substring(0, pos).trim() + "\n";
         text = text.substring(pos + 1);
      }
      text = text.trim();
      if (text.length() > 0) {
         result = result + text;
      }

      setOut(node, result);
   }

   /**
    * Sets the output value to a vector of all the value symbols found,
    * or null.
    *
    * @param node    the parse tree node
    */
   protected void outSnmpIndexPart(Node node) {
      setOut(node, getOut(node.childOfType(Node.TYPEORVALUELIST)));

   }

   /**
    * Sets the output value to a vector of all the value symbols found,
    * or null.
    *
    * @param node    the parse tree node
    */
   protected void outTypeOrValueList(Node node) {
      Object obj;
      Vector v = new Vector();
      for (int i = 0; i < node.children(); i++) {
         obj = getOut(node.childAt(i));
         if (obj != null) {
            v.addElement(obj);
         }
      }
      if (v.size() > 0) {
         setOut(node, v);
      }
   }

   /**
    * Sets the output value to the value symbol found.
    *
    * @param node    the parse tree node
    */
   protected void outTypeOrValue(Node node) {
      Object obj = getOut(node.childAt(0));
      if (node.childrenOfType(Node.VALUE) > 0) {
         if (obj instanceof ValueSymbol) {
            setOut(node, obj);
         } else {
            mib.addWarning("unsupported index (must be a value field)",
                           node.firstLine(),
                           node.lastLine());
         }
      } else {
         mib.addWarning("type indices not supported",
                        node.firstLine(),
                        node.lastLine());
      }
   }

   /**
    * Sets the output value to that of the value node.
    *
    * @param node    the parse tree node
    */
   protected void outSnmpDefValPart(Node node) {
      setOut(node, getOut(node.childOfType(Node.VALUE)));
   }

   /**
    * Sets the output value to the contents of the string. The string
    * value is processed to have the initial and trailing " removed,
    * and "" converted into ".
    *
    * @param node    the parse tree node
    */
   protected void outTCSTRING(Node node) {
      String text = node.toString();

      // Remove starting and trailing "
      if (text.startsWith("\"")) {
         text = text.substring(1);
      }
      if (text.endsWith("\"")) {
         text = text.substring(0, text.length() - 1);
      }

      // Replace "" with "
      int pos = 0;
      while ((pos = text.indexOf("\"\"", pos)) >= 0) {
         text = text.substring(0, pos) + text.substring(pos + 1);
      }

      // Return the result
      setOut(node, text);
   }

   /**
    * Sets the output value to the token image.
    *
    * @param node    the parse tree node
    */
   protected void outTUCASEFIRST_IDENT(Node node) {
      setOut(node, node.toString());
   }

   /**
    * Sets the output value to the token image.
    *
    * @param node    the parse tree node
    */
   protected void outTLCASEFIRST_IDENT(Node node) {
      setOut(node, node.toString());
   }

   /**
    * Sets the output value to the number represented by the token.
    * The actual number object may be either an Integer or a Long.
    *
    * @param node    the parse tree node
    */
   protected void outTNUMBER(Node node) {
      Number num;

      try {
         num = new Integer(node.toString());
      } catch (Exception e) {
         num = new Long(node.toString());
      }
      setOut(node, num);
   }
}



/**
 * A private class to temporary hold the object identifier values.
 * The object identifiers are ordered in a tree structure, each
 * component only aware of it's parent.
 *
 * @author  Per Cederberg, per@percederberg.net
 */
class ObjectId extends Object {

   /**
    * The parent object id.
    */
   private ObjectId        parent;

   /**
    * The object id symbol.
    */
   private ValueSymbol     symbol;

   /**
    * The object id value.
    */
   private int             value;

   /**
    * Creates a root object identifier.
    *
    * @param   symbol      the child symbol
    */
   public ObjectId(ValueSymbol symbol) {
      this(null, symbol, -1);
   }

   /**
    * Creates a new child object id with the given value.
    *
    * @param   parent     the parent object id
    * @param   value      the id value
    */
   public ObjectId(ObjectId parent, int value) {
      this(parent, null, value);
   }

   /**
    * Creates a new child object id with the given values.
    *
    * @param   parent     the parent object id
    * @param   symbol     the child symbol, or null
    * @param   value      the id value, or -1
    */
   public ObjectId(ObjectId parent, ValueSymbol symbol, int value) {
      this.parent = parent;
      this.symbol = symbol;
      this.value = value;
   }

   /**
    * Checks if this object id is the root.
    *
    * @return true if this is the root object id, or
    *         false otherwise
    */
   public boolean isRoot() {
      return parent == null;
   }

   /**
    * Checks if this object id is the root and has a symbol value.
    *
    * @return true if this is the symbolic root object id, or
    *         false otherwise
    */
   public boolean isSymbolicRoot() {
      return parent == null && symbol != null;
   }

   /**
    * Checks if this object id is simple, i.e. if it's parent is a
    * symbolic root and it contains a single integer value.
    *
    * @return true if this is a simple object id, or
    *         false otherwise
    */
   public boolean isSimple() {
      return !this.isRoot() && parent.isSymbolicRoot() && this.value >= 0;
   }

   /**
    * Returns the parent symbol.
    *
    * @return  the parent symbol, or null for none
    */
   public ValueSymbol getParentSymbol() {
      if (parent != null) {
         return parent.symbol;
      } else {
         return null;
      }
   }

   /**
    * Returns this object id value, or -1 if it has no numeric value.
    *
    * @return the object id value
    */
   public int getValue() {
      return this.value;
   }

   /**
    * Returns a string description of this object identifier.
    *
    * @return a string description of this object
    */
   public String toString() {
      String    base = "";
      if (parent != null) {
         base = parent.toString();
      }
      if (symbol == null) {
         return base + "." + value;
      } else if (value < 0) {
         return base + "." + symbol;
      } else {
         return base + "." + symbol + "(" + value + ")";
      }
   }
}

