/*
 * @(#)Production.java          1.3              31 August 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.asn1.node;

import java.util.Enumeration;
import java.util.Vector;

/**
 * This class represents a production node in the parse tree. A
 * production node is equivalent to a grammar rule (or production),
 * sometimes also referred to as a syntesized node.
 *
 * @version  1.3
 * @author   Per Cederberg, per@percederberg.net
 */
public class Production extends Node {

    /**
     * The child nodes.
     */
    private Vector children = new Vector();

    /**
     * Create a new node without children or parent.
     *
     * @param  kind      the kind of node (a node constant value)
     */
    public Production(int kind) {
        this.kind = kind;
    }

    /**
     * Create a new node with the given parent, but without children.
     *
     * @param  kind       the kind of node
     * @param  parent     the node parent
     */
    public Production(int kind, Node parent) {
        this.kind = kind;
        setParent(parent);
    }

    /**
     * Adds a child to the node. The child will be added last, i.e. as
     * the rightmost child.
     *
     * @param  child   the child to add
     */
    public void addChild(Node child) {
        children.add(child);
    }

    /**
     * Returns the child coming after the given child.
     *
     * @param  child     a child node
     *
     * @return the next child, or null if no next child
     */
    public Node childAfter(Node child) {
        int i = children.indexOf(child);

        if (i + 1 < children.size()) {
            return (Node) children.elementAt(i + 1);
        } else {
            return null;
        }
    }

    /**
     * Returns a child with the given index.
     *
     * @param  index   a child index, from 0 to children() - 1
     *
     * @return  a child node, or null
     */
    public Node childAt(int index) {
        return (Node) children.elementAt(index);
    }

    /**
     * Returns the first child of the given type.
     *
     * @param  type    a node type constant value
     *
     * @return a child node, or null
     */
    public Node childOfType(int type) {
        Enumeration enum = children.elements();
        Node        temp;

        while (enum.hasMoreElements()) {
            temp = (Node) enum.nextElement();
            if (temp.isType(type)) {
                return temp;
            }
        }

        return null;
    }

    /**
     * Returns the number of children to the current node.
     *
     * @return  the number of children, or zero for no children
     */
    public int children() {
        return children.size();
    }

    /**
     * Returns the number of children of a given type to the current node.
     *
     * @param type    the type number
     *
     * @return the number of children, or zero for no children
     */
    public int childrenOfType(int type) {
        Enumeration enum = children.elements();
        Node        temp;
        int         counter = 0;

        while (enum.hasMoreElements()) {
            temp = (Node) enum.nextElement();
            if (temp.isType(type)) {
                counter++;
            }
        }

        return counter;
    }

    /**
     * Finds the first line for the node.
     *
     * @return   the first line, or -1 if no token present
     */
    public int firstLine() {
        if (children() > 0) {
            return childAt(0).firstLine();
        } else {
            return -1;
        }
    }

    /**
     * Finds the first column for the node.
     *
     * @return   the first column, or -1 if no token present
     */
    public int firstColumn() {
        if (children() > 0) {
            return childAt(0).firstColumn();
        } else {
            return -1;
        }
    }

    /**
     * Finds the last line for the node.
     *
     * @return   the last line, or -1 if no token present
     */
    public int lastLine() {
        if (children() > 0) {
            return childAt(children() - 1).lastLine();
        } else {
            return -1;
        }
    }

    /**
     * Finds the last column for the node.
     *
     * @return   the last column, or -1 if no token present
     */
    public int lastColumn() {
        if (children() > 0) {
            return childAt(children() - 1).lastColumn();
        } else {
            return -1;
        }
    }
}
