/*
 * @(#)SizeConstraint.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.type;

/**
 * A class representing a size constraint (for strings).
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
public class SizeConstraint extends Constraint {

    /**
     * Contains the fixed value or null.
     */
    private ValueConstraint fixedValue;

    /**
     * Contains the size value range or null.
     */
    private ValueRangeConstraint range;

    /**
     * Creates a new size constraint with fixed size.
     *
     * @param  value      the fixed size value
     */
    public SizeConstraint(ValueConstraint value) {
        this.fixedValue = value;
        this.range = null;
    }

    /**
     * Creates a new size constraint with a range of values.
     *
     * @param  range    a value constraint of the size range
     */
    public SizeConstraint(ValueRangeConstraint range) {
        this.fixedValue = null;
        this.range = range;
    }

    /**
     * Checks if this constraint equals another constraint.
     *
     * @param obj      the object to compare with
     *
     * @return true if the objects are equal, or
     *         false otherwise
     */
    public boolean equals(Object obj) {
        if (obj instanceof SizeConstraint) {
            SizeConstraint c = (SizeConstraint) obj;
            return equalsValue(this.fixedValue, c.fixedValue) &&
                equalsValue(this.range, c.range);
        } else {
            return false;
        }
    }

    /**
     * Returns a string description of this constraint.
     *
     * @return a string description
     */
    public String toString() {
        if (fixedValue != null) {
            return "SIZE " + fixedValue;
        } else {
            return "SIZE " + range;
        }
    }

    /**
     * Transfers the constraint information from this object to a type
     * converter object. The calling conventions declared in the
     * TypeConverter class are followed.
     *
     * @param   converter     a type converter
     */
    public void transferConstraint(TypeConverter converter) {
        if (fixedValue == null) {
            converter.transferSizeLimits(range.minValue(),
                                         range.maxValue());
        } else {
            converter.transferSizeLimits(fixedValue.value(),
                                         fixedValue.value());
        }
    }

}
