/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \file MLV_sound.h
 *
 * \author Adrien Boussicault
 * \author Marc Zipstein
 *
 * \brief Ce fichier définit les prototypes des fonctions permettant de faire jouer de la musique à l'aide de la librairie MLV.
 */

#ifndef __MLV_SOUND_H__
#define __MLV_SOUND_H__

#ifndef MEMORY_DEBUG
#include <SDL/SDL.h>
#else
#include "memory_debug.h"
#endif

/**
 * \brief Type codant une partion de musique provenant d'un fichier.
 */
typedef struct _MLV_Sheet_music MLV_Sheet_music;

/**
 * \brief Type codant un flux audio.
 */
typedef struct _MLV_Stream MLV_Stream;

/**
 * \brief Cette fonction initialise la libraire MLV pour pouvoir jouer de la musique.
 *
 * Cette fonction doit être executée avant toutes les autes
 * fonctions. L'initialisation permet de dire à la librairie
 * quel est le format et la fréquence d'échatillonage des 
 * pistes sonores qui seront envoyées à la carte audio.
 *
 * La fréquence d'échantillonnage est la fréquence à laquelle
 * un échantillon à été enregistré. Cela veux dire que c'est à
 * cette fréquence qu'un échantillon doit être envoyé à la
 * carte son.
 * Le format correspond à la taille mémoire d'un échantillon.
 *
 * Lorsque vous enregistrez un morceau de musique, vous devez
 * généralement configurer ces deux informations.
 *  
 * \param freq  Fréquence d'échantillonnage. Valeure commune : 11025, 22050 et 44100.
 * \param format Format de la piste audio. 
 *               Les valeurs possibles sont :
 *                   AUDIO_U8 (Unsigned 8-bit samples)
 *                   AUDIO_S8 (Signed 8-bit samples)
 *               Les valeurs suivantes ne sont pas touours supportés par
 *               tous les périphériques audios :
 *                   AUDIO_U16 or AUDIO_U16LSB (unsigned 16-bit little-endian)
 *                   AUDIO_S16 or AUDIO_S16LSB (signed 16-bit little-endian)
 *                   AUDIO_U16MSB (unsigned 16-bit big-endian)
 *                   AUDIO_S16MSB (signed 16-bit big-endian)
 *                   AUDIO_U16SYS = 
 *                         AUDIO_U16LSB or AUDIO_U16MSB
 *                         depending on hardware CPU endianness
 *                   AUDIO_S16SYS = 
 *                        AUDIO_S16LSB or AUDIO_S16MSB
 *                        depending on hardware CPU endianness
 *
 *              Voir la documentation de la SDL :
 *                   http://www.libsdl.org/cgi/docwiki.cgi/SDL_AudioSpec
 */
void MLV_init_sound( int freq, Uint16 format );

/**
 * \brief Ferme proprement les différents périphériques audios.
 */
void MLV_close_sound();

/**
 * \brief Charge un ficher contenant de la musique en mémoire.
 *
 * Cette fonction prends en paramètre le chemin d'acces du 
 * fichier contenant la musique et renvoie un pointeur vers l'espace mémoire où
 * a été chargé la musique.
 *
 * Le format de fichier accepté est le .wav
 *
 * \param file_music Chemin d'accès vers un fichier contenant de la musique codée en format .wav.
 */
MLV_Sheet_music* MLV_load_music( const char* file_music );

/**
 * \brief Ferme un morceau de musique chargé en mémoire.
 *
 * \param music Le morceau de musique à fermer
 */
void MLV_close_music( MLV_Sheet_music* music );

/**
 * \brief Joue une musique qui est chargée en mémoire.
 *
 * \param music Le morceau de musique à jouer.
 * \param volume Le volume sonore.
 */
void MLV_play_music( MLV_Sheet_music* music, float volume );

/**
 *i \brief Arrête toutes les musiques.
 */
void MLV_stop_all_music();

#endif
