# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import pango
import os.path
import hashlib

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Raw image browser
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class RawImageBrowser (object):

  def __init__ (self, datasource):
    self.datasource = datasource
    self.path = datasource.get_attribute ('path')
    self.size = None
    self.md5 = datasource.get_attribute ('md5')
    self.sector_size = 512

  def getsize (self):
    if not self.size:
      self.size = os.path.getsize (self.path)
    return self.size

  def readsector (self, i):
    fp = open (self.path)
    fp.seek (i * self.sector_size)
    data = fp.read (self.sector_size)
    fp.close ()

    return data

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
(DATASOURCE_ICON, DATASOURCE_UID, DATASOURCE_NAME, DATASOURCE_OBJ) = range (4)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Widget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.VBox.__init__ (self)
    self.mediator = mediator
    self.set_border_width (5)
    self.set_spacing (10)
    self.show ()

    # menubar
    menubar = gtk.MenuBar ()
    menubar.show ()
    self.pack_start (menubar, False, False)

    item = gtk.MenuItem ('_File')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    item = gtk.ImageMenuItem (gtk.STOCK_CLOSE)
    item.connect ("activate", self.on_file_close)
    item.show ()
    menu.append (item)

    # toolbar
    self.tooltips = gtk.Tooltips ()

    toolbar = gtk.Toolbar ()
    toolbar.set_style (gtk.TOOLBAR_ICONS)
    toolbar.set_tooltips (True)
    toolbar.show ()
    self.pack_start (toolbar, False, False)

    toolitem = gtk.ToolButton (gtk.STOCK_ADD)
    toolitem.connect ("clicked", self.on_datasource_add)
    toolitem.show ()
    toolitem.set_tooltip (self.tooltips, "Add datasource to case")
    toolbar.insert (toolitem, -1)

    self.remove_toolitem = gtk.ToolButton (gtk.STOCK_REMOVE)
    self.remove_toolitem.set_sensitive (False)
    self.remove_toolitem.connect ("clicked", self.on_datasource_remove)
    self.remove_toolitem.show ()
    self.remove_toolitem.set_tooltip (self.tooltips, "Remove datasource from case")
    toolbar.insert (self.remove_toolitem, -1)

    # datasource listview
    vbox_ds = gtk.VBox ()
    vbox_ds.set_spacing (5)
    vbox_ds.show ()
    self.pack_start (vbox_ds)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.show ()
    vbox_ds.pack_start (sw)

    datastore = gtk.ListStore (gtk.gdk.Pixbuf, str, str, object)
    self.listview = gtk.TreeView (datastore)
    self.listview.set_rules_hint (True)
    selection = self.listview.get_selection ()
    selection.set_mode (gtk.SELECTION_MULTIPLE)
    selection.connect ('changed', self.on_datasource_selected)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'pixbuf', DATASOURCE_ICON)
    self.listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    renderer.set_property ('xalign', 1.0)
    tvcolumn = gtk.TreeViewColumn ('UID')
    tvcolumn.set_resizable (True)
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', DATASOURCE_UID)
    self.listview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ('Name')
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', DATASOURCE_NAME)
    self.listview.append_column (tvcolumn)

    self.listview.show ()
    sw.add (self.listview)

    # datasource properties
    self.properties_table = gtk.Table (4, 4)
    self.properties_table.set_row_spacings (10)
    self.properties_table.set_col_spacings (5)
    self.properties_table.set_sensitive (False)
    self.properties_table.show ()

    label = gtk.Label ()
    label.set_markup ('<b>Name</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    self.properties_table.attach (label, 0, 1, 0, 1, gtk.FILL, 0)

    self.name_entry = gtk.Entry ()
    self.name_entry.connect ('changed', self.on_name_changed)
    self.name_entry.show ()
    self.properties_table.attach (self.name_entry, 1, 4, 0, 1, yoptions=0)

    label = gtk.Label ()
    label.set_markup ('<b>Size</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    self.properties_table.attach (label, 0, 1, 1, 2, gtk.FILL, 0)

    self.size_entry = gtk.Entry ()
    self.size_entry.set_editable (False)
    self.size_entry.show ()
    self.properties_table.attach (self.size_entry, 1, 4, 1, 2, gtk.FILL, 0)

    label = gtk.Label ()
    label.set_markup ('<b>MD5 hash</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    self.properties_table.attach (label, 0, 1, 2, 3, gtk.FILL, 0)

    self.md5_entry = gtk.Entry ()
    self.md5_entry.set_editable (False)
    self.md5_entry.show ()
    self.properties_table.attach (self.md5_entry, 1, 3, 2, 3, gtk.FILL, 0)

    self.md5_button = gtk.Button ('Evaluate')
    self.md5_button.connect ('clicked', self.on_md5_evaluate)
    self.md5_button.show ()
    self.properties_table.attach (self.md5_button, 3, 4, 2, 3, gtk.FILL, 0)
    vbox_ds.pack_start (self.properties_table, False, True)

    label = gtk.Label ()
    label.set_markup ('<b>Path</b>')
    label.set_alignment (1.0, -1)
    label.show ()
    self.properties_table.attach (label, 0, 1, 3, 4, gtk.FILL, 0)

    self.path_filechooserbutton = gtk.FileChooserButton ('Select raw image file')
    self.path_filechooserbutton.connect ('file-set', self.on_choose_path)
    self.path_filechooserbutton.show ()

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.dd')
    filter.add_pattern ('*.raw')
    filter.add_pattern ('*.img')
    self.path_filechooserbutton.set_filter (filter)
    self.properties_table.attach (self.path_filechooserbutton, 1, 4, 3, 4, gtk.FILL, 0)

    # status bar
    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_IN)
    frame.show ()
    self.pack_end (frame, False, False)

    self.status_label = gtk.Label ()
    self.status_label.set_alignment (0, -1)
    self.status_label.show ()
    frame.add (self.status_label)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_case (self, case):
    model = self.listview.get_model ()
    model.clear ()

    # populate datasources from case
    for uid, datasource in sorted ((d.uid, d) for d in case.iter_datasources ()):
      name = datasource.get_attribute ('name')
      model.append ((None, uid, name, datasource))

    # define case
    self.case = case

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set datasource definition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_datasource_definition (self):
    name = self.selected_datasource.get_attribute ('name')
    path = self.selected_datasource.get_attribute ('path')
    size = self.selected_datasource.get_attribute ('size')
    md5 = self.selected_datasource.get_attribute ('md5')

    self.properties_table.set_sensitive (True)
    self.name_entry.set_text (name)
    self.size_entry.set_text (size or '')
    self.md5_entry.set_text (md5 or '')

    if path:
      self.path_filechooserbutton.set_filename (path)

    self.md5_button.set_sensitive (not md5)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Reset datasource definition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def reset_datasource_definition (self):
    self.properties_table.set_sensitive (False)
    self.name_entry.set_text ('')
    self.size_entry.set_text ('')
    self.md5_entry.set_text ('')
    self.path_filechooserbutton.unselect_all ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle datasource selection
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_datasource_selected (self, selection, *args):
    model, pathlist = selection.get_selected_rows ()

    if pathlist:
      self.remove_toolitem.set_sensitive (True)

      if len (pathlist) == 1:
        self.selected_iter = model.get_iter (pathlist[0])
        self.selected_datasource = model.get_value (self.selected_iter, DATASOURCE_OBJ)
        self.set_datasource_definition ()
      else:
        self.selected_iter = None
        self.selected_datasource = None
        self.reset_datasource_definition ()

      datasources = [model.get_value (model.get_iter (path), DATASOURCE_OBJ) for path in pathlist]
      self.mediator.emit ('datasource.selected', datasources)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle path choose
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_choose_path (self, widget, *args):
    path = widget.get_filename ()

    if os.path.exists (path):
      self.selected_datasource.set_attribute ('path', path)

      browser = RawImageBrowser (self.selected_datasource)
      size = browser.getsize ()

      self.size_entry.set_text (str (size))
      self.md5_entry.set_text ('')
      self.md5_button.set_sensitive (True)

      self.selected_datasource.set_attribute ('size', str (size))
      self.selected_datasource.del_attribute ('md5')
      self.case.set_modified (True)
      self.mediator.emit ('case-modified', self.case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle md5 evaluation
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_md5_evaluate (self, widget, *args):
    pct = 0
    self.status_label.set_text ('Calculating hash ... %d %%' % pct)
    self.mediator.call ('ui.flush')
    self.md5_button.set_sensitive (False)

    browser = RawImageBrowser (self.selected_datasource)
    size = int (self.selected_datasource.get_attribute ('size'))
    sectors = size / browser.sector_size
    md5_hash = hashlib.md5 ()

    for i in range (sectors):
      data = browser.readsector (i)
      md5_hash.update (data)

      npct = i * 100 / sectors
      if npct > pct:
        pct = npct
        self.status_label.set_text ('Calculating hash ... %d %%' % pct)
        self.mediator.call ('ui.flush')

    hash_value = md5_hash.hexdigest ()

    self.selected_datasource.set_attribute ('md5', hash_value)
    self.md5_entry.set_text (hash_value)
    self.status_label.set_text ('')
    self.case.set_modified (True)
    self.mediator.emit ('case-modified', self.case)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle name change
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_name_changed (self, widget, *args):
    name = widget.get_text ()

    if self.selected_datasource:
      self.selected_datasource.set_attribute ('name', name)
      model = self.listview.get_model ()
      model.set_value (self.selected_iter, DATASOURCE_NAME, name)

      self.case.set_modified (True)
      self.mediator.emit ('case-modified', self.case)
    
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle datasource addition
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_datasource_add (self, widget, *args):

    fs = gtk.FileChooserDialog ('Select raw image files')
    fs.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
    fs.add_button (gtk.STOCK_OK, gtk.RESPONSE_OK)
    fs.set_select_multiple (True)

    filter = gtk.FileFilter ()
    filter.add_pattern ('*.raw')
    filter.add_pattern ('*.dd')
    filter.add_pattern ('*.img')
    fs.set_filter (filter)

    rc = fs.run ()
    filenames = fs.get_filenames ()
    fs.destroy ()

    if rc != gtk.RESPONSE_OK:
      return

    # create datasources
    model = self.listview.get_model ()

    for filename in filenames:
      name = os.path.splitext (os.path.basename (filename))[0]
      datasource = self.case.create_datasource ()
      datasource.set_attribute ('path', filename)
      datasource.set_attribute ('name', name)

      browser = RawImageBrowser (datasource)
      size = browser.getsize ()
      datasource.set_attribute ('size', str (size))

      self.case.add_datasource (datasource)
      model.append ((None, datasource.uid, name, datasource))

    # set case modified
    if len (filenames) > 0:
      self.case.set_modified (True)
      self.mediator.emit ('case-modified', self.case)
      
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Handle datasource remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_datasource_remove (self, widget, *args):
    model, pathlist = self.listview.get_selection ().get_selected_rows ()
    iters = [model.get_iter (path) for path in pathlist]

    if iters:
      if len (iters) == 1:
        name = model.get_value (iters[0], DATASOURCE_NAME)
        title = 'You are about to remove "%s" datasource. Are you sure?' % name
      else:
        title = 'You are about to remove %d datasources. Are you sure?' % len (iters)

      dialog = gtk.MessageDialog (None,
                    gtk.DIALOG_MODAL,
                    gtk.MESSAGE_QUESTION,
                    gtk.BUTTONS_YES_NO,
                    title)
      rc = dialog.run ()
      dialog.destroy ()

      if rc != gtk.RESPONSE_YES:
        return
        
      for iter in iters:
        datasource = model.get_value (iter, DATASOURCE_OBJ)
        self.case.remove_datasource (datasource)
        model.remove (iter)
 
      self.case.set_modified (True)
      self.mediator.emit ('case-modified', self.case)
      self.remove_toolitem.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief close working area
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_file_close (self, widget, *args):
    print 'NOT IMPLEMENTED'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Icon
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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='''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'spider'
    self.name = 'Spider Data Center'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Spider - Item-datasource association'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.connect ('case.opened', self.on_open)
    self.mediator.connect ('case.created', self.on_open)
    self.mediator.connect ('case.closed', self.on_close)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.call ('ui.working-area.del', self.id)
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Create window for open case
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_open (self, case):
    working_area = self.mediator.call ('ui.working-area.get', self.id)

    if not working_area:
      widget = Widget (self.mediator)
      widget.set_case (case)

      working_area = self.mediator.call ('ui.working-area.new', self.id)
      working_area.set_default_size (600, 400)
      working_area.set_title (self.name)
      working_area.set_icon (self.icon_data)
      working_area.set_widget (widget)

    working_area.show ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Destroy working areas when closing clase
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_close (self, case):
    self.mediator.call ('ui.working-area.close', self.id)
