// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API io module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include <mobius/io/uri_reader.h>

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io module methods
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef io_methods[] =
{
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: data structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
typedef struct
{
  PyObject_HEAD
  mobius::io::uri_reader *obj;
} io_uri_reader_o;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void io_uri_reader_tp_dealloc (io_uri_reader_o *);
static PyObject *io_uri_reader_tp_new (PyTypeObject *, PyObject *, PyObject *);
static PyObject *io_uri_reader_get_size (io_uri_reader_o *);
static PyObject *io_uri_reader_get_eof (io_uri_reader_o *);
static PyObject *io_uri_reader_f_read (io_uri_reader_o *, PyObject *);
static PyObject *io_uri_reader_f_tell (io_uri_reader_o *, PyObject *);
static PyObject *io_uri_reader_f_seek (io_uri_reader_o *, PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef io_uri_reader_getsetters[] =
{
  {
    (char *) "size",
    (getter) io_uri_reader_get_size,
    (setter) 0,
    (char *) "data size in bytes", NULL
  },
  {
    (char *) "eof",
    (getter) io_uri_reader_get_eof,
    (setter) 0,
    (char *) "end of file indicator", NULL
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef io_uri_reader_methods[] =
{
  {
    (char *) "read",
    (PyCFunction) io_uri_reader_f_read,
    METH_VARARGS,
    "read bytes from data"
  },
  {
    (char *) "tell",
    (PyCFunction) io_uri_reader_f_tell,
    METH_VARARGS,
    "get current read position"
  },
  {
    (char *) "seek",
    (PyCFunction) io_uri_reader_f_seek,
    METH_VARARGS,
    "set current read position"
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyTypeObject io_uri_reader_t =
{
  PyObject_HEAD_INIT (0)
  0,                                       		// ob_size
  "io.uri_reader",                         		// tp_name
  sizeof (io_uri_reader_o),                		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) io_uri_reader_tp_dealloc,   		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "reader for URI resources",              		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  io_uri_reader_methods,                   		// tp_methods
  0,                                       		// tp_members
  io_uri_reader_getsetters,                             // tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  io_uri_reader_tp_new                     		// tp_new
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: tp_new
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
io_uri_reader_tp_new (PyTypeObject *type, PyObject *args, PyObject *kwds)
{
  const char *arg_uri;

  if (!PyArg_ParseTuple (args, "s", &arg_uri))
    return NULL;

  io_uri_reader_o *self = (io_uri_reader_o *) type->tp_alloc (type, 0);
  if (self != NULL)
    {
      try
        {
          self->obj = new mobius::io::uri_reader (arg_uri);
        }
      catch (const std::runtime_error& e)
        {
          Py_DECREF (self);
          PyErr_SetString (PyExc_IOError, e.what ());
          self = NULL;
        }
    }

  return (PyObject *) self;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
io_uri_reader_tp_dealloc (io_uri_reader_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
io_uri_reader_get_size (io_uri_reader_o *self)
{
  //!\todo: return correct type
  return PyInt_FromSize_t (self->obj->get_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: eof getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
io_uri_reader_get_eof (io_uri_reader_o *self)
{
  return PyBool_FromLong (self->obj->eof ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: read
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
io_uri_reader_f_read (io_uri_reader_o *self, PyObject *args)
{
  // parse input args
  std::uint64_t arg_size;

  if (!PyArg_ParseTuple (args, "K", &arg_size))
    return NULL;

  // execute C++ function
  mobius::bytearray data = self->obj->read (arg_size);

  return PyString_FromStringAndSize ((const char *) data.data (), data.size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: tell
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
io_uri_reader_f_tell (io_uri_reader_o *self, PyObject *args)
{
  return PyInt_FromSize_t (self->obj->tell ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io.uri_reader: seek
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
io_uri_reader_f_seek (io_uri_reader_o *self, PyObject *args)
{
  // parse input args
  std::int64_t arg_offset;
  int arg_whence = 0;

  if (!PyArg_ParseTuple (args, "L|i", &arg_offset, &arg_whence))
    return NULL;

  // execute C++ function
  mobius::io::seekable_reader::whence w;

  if (arg_whence == 0)
    w = mobius::io::seekable_reader::whence::beginning;

  else if (arg_whence == 1)
    w = mobius::io::seekable_reader::whence::current;

  else if (arg_whence == 2)
    w = mobius::io::seekable_reader::whence::end;
  
  self->obj->seek (arg_offset, w);

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief io module initialisation function
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyMODINIT_FUNC
initio ()
{
  if (PyType_Ready (&io_uri_reader_t) < 0) return;

  PyObject* m = Py_InitModule3 ("io", io_methods, "Mobius Forensic Toolkit API wrapper");
  if (m == NULL)
    return;

  Py_INCREF (&io_uri_reader_t);
  PyModule_AddObject (m, "uri_reader", (PyObject *) &io_uri_reader_t);
}
