// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "filesystem.h"
#include "filesystem_impl_null.h"
#include "filesystem_impl_ext2.h"
#include "filesystem_impl_hfs.h"
#include "filesystem_impl_iso.h"
#include "filesystem_impl_ntfs.h"
#include "filesystem_impl_vfat.h"
#include <mobius/io/sector_reader_adaptor.h>
#include <mobius/partition/partition_system.h>

namespace mobius
{
namespace filesystem
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief default constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
filesystem::filesystem ()
 : impl_ (std::make_shared <filesystem_impl_null> ())
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param ptr smart pointer to implementation object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
filesystem::filesystem (std::shared_ptr <filesystem_impl_base> impl)
 : is_valid_ (true), impl_ (impl)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param reader read stream object
//! \param offset offset from the beginning of the stream
//! \param type filesystem type (default = "autodetect")
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
filesystem::filesystem (
        mobius::io::reader reader,
        std::uint64_t offset,
        const std::string& type)
 : is_valid_ (true)
{
  // detect filesystem type, if type == "autodetect"
  std::string fs_type = type;
  
  if (type == "autodetect")
    {
      if (filesystem_impl_vfat::is_instance (reader, offset))
        fs_type = "vfat";

      else if (filesystem_impl_ntfs::is_instance (reader, offset))
        fs_type = "ntfs";

      else if (filesystem_impl_ext2::is_instance (reader, offset))
        fs_type = "ext2";

      else if (filesystem_impl_hfs::is_instance (reader, offset))
        fs_type = "hfs";

      else if (filesystem_impl_iso::is_instance (reader, offset))
        fs_type = "iso";
    }

  // create impl_ according to fs_type
  if (fs_type == "ext2")
    impl_ = std::make_shared <filesystem_impl_ext2> (reader, offset);

  else if (fs_type == "hfs")
    impl_ = std::make_shared <filesystem_impl_hfs> (reader, offset);

  else if (fs_type == "iso")
    impl_ = std::make_shared <filesystem_impl_iso> (reader, offset);

  else if (fs_type == "ntfs")
    impl_ = std::make_shared <filesystem_impl_ntfs> (reader, offset);

  else if (fs_type == "vfat")
    impl_ = std::make_shared <filesystem_impl_vfat> (reader, offset);

  else
    {
      impl_ = std::make_shared <filesystem_impl_null> ();
      is_valid_ = false;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get filesystems from a given stream
//! \param reader read stream
//! \param sector_size stream sector size in bytes
//! \return list of filesystems detected
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const std::vector <filesystem>
get_filesystems (mobius::io::reader reader, std::uint32_t sector_size)
{
  std::vector <filesystem> filesystems;

  // check if read contains a filesystem
  filesystem fs (reader, 0);
  if (fs)
    filesystems.push_back (fs);

  // scan partitions
  mobius::io::sector_reader_adaptor sector_reader (reader, sector_size);
  mobius::partition::partition_system psystem (sector_reader);

  for (auto p : psystem.get_partitions ())
    {
      mobius::filesystem::filesystem fs (reader, p.get_starting_address ());
      if (fs)
        filesystems.push_back (fs);
    }
    
  return filesystems;
}

} // namespace filesystem
} // namespace mobius
