// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "filesystem_get_metadata.h"
#include "filesystem.h"
#include "filesystem_ext2.h"
#include "filesystem_hfs.h"
#include "filesystem_iso.h"
#include "filesystem_ntfs.h"
#include "filesystem_vfat.h"
#include <mobius/string_functions.h>

namespace mobius
{
namespace filesystem
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get metadata from filesystem_ext2
//! \param obj filesystem_ext2 object
//! \return list of attributes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static const metadata_container_type
get_metadata (const filesystem_ext2& obj)
{
  metadata_container_type metadata = {
    std::make_tuple ("type",
                     "type",
                     "std::string",
                     obj.get_type ()
                    ),
    std::make_tuple ("offset",
                     "offset in bytes",
                     "std::uint64_t",
                     std::to_string (obj.get_offset ())
                    ),
    std::make_tuple ("size",
                     "size",
                     "size_type",
                     std::to_string (obj.get_size ()) + " bytes"
                    ),
    std::make_tuple ("name",
                     "name",
                     "std::string",
                     obj.get_name ()
                    ),
    std::make_tuple ("fs_type",
                     "filesystem type",
                     "std::string",
                     obj.get_fs_type ()
                    ),
    std::make_tuple ("volume_uuid",
                     "volume UUID",
                     "std::string",
                     obj.get_volume_uuid ()
                    ),
    std::make_tuple ("volume_name",
                     "volume name",
                     "std::string",
                     obj.get_volume_name ()
                    ),
    std::make_tuple ("creator_os",
                     "creator operating system",
                     "std::uint32_t",
                     std::to_string (obj.get_creator_os ())
                    ),
    std::make_tuple ("inodes_count",
                     "total inode count",
                     "std::uint32_t",
                     std::to_string (obj.get_inodes_count ())
                    ),
    std::make_tuple ("blocks_count",
                     "total block count",
                     "std::uint64_t",
                     std::to_string (obj.get_blocks_count ())
                    ),
    std::make_tuple ("root_blocks_count",
                     "number of blocks allocated by root",
                     "std::uint64_t",
                     std::to_string (obj.get_root_blocks_count ())
                    ),
    std::make_tuple ("free_blocks_count",
                     "free blocks count",
                     "std::uint64_t",
                     std::to_string (obj.get_free_blocks_count ())
                    ),
    std::make_tuple ("free_inodes_count",
                     "free inode count",
                     "std::uint32_t",
                     std::to_string (obj.get_free_inodes_count ())
                    ),
    std::make_tuple ("first_data_block",
                     "first data block",
                     "std::uint32_t",
                     std::to_string (obj.get_first_data_block ())
                    ),
    std::make_tuple ("block_size",
                     "block size",
                     "std::uint64_t",
                     std::to_string (obj.get_block_size ()) + " bytes"
                    ),
    std::make_tuple ("cluster_size",
                     "cluster size",
                     "std::uint64_t",
                     std::to_string (obj.get_cluster_size ()) + " bytes"
                    ),
    std::make_tuple ("blocks_per_group",
                     "blocks per group",
                     "std::uint32_t",
                     std::to_string (obj.get_blocks_per_group ())
                    ),
    std::make_tuple ("clusters_per_group",
                     "clusters per group",
                     "std::uint32_t",
                     std::to_string (obj.get_clusters_per_group ())
                    ),
    std::make_tuple ("inodes_per_group",
                     "inodes per group",
                     "std::uint32_t",
                     std::to_string (obj.get_inodes_per_group ())
                    ),
    std::make_tuple ("creation_time",
                     "filesystem creation date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_creation_time ())
                    ),
    std::make_tuple ("last_mount_time",
                     "last mount date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_last_mount_time ())
                    ),
    std::make_tuple ("last_write_time",
                     "last write date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_last_write_time ())
                    ),
    std::make_tuple ("last_check_time",
                     "last check date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_last_check_time ())
                    ),
    std::make_tuple ("first_error_time",
                     "first error date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_first_error_time ())
                    ),
    std::make_tuple ("last_error_time",
                     "last error date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_last_error_time ())
                    ),
    std::make_tuple ("mount_count",
                     "number of mounts since the last fsck",
                     "std::uint32_t",
                     std::to_string (obj.get_mount_count ())
                    ),
    std::make_tuple ("max_mount_count",
                     "number of mounts for mandatory fsck",
                     "std::uint32_t",
                     std::to_string (obj.get_max_mount_count ())
                    ),
    std::make_tuple ("state",
                     "file system state",
                     "std::uint16_t",
                     std::to_string (obj.get_state ())
                    ),
    std::make_tuple ("errors",
                     "action when detecting errors",
                     "std::uint16_t",
                     std::to_string (obj.get_errors ())
                    ),
    std::make_tuple ("error_count",
                     "number of errors",
                     "std::uint32_t",
                     std::to_string (obj.get_error_count ())
                    ),
    std::make_tuple ("minor_revision_level",
                     "minor revision level",
                     "std::uint16_t",
                     std::to_string (obj.get_minor_revision_level ())
                    ),
    std::make_tuple ("check_interval",
                     "max check interval time",
                     "std::uint32_t",
                     std::to_string (obj.get_check_interval ()) + "seconds"
                    ),
    std::make_tuple ("revision_level",
                     "revision level",
                     "std::uint32_t",
                     std::to_string (obj.get_revision_level ())
                    ),
    std::make_tuple ("def_resuid",
                     "default uid for reserved blocks",
                     "std::uint16_t",
                     std::to_string (obj.get_def_resuid ())
                    ),
    std::make_tuple ("def_resgid",
                     "default gid for reserved blocks",
                     "std::uint16_t",
                     std::to_string (obj.get_def_resgid ())
                    ),
    std::make_tuple ("first_inode",
                     "first non-reserved inode",
                     "std::uint32_t",
                     std::to_string (obj.get_first_inode ())
                    ),
    std::make_tuple ("inode_size",
                     "inode size",
                     "std::uint16_t",
                     std::to_string (obj.get_inode_size ()) + " bytes"
                    ),
    std::make_tuple ("block_group_number",
                     "block group number of this superblock",
                     "std::uint16_t",
                     std::to_string (obj.get_block_group_number ())
                    ),
    std::make_tuple ("feature_compat",
                     "compatible features flags",
                     "std::uint32_t",
                     std::to_string (obj.get_feature_compat ())
                    ),
    std::make_tuple ("feature_incompat",
                     "incompatible features flags",
                     "std::uint32_t",
                     std::to_string (obj.get_feature_incompat ())
                    ),
    std::make_tuple ("feature_read_only",
                     "readonly compatible features flags",
                     "std::uint32_t",
                     std::to_string (obj.get_feature_read_only ())
                    ),
    std::make_tuple ("last_mount_point",
                     "last mounting point",
                     "std::string",
                     obj.get_last_mount_point ()
                    ),
    std::make_tuple ("journal_uuid",
                     "journal superblock UUID",
                     "std::string",
                     obj.get_journal_uuid ()
                    ),
    std::make_tuple ("journal_inode",
                     "journal inode number",
                     "std::uint32_t",
                     std::to_string (obj.get_journal_inode ())
                    ),
    std::make_tuple ("journal_device",
                     "journal device number",
                     "std::uint32_t",
                     std::to_string (obj.get_journal_device ())
                    ),
    std::make_tuple ("default_hash_version",
                     "default hash algorithm",
                     "std::uint8_t",
                     std::to_string (obj.get_default_hash_version ())
                    ),
    std::make_tuple ("default_mount_options",
                     "default mount options",
                     "std::uint32_t",
                     std::to_string (obj.get_default_mount_options ())
                    ),
    std::make_tuple ("mount_options",
                     "mount options",
                     "std::string",
                     obj.get_mount_options ()
                    ),
    std::make_tuple ("kib_written",
                     "KiB written into this filesystem",
                     "std::uint32_t",
                     std::to_string (obj.get_kib_written ())
                    ),
    std::make_tuple ("checksum",
                     "superblock checksum",
                     "std::uint32_t",
                     std::to_string (obj.get_checksum ())
                    ),
  };

  return metadata;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get metadata from filesystem_hfs
//! \param obj filesystem_hfs object
//! \return list of attributes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static const metadata_container_type
get_metadata (const filesystem_hfs& obj)
{
  metadata_container_type metadata = {
    std::make_tuple ("type",
                     "type",
                     "std::string",
                     obj.get_type ()
                    ),
    std::make_tuple ("offset",
                     "offset in bytes",
                     "std::uint64_t",
                     std::to_string (obj.get_offset ())
                    ),
    std::make_tuple ("size",
                     "size",
                     "size_type",
                     std::to_string (obj.get_size ()) + " bytes"
                    ),
    std::make_tuple ("name",
                     "name",
                     "std::string",
                     obj.get_name ()
                    ),
    std::make_tuple ("fs_type",
                     "filesystem type",
                     "std::string",
                     obj.get_fs_type ()
                    ),
    std::make_tuple ("volume_uuid",
                     "volume UUID",
                     "std::string",
                     obj.get_volume_uuid ()
                    ),
    std::make_tuple ("volume_name",
                     "volume name",
                     "std::string",
                     obj.get_volume_name ()
                    ),
    std::make_tuple ("signature",
                     "file system signature",
                     "std::string",
                     obj.get_signature ()
                    ),
    std::make_tuple ("version",
                     "file system version",
                     "std::uint16_t",
                     std::to_string (obj.get_version ())
                    ),
    std::make_tuple ("attributes",
                     "attributes",
                     "std::uint32_t",
                     std::to_string (obj.get_attributes ())
                    ),
    std::make_tuple ("creation_time",
                     "file system creation date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_creation_time ())
                    ),
    std::make_tuple ("last_modification_time",
                     "last modification date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_last_modification_time ())
                    ),
    std::make_tuple ("last_backup_time",
                     "last backup date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_last_backup_time ())
                    ),
    std::make_tuple ("last_check_time",
                     "last check date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_last_check_time ())
                    ),
    std::make_tuple ("file_count",
                     "total number of files",
                     "std::uint32_t",
                     std::to_string (obj.get_file_count ())
                    ),
    std::make_tuple ("folder_count",
                     "total number of folders",
                     "std::uint32_t",
                     std::to_string (obj.get_folder_count ())
                    ),
    std::make_tuple ("block_size",
                     "block size",
                     "std::uint32_t",
                     std::to_string (obj.get_block_size ()) + " bytes"
                    ),
    std::make_tuple ("total_blocks",
                     "number of allocation blocks",
                     "std::uint32_t",
                     std::to_string (obj.get_total_blocks ())
                    ),
    std::make_tuple ("free_blocks",
                     "number of unused allocation blocks",
                     "std::uint32_t",
                     std::to_string (obj.get_free_blocks ())
                    ),
    std::make_tuple ("mount_count",
                     "number of times the file system was mounted",
                     "std::uint32_t",
                     std::to_string (obj.get_mount_count ())
                    ),
  };

  return metadata;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get metadata from filesystem_iso
//! \param obj filesystem_iso object
//! \return list of attributes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static const metadata_container_type
get_metadata (const filesystem_iso& obj)
{
  metadata_container_type metadata = {
    std::make_tuple ("type",
                     "type",
                     "std::string",
                     obj.get_type ()
                    ),
    std::make_tuple ("offset",
                     "offset in bytes",
                     "std::uint64_t",
                     std::to_string (obj.get_offset ())
                    ),
    std::make_tuple ("size",
                     "size",
                     "size_type",
                     std::to_string (obj.get_size ()) + " bytes"
                    ),
    std::make_tuple ("name",
                     "name",
                     "std::string",
                     obj.get_name ()
                    ),
    std::make_tuple ("fs_type",
                     "filesystem type",
                     "std::string",
                     obj.get_fs_type ()
                    ),
    std::make_tuple ("volume_descr_version",
                     "volume description version",
                     "std::uint8_t",
                     std::to_string (obj.get_volume_descr_version ())
                    ),
    std::make_tuple ("system_id",
                     "system identifier",
                     "std::string",
                     obj.get_system_id ()
                    ),
    std::make_tuple ("volume_id",
                     "volume identifier",
                     "std::string",
                     obj.get_volume_id ()
                    ),
    std::make_tuple ("volume_space_size",
                     "volume space size",
                     "std::uint32_t",
                     std::to_string (obj.get_volume_space_size ())
                    ),
    std::make_tuple ("volume_set_size",
                     "volume set size",
                     "std::uint16_t",
                     std::to_string (obj.get_volume_set_size ())
                    ),
    std::make_tuple ("volume_sequence_number",
                     "volume sequence number",
                     "std::uint16_t",
                     std::to_string (obj.get_volume_sequence_number ())
                    ),
    std::make_tuple ("logical_block_size",
                     "logical block size",
                     "std::uint16_t",
                     std::to_string (obj.get_logical_block_size ()) + " bytes"
                    ),
    std::make_tuple ("path_table_size",
                     "path table size",
                     "std::uint32_t",
                     std::to_string (obj.get_path_table_size ()) + " bytes"
                    ),
    std::make_tuple ("type_l_path_table",
                     "location of type L path table",
                     "std::uint32_t",
                     std::to_string (obj.get_type_l_path_table ())
                    ),
    std::make_tuple ("type_l_optional_path_table",
                     "location of optional type L path table",
                     "std::uint32_t",
                     std::to_string (obj.get_type_l_optional_path_table ())
                    ),
    std::make_tuple ("type_m_path_table",
                     "location of type M path table",
                     "std::uint32_t",
                     std::to_string (obj.get_type_m_path_table ())
                    ),
    std::make_tuple ("type_m_optional_path_table",
                     "location of optional type M path table",
                     "std::uint32_t",
                     std::to_string (obj.get_type_m_optional_path_table ())
                    ),
    std::make_tuple ("volume_set_id",
                     "volume set identifier",
                     "std::string",
                     obj.get_volume_set_id ()
                    ),
    std::make_tuple ("publisher_id",
                     "publisher identifier",
                     "std::string",
                     obj.get_publisher_id ()
                    ),
    std::make_tuple ("data_preparer_id",
                     "data preparer identifier",
                     "std::string",
                     obj.get_data_preparer_id ()
                    ),
    std::make_tuple ("application_id",
                     "application identifier",
                     "std::string",
                     obj.get_application_id ()
                    ),
    std::make_tuple ("copyright_file_id",
                     "copyright file identifier",
                     "std::string",
                     obj.get_copyright_file_id ()
                    ),
    std::make_tuple ("abstract_file_id",
                     "abstract file identifier",
                     "std::string",
                     obj.get_abstract_file_id ()
                    ),
    std::make_tuple ("bibliographic_file_id",
                     "bibliographic file identifier",
                     "std::string",
                     obj.get_bibliographic_file_id ()
                    ),
    std::make_tuple ("creation_time",
                     "volume creation date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_creation_time ())
                    ),
    std::make_tuple ("last_modification_time",
                     "volume last modification date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_last_modification_time ())
                    ),
    std::make_tuple ("expiration_time",
                     "volume expiration date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_expiration_time ())
                    ),
    std::make_tuple ("effective_time",
                     "volume effective date/time",
                     "mobius::datetime::datetime",
                     to_string (obj.get_effective_time ())
                    ),
    std::make_tuple ("file_structure_version",
                     "file structure version",
                     "std::uint8_t",
                     std::to_string (obj.get_file_structure_version ())
                    ),
  };

  return metadata;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get metadata from filesystem_ntfs
//! \param obj filesystem_ntfs object
//! \return list of attributes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static const metadata_container_type
get_metadata (const filesystem_ntfs& obj)
{
  metadata_container_type metadata = {
    std::make_tuple ("type",
                     "type",
                     "std::string",
                     obj.get_type ()
                    ),
    std::make_tuple ("offset",
                     "offset in bytes",
                     "std::uint64_t",
                     std::to_string (obj.get_offset ())
                    ),
    std::make_tuple ("size",
                     "size",
                     "size_type",
                     std::to_string (obj.get_size ()) + " bytes"
                    ),
    std::make_tuple ("name",
                     "name",
                     "std::string",
                     obj.get_name ()
                    ),
    std::make_tuple ("fs_type",
                     "filesystem type",
                     "std::string",
                     obj.get_fs_type ()
                    ),
    std::make_tuple ("oem_name",
                     "OEM name",
                     "std::string",
                     obj.get_oem_name ()
                    ),
    std::make_tuple ("sectors",
                     "number of sectors",
                     "std::uint64_t",
                     std::to_string (obj.get_sectors ())
                    ),
    std::make_tuple ("sector_size",
                     "sector size",
                     "std::uint32_t",
                     std::to_string (obj.get_sector_size ()) + " bytes"
                    ),
    std::make_tuple ("sectors_per_cluster",
                     "sectors per cluster",
                     "std::uint32_t",
                     std::to_string (obj.get_sectors_per_cluster ())
                    ),
    std::make_tuple ("cluster_size",
                     "cluster size",
                     "std::uint32_t",
                     std::to_string (obj.get_cluster_size ()) + " bytes"
                    ),
    std::make_tuple ("reserved_sectors",
                     "number of reserved sectors",
                     "std::uint32_t",
                     std::to_string (obj.get_reserved_sectors ())
                    ),
    std::make_tuple ("media_descriptor",
                     "media descriptor",
                     "std::uint16_t",
                     std::to_string (obj.get_media_descriptor ())
                    ),
    std::make_tuple ("sectors_per_track",
                     "sectors per track",
                     "std::uint32_t",
                     std::to_string (obj.get_sectors_per_track ())
                    ),
    std::make_tuple ("heads",
                     "number of heads",
                     "std::uint32_t",
                     std::to_string (obj.get_heads ())
                    ),
    std::make_tuple ("hidden_sectors",
                     "number of hidden sectors",
                     "std::uint64_t",
                     std::to_string (obj.get_hidden_sectors ())
                    ),
    std::make_tuple ("mft_cluster",
                     "$MFT cluster",
                     "std::uint64_t",
                     std::to_string (obj.get_mft_cluster ())
                    ),
    std::make_tuple ("mft_mirror_cluster",
                     "$MFTMirr cluster",
                     "std::uint64_t",
                     std::to_string (obj.get_mft_mirror_cluster ())
                    ),
    std::make_tuple ("mft_offset",
                     "$MFT offset",
                     "std::uint64_t",
                     std::to_string (obj.get_mft_offset ())
                    ),
    std::make_tuple ("mft_mirror_offset",
                     "$MFTMirr offset",
                     "std::uint64_t",
                     std::to_string (obj.get_mft_mirror_offset ())
                    ),
    std::make_tuple ("clusters_per_frs",
                     "clusters per file record segment",
                     "std::uint32_t",
                     std::to_string (obj.get_clusters_per_frs ())
                    ),
    std::make_tuple ("frs_size",
                     "file record segment size",
                     "std::uint32_t",
                     std::to_string (obj.get_frs_size ()) + " bytes"
                    ),
    std::make_tuple ("clusters_per_index_block",
                     "clusters per index block",
                     "std::uint32_t",
                     std::to_string (obj.get_clusters_per_index_block ())
                    ),
    std::make_tuple ("volume_serial_number",
                     "volume serial number",
                     "std::uint64_t",
                     "0x" + mobius::string::to_hex (obj.get_volume_serial_number (), 16)
                    ),
    std::make_tuple ("checksum",
                     "checksum",
                     "std::uint32_t",
                     std::to_string (obj.get_checksum ())
                    ),
  };

  return metadata;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get metadata from filesystem_vfat
//! \param obj filesystem_vfat object
//! \return list of attributes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static const metadata_container_type
get_metadata (const filesystem_vfat& obj)
{
  metadata_container_type metadata = {
    std::make_tuple ("type",
                     "type",
                     "std::string",
                     obj.get_type ()
                    ),
    std::make_tuple ("offset",
                     "offset in bytes",
                     "std::uint64_t",
                     std::to_string (obj.get_offset ())
                    ),
    std::make_tuple ("size",
                     "size",
                     "size_type",
                     std::to_string (obj.get_size ()) + " bytes"
                    ),
    std::make_tuple ("name",
                     "name",
                     "std::string",
                     obj.get_name ()
                    ),
    std::make_tuple ("fs_type",
                     "filesystem type",
                     "std::string",
                     obj.get_fs_type ()
                    ),
    std::make_tuple ("oem_name",
                     "OEM name",
                     "std::string",
                     obj.get_oem_name ()
                    ),
    std::make_tuple ("sector_size",
                     "sector size",
                     "std::uint32_t",
                     std::to_string (obj.get_sector_size ()) + " bytes"
                    ),
    std::make_tuple ("sectors_per_cluster",
                     "sectors per cluster",
                     "std::uint32_t",
                     std::to_string (obj.get_sectors_per_cluster ())
                    ),
    std::make_tuple ("reserved_sectors",
                     "number of reserved sectors",
                     "std::uint32_t",
                     std::to_string (obj.get_reserved_sectors ())
                    ),
    std::make_tuple ("fats",
                     "number of FATs",
                     "std::uint32_t",
                     std::to_string (obj.get_fats ())
                    ),
    std::make_tuple ("root_dir_entries",
                     "number of entries in root directory",
                     "std::uint32_t",
                     std::to_string (obj.get_root_dir_entries ())
                    ),
    std::make_tuple ("sectors",
                     "number of sectors",
                     "std::uint64_t",
                     std::to_string (obj.get_sectors ())
                    ),
    std::make_tuple ("media_descriptor",
                     "media descriptor type",
                     "std::uint16_t",
                     std::to_string (obj.get_media_descriptor ())
                    ),
    std::make_tuple ("sectors_per_fat",
                     "sectors per FAT",
                     "std::uint64_t",
                     std::to_string (obj.get_sectors_per_fat ())
                    ),
    std::make_tuple ("sectors_per_track",
                     "sectors per track",
                     "std::uint32_t",
                     std::to_string (obj.get_sectors_per_track ())
                    ),
    std::make_tuple ("heads",
                     "number of heads",
                     "std::uint32_t",
                     std::to_string (obj.get_heads ())
                    ),
    std::make_tuple ("hidden_sectors",
                     "number of hidden sectors",
                     "std::uint64_t",
                     std::to_string (obj.get_hidden_sectors ())
                    ),
    std::make_tuple ("logical_drive_number",
                     "logical drive number",
                     "std::uint16_t",
                     std::to_string (obj.get_logical_drive_number ())
                    ),
    std::make_tuple ("volume_id",
                     "volume ID",
                     "std::uint32_t",
                     "0x" + mobius::string::to_hex (obj.get_volume_id (), 8)
                    ),
    std::make_tuple ("volume_label",
                     "volume label",
                     "std::string",
                     obj.get_volume_label ()
                    ),
    std::make_tuple ("flags",
                     "flags",
                     "std::uint32_t",
                     std::to_string (obj.get_flags ())
                    ),
    std::make_tuple ("version",
                     "filesystem version",
                     "std::uint32_t",
                     std::to_string (obj.get_version ())
                    ),
    std::make_tuple ("root_dir_cluster",
                     "root directory cluster",
                     "std::uint32_t",
                     std::to_string (obj.get_root_dir_cluster ())
                    ),
    std::make_tuple ("fs_info_sector",
                     "filesystem information sector",
                     "std::uint64_t",
                     std::to_string (obj.get_fs_info_sector ())
                    ),
    std::make_tuple ("backup_boot_sector",
                     "backup boot sector",
                     "std::uint64_t",
                     std::to_string (obj.get_backup_boot_sector ())
                    ),
    std::make_tuple ("fat_entry_size",
                     "FAT entry size",
                     "std::uint32_t",
                     std::to_string (obj.get_fat_entry_size ()) + " bits"
                    ),
  };

  return metadata;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get metadata from filesystem
//! \param obj filesystem object
//! \return list of attributes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const metadata_container_type
get_metadata (const filesystem& obj)
{
  metadata_container_type metadata;

  if (obj.get_type () == "ext2")
    metadata = get_metadata (filesystem_ext2 (obj));

  else if (obj.get_type () == "hfs")
    metadata = get_metadata (filesystem_hfs (obj));

  else if (obj.get_type () == "iso")
    metadata = get_metadata (filesystem_iso (obj));

  else if (obj.get_type () == "ntfs")
    metadata = get_metadata (filesystem_ntfs (obj));

  else if (obj.get_type () == "vfat")
    metadata = get_metadata (filesystem_vfat (obj));

  return metadata;
}

} // namespace filesystem
} // namespace mobius
