// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "filesystem_impl_hfs.h"
#include <mobius/decoder/data_decoder.h>

namespace mobius
{
namespace filesystem
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if stream contains an instance of vfat filesystem
//! \param reader stream reader
//! \param offset offset from the beginning of the stream
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
filesystem_impl_hfs::is_instance (mobius::io::reader reader, std::uint64_t offset)
{
  reader.seek (offset + 1024);
  auto data = reader.read (2);
  
  return data == "HX" || data == "H+";
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param reader stream reader
//! \param offset offset from the beginning of the stream
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
filesystem_impl_hfs::filesystem_impl_hfs (mobius::io::reader reader, std::uint64_t offset)
 : filesystem_impl_base (reader, offset)
{
  // create decoder
  reader.seek (offset + 1024);
  mobius::decoder::data_decoder decoder (reader);
  
  // decode data
  signature_ = decoder.get_string_by_size (2);
  version_ = decoder.get_uint16_be ();
  attributes_ = decoder.get_uint32_be ();
  decoder.skip (8);             // lastMountedVersion, journalInfoBlock
  creation_time_ = decoder.get_hfs_datetime ();
  last_modification_time_ = decoder.get_hfs_datetime ();
  last_backup_time_ = decoder.get_hfs_datetime ();;
  last_check_time_ = decoder.get_hfs_datetime ();
  file_count_ = decoder.get_uint32_be ();
  folder_count_ = decoder.get_uint32_be ();
  block_size_ = decoder.get_uint32_be ();
  total_blocks_ = decoder.get_uint32_be ();
  free_blocks_ = decoder.get_uint32_be ();
  decoder.skip (16);            // nextAllocation...nextCatalogID
  mount_count_ = decoder.get_uint32_be ();
  decoder.skip (32);            // encodingsBitmap..finderInfo[5]
  volume_uuid_ = decoder.get_bytearray_by_size (8).to_hexstring ();
  
  // derived information
  size_ = total_blocks_ * block_size_;
  fs_type_ = (version_ == 4) ? "HFS+" : "HFSX";
  name_ = fs_type_ + " (uuid: " + volume_uuid_ + ')';
}

} // namespace filesystem
} // namespace mobius
