// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include "api_datetime.h"
#include "api_folder.h"
#include "api_file.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void api_folder_tp_dealloc (api_folder_o *);
static PyObject *api_folder_tp_new (PyTypeObject *, PyObject *, PyObject *);
static PyObject *api_folder_get_is_valid (api_folder_o *);
static PyObject *api_folder_get_url (api_folder_o *);
static PyObject *api_folder_get_is_file (api_folder_o *);
static PyObject *api_folder_get_is_folder (api_folder_o *);
static PyObject *api_folder_get_exists (api_folder_o *);
static PyObject *api_folder_get_size (api_folder_o *);
static PyObject *api_folder_get_user_id (api_folder_o *);
static PyObject *api_folder_get_user_name (api_folder_o *);
static PyObject *api_folder_get_group_id (api_folder_o *);
static PyObject *api_folder_get_group_name (api_folder_o *);
static PyObject *api_folder_get_permissions (api_folder_o *);
static PyObject *api_folder_get_last_access_time (api_folder_o *);
static PyObject *api_folder_get_last_modification_time (api_folder_o *);
static PyObject *api_folder_get_last_metadata_time (api_folder_o *);
static PyObject *api_folder_get_children (api_folder_o *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef api_folder_getsetters[] =
{
  {
    (char *) "is_valid",
    (getter) api_folder_get_is_valid,
    (setter) 0,
    (char *) "check if folder is valid", NULL
  },
  {
    (char *) "url",
    (getter) api_folder_get_url,
    (setter) 0,
    (char *) "URL as string", NULL
  },
  {
    (char *) "is_file",
    (getter) api_folder_get_is_file,
    (setter) 0,
    (char *) "check if entry is a file", NULL
  },
  {
    (char *) "is_folder",
    (getter) api_folder_get_is_folder,
    (setter) 0,
    (char *) "check if entry is a folder", NULL
  },
  {
    (char *) "exists",
    (getter) api_folder_get_exists,
    (setter) 0,
    (char *) "check if folder exists", NULL
  },
  {
    (char *) "size",
    (getter) api_folder_get_size,
    (setter) 0,
    (char *) "folder size in bytes", NULL
  },
  {
    (char *) "user_id",
    (getter) api_folder_get_user_id,
    (setter) 0,
    (char *) "owner's user ID", NULL
  },
  {
    (char *) "user_name",
    (getter) api_folder_get_user_name,
    (setter) 0,
    (char *) "owner's user name", NULL
  },
  {
    (char *) "group_id",
    (getter) api_folder_get_group_id,
    (setter) 0,
    (char *) "group ID", NULL
  },
  {
    (char *) "group_name",
    (getter) api_folder_get_group_name,
    (setter) 0,
    (char *) "group name", NULL
  },
  {
    (char *) "permissions",
    (getter) api_folder_get_permissions,
    (setter) 0,
    (char *) "access permission mask", NULL
  },
  {
    (char *) "last_access_time",
    (getter) api_folder_get_last_access_time,
    (setter) 0,
    (char *) "last access time", NULL
  },
  {
    (char *) "last_modification_time",
    (getter) api_folder_get_last_modification_time,
    (setter) 0,
    (char *) "last data modification time", NULL
  },
  {
    (char *) "last_metadata_time",
    (getter) api_folder_get_last_metadata_time,
    (setter) 0,
    (char *) "last metadata modification time", NULL
  },
  {
    (char *) "children",
    (getter) api_folder_get_children,
    (setter) 0,
    (char *) "folder entries", NULL
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject api_folder_t =
{
  PyObject_HEAD_INIT (0)
  0,                                       		// ob_size
  "api.folder",                             		// tp_name
  sizeof (api_folder_o),                    		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) api_folder_tp_dealloc,       		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "folder class with support for URL's",   		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  0,                                       		// tp_methods
  0,                                       		// tp_members
  api_folder_getsetters,                    		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  api_folder_tp_new                         		// tp_new
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: tp_new
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_tp_new (PyTypeObject *type, PyObject *args, PyObject *kwds)
{
  const char *arg_uri = nullptr;

  if (!PyArg_ParseTuple (args, "|s", &arg_uri))
    return NULL;

  api_folder_o *self = (api_folder_o *) type->tp_alloc (type, 0);
  if (self != NULL)
    {
      try
        {
          if (arg_uri)
            self->obj = new mobius::io::folder (arg_uri);
          else
            self->obj = new mobius::io::folder ();
        }
      catch (const std::runtime_error& e)
        {
          Py_DECREF (self);
          PyErr_SetString (PyExc_IOError, e.what ());
          self = NULL;
        }
    }

  return (PyObject *) self;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
api_folder_tp_dealloc (api_folder_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: is_valid getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_is_valid (api_folder_o *self)
{
  return PyBool_FromLong (self->obj->operator bool ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: url getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_url (api_folder_o *self)
{
  return PyString_FromString (self->obj->get_url ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: is_file getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_is_file (api_folder_o *self)
{
  return PyBool_FromLong (self->obj->is_file ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: is_folder getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_is_folder (api_folder_o *self)
{
  return PyBool_FromLong (self->obj->is_folder ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: exists getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_exists (api_folder_o *self)
{
  return PyBool_FromLong (self->obj->exists ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_size (api_folder_o *self)
{
  return PyInt_FromLong (self->obj->get_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: user_id getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_user_id (api_folder_o *self)
{
  return PyInt_FromLong (self->obj->get_user_id ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: user_name getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_user_name (api_folder_o *self)
{
  return PyString_FromString (self->obj->get_user_name ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: group_id getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_group_id (api_folder_o *self)
{
  return PyInt_FromLong (self->obj->get_group_id ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: group_name getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_group_name (api_folder_o *self)
{
  return PyString_FromString (self->obj->get_group_name ().c_str ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: permissions getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_permissions (api_folder_o *self)
{
  return PyInt_FromLong (self->obj->get_permissions ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: last_access_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_last_access_time (api_folder_o *self)
{
  return PyDateTime_from_datetime (self->obj->get_last_access_time ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: last_modification_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_last_modification_time (api_folder_o *self)
{
 return PyDateTime_from_datetime (self->obj->get_last_modification_time ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: last_metadata_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_last_metadata_time (api_folder_o *self)
{
 return PyDateTime_from_datetime (self->obj->get_last_metadata_time ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.folder: children getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_folder_get_children (api_folder_o *self)
{
  PyObject *ret = PyList_New (0);
  if (ret == NULL)
    return NULL;

  try
    {
      for (const auto& child : *(self->obj))
        {
          if (child.is_file ())
            {
              api_file_o *item = (api_file_o *) _PyObject_New (&api_file_t);

              if (item == NULL)
                {
                  Py_CLEAR (ret);
                  return NULL;
                }

              item->obj = new mobius::io::file (child);
              PyList_Append (ret, (PyObject *) item);
            }
          else
            {
              api_folder_o *item = (api_folder_o *) api_folder_t.tp_alloc (&api_folder_t, 0);

              if (item == NULL)
                {
                  Py_CLEAR (ret);
                  return NULL;
                }

              item->obj = new mobius::io::folder (child);
              PyList_Append (ret, (PyObject *) item);
            }
        }
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
      Py_CLEAR (ret);
      return NULL;
    }

  return ret;
}
