# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import mobius.api
import mobius.deprecated
import mobius.xml
import os.path
import hashlib
import datetime
import shutil
import mobius.pyant.vfs

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief registry constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
RC_HIVE_ADDED, RC_HIVE_INVALID, RC_HIVE_TYPE_UNKNOWN, RC_HIVE_EXISTS = range (4)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief registry model
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Registry (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.__updated = False
    self.__registry = mobius.ant.registry.registry ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief iterate through root keys
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __iter__ (self):
    self.__update_root_keys ()

    for key in self.__registry.keys:
      yield RegistryKey (key)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief return number of files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __len__ (self):
    return len (self.__registry.files)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add file to hive
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_file_by_path (self, role, path, localpath):
    self.__registry.add_file_by_path (role, path, localpath)
    self.__updated = False
    return RC_HIVE_ADDED

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Remove file from hive
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def remove_file (self, uid):
    self.__registry.remove_file (uid)
    self.__updated = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get files
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_files (self):
    return iter (self.__registry.files)

  files = property (__get_files, None)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get key by path
  # @param path key path
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_key_by_path (self, path):
    self.__update_root_keys ()

    key = self.__registry.get_key_by_path (path)

    if key:
      key = RegistryKey (key)

    return key

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get value by path
  # @param path value path
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_value_by_path (self, path):
    self.__update_root_keys ()
    return self.__registry.get_value_by_path (path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get value data by path
  # @param path value path
  # @return data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_data_by_path (self, path):
    self.__update_root_keys ()
    return self.__registry.get_data_by_path (path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Update root keys
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __update_root_keys (self):
    if not self.__updated:
 
      # set updated
      self.__updated = True

      # call handlers for post update
      for hid, handler in (mobius.mediator.call ('app.list-resources', 'registry.handler') or []):
        handler (self)
    
      # emit registry.updated event
      mobius.mediator.emit ('registry.updated', self)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief logical registry key
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class RegistryKey (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @ brief initialize key
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, key):
    self.__key = key
    self.name = key.name
    self.last_modification_time = key.last_modification_time
    self.classname = key.classname

    self.__subkeys_loaded = False
    self.__subkeys = []

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @ brief iterate through RegistryKey children and key children
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __iter__ (self):
    self.__load_subkeys_on_demand ()

    for c in self.__subkeys:
      yield c

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @ brief add a link to a hivefile key
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_link (self, key, name):
    child = RegistryKey (key)
    child.name = name
    self.__subkeys.append (child)

    return child

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get subkey by name
  # @param name subkey name
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_key_by_name (self, name):
    self.__load_subkeys_on_demand ()

    for k in self.__subkeys:
      if k.name and k.name.upper () == name.upper ():
        return k

    return None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get key by path
  # @param path key path
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_key_by_path (self, path):

    # supress leading '\'
    path = path.lstrip ('\\')

    # if '\' in path, find remaining path
    if '\\' in path:
      key_name, remaining_path = path.split ('\\', 1)
      subkey = self.get_key_by_name (key_name)
    
      # if key is found, recurse
      if subkey:
        key = subkey.get_key_by_path (remaining_path)

      # otherwise, no key found
      else:
        key = None
  
    # otherwise, find key by name
    else:
      key = self.get_key_by_name (path)

    # return key
    return key

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get value by name
  # @param name value name
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_value_by_name (self, name):
    return self.__key.get_value_by_name (name)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get value by path
  # @param path value path
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_value_by_path (self, path):
    return self.__key.get_value_by_path (path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get value data by name
  # @param name value name
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_data_by_name (self, name):
    return self.__key.get_data_by_name (name)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get value data by path
  # @param name value name
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_data_by_path (self, path):
    return self.__key.get_data_by_path (path)
   
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief check if key has subkeys
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def has_subkeys (self):
    self.__load_subkeys_on_demand ()
    return bool (self.__subkeys)
      
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief return subkeys
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_subkeys (self):
    self.__load_subkeys_on_demand ()
    return self.__subkeys

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief return values
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_values (self):
    return self.__key.values

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @ brief load subkeys from hivefile key
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __load_subkeys_on_demand (self):
    if self.__subkeys_loaded:
      return

    for k in self.__key.subkeys:
      self.__subkeys.append (RegistryKey (k))

    self.__subkeys_loaded = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief properties
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  subkeys = property (__get_subkeys, None)
  values = property (__get_values, None)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief generic dataholder
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class dataholder (object):
  pass

import sys
import datetime

STARTTIME = None
DEBUG = False

def LOG (*args):
  if not DEBUG:
    return

  now = datetime.datetime.utcnow ()

  global STARTTIME
  if not STARTTIME:
    STARTTIME = now

  d = now - STARTTIME

  print >> sys.stderr, now, '%d.%06d' % (d.seconds, d.microseconds), ' '.join (str (a) for a in args)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief ant: win registry
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Ant (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, item):
    self.__item = item
    self.__data = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief check if data is cached
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def is_data_cached (self):
    is_cached = False

    if self.__item.datasource:
      data = self.__data

      # get cached data, if any            
      if not data:
        path = mobius.deprecated.get_item_data_path (self.__item, 'registry', 'data.xml')

        if os.path.exists (path):
          pickle = mobius.xml.Pickle ()
          data = pickle.load (path)

      # check if data is up to date
      if data and data.mtime >= self.__item.datasource.mtime:
        is_cached = True

    return is_cached

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get registry data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_data (self):

    # if item has datasource, retrieves data
    if self.__item.datasource:

      # try to retrieve cached data
      if not self.__data:
        path = mobius.deprecated.get_item_data_path (self.__item, 'registry', 'data.xml')

        if os.path.exists (path):
          pickle = mobius.xml.Pickle ()
          self.__data = pickle.load (path)

      # no data yet, retrieves data from disk
      if not self.__data or self.__item.datasource.mtime > self.__data.mtime:
        self.retrieve_data ()

    # no datasource, removes old cached data
    else:
      self.__remove_data ()
      self.__data = None
      return []

    # builds return list
    ret = []

    for data in self.__data.reg_list:
      registry = Registry ()

      for hivefile in data.hivefiles:
        localpath = mobius.deprecated.get_item_data_path (self.__item, 'registry', '%04d' %   data.idx, hivefile.filename)
        registry.add_file_by_path (hivefile.role, hivefile.path, localpath)

      ret.append (registry)
            
    return ret

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief retrieve artifacts
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def retrieve_data (self):
    self.__data = None

    # get item VFS
    vfs = mobius.pyant.vfs.get_item_vfs (self.__item)
    if not vfs:
      raise Exception, "Datasource is not available"

    # retrieve artifacts
    next_idx = 1
    registry = None
    hashes = set ()
    reg_list = []
    user_hivefiles = []

    # scan VFS root entries
    for root in vfs:
            
      # system entries
      system_hivefiles = self.__get_system_hivefiles (root)
      
      if system_hivefiles:
        registry = dataholder ()
        registry.idx = next_idx
        registry.hivefiles = system_hivefiles
        reg_list.append (registry)
        next_idx += 1

        if user_hivefiles:      # previously found NTUSER hivefiles
          registry.hivefiles.extend (user_hivefiles)
          user_hivefiles = []

      # user hivefiles
      user_hivefiles = self.__get_user_hivefiles (root)

      if user_hivefiles and registry:
        registry.hivefiles.extend (user_hivefiles)
        user_hivefiles = []

    # retrieve hivefiles
    for registry in reg_list:
      for hivefile in registry.hivefiles:
        hash_md5 = self.__retrieve_file (hivefile.entry, registry.idx, hivefile.filename)
        hashes.add (hash_md5)
        del hivefile.entry

    # build data model
    self.__data = dataholder ()
    self.__data.mtime = datetime.datetime.utcnow ()
    self.__data.reg_list = reg_list

    # save data model
    path = mobius.deprecated.create_item_data_path (self.__item, 'registry', 'data.xml')
    pickle = mobius.xml.Pickle ()
    pickle.save (path, self.__data)

    # save KFF
    if hashes:
      path = mobius.deprecated.create_case_path (self.__item.case, 'hashset', '%04d-registry.ignore' % self.__item.uid)

      fp = open (path, 'w')
      for h in hashes:
        fp.write ('%s\n' % h)
      fp.close ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief remove registry data for an item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __remove_data (self):

    # data/registry
    path = mobius.deprecated.get_item_data_path (self.__item, 'registry')
    if os.path.exists (path):
      shutil.rmtree (path)

    # .ignore
    path = mobius.deprecated.get_case_path (self.__item.case, 'hashset', '%04d-registry.ignore' % self.__item.uid)
    if os.path.exists (path):
      os.remove (path)
    
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get system hivefiles
  # @param filesystem's root entry
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_system_hivefiles (self, root):
    LOG ('__get_system_hivefiles (1)', root.path)
    entries = []

    # %windir%/system32/config    
    config_dir = root.get_child_by_path ('windows/system32/config', False)
    LOG ('__get_system_hivefiles (2)', config_dir)

    if config_dir:
      components = config_dir.get_child_by_name ('components', False)
      LOG ('__get_system_hivefiles (3)', components)
      if components:
        entries.append (components)

      sam = config_dir.get_child_by_name ('sam', False)
      LOG ('__get_system_hivefiles (4)', sam)
      if sam:
        entries.append (sam)

      security = config_dir.get_child_by_name ('security', False)
      LOG ('__get_system_hivefiles (5)', security)
      if security:
        entries.append (security)

      software = config_dir.get_child_by_name ('software', False)
      LOG ('__get_system_hivefiles (6)', software)
      if software:
        entries.append (software)

      system = config_dir.get_child_by_name ('system', False)
      LOG ('__get_system_hivefiles (7)', system)
      if system:
        entries.append (system)

      default = config_dir.get_child_by_name ('default', False)
      LOG ('__get_system_hivefiles (8)', default)
      if default:
        entries.append (default)

      LOG ('__get_system_hivefiles (9)')

    # create hivefiles
    hivefiles = []

    for entry in entries:
      parent = entry.get_parent ()

      hivefile = dataholder ()
      hivefile.path = entry.path.replace ('/', '\\')
      hivefile.role = entry.name.upper ()
      hivefile.filename = entry.name.upper ()
      hivefile.entry = entry
      hivefiles.append (hivefile)

    return hivefiles

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get user hivefiles
  # @param filesystem's root entry
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_user_hivefiles (self, root):
    LOG ('__get_user_hivefiles (1)', root.path)
    entries = []
    
    # %profilesdir%/*/ntuser.dat
    users_dir = root.get_child_by_name ('users', False)

    if not users_dir:
      users_dir = root.get_child_by_name ('documents and settings', False)

    LOG ('__get_user_hivefiles (2)', users_dir)
    if users_dir:
      for child in users_dir.get_children ():
        if child.is_folder ():
          ntuser = child.get_child_by_name ('ntuser.dat', False)
          if ntuser:
            entries.append (ntuser)

    # %windir%/serviceprofiles/{localservice,networkservice}/ntuser.dat
    config_dir = root.get_child_by_path ('windows/serviceprofiles', False)
    LOG ('__get_user_hivefiles (3)', config_dir)

    if config_dir:
      localservice = config_dir.get_child_by_path ('localservice/ntuser.dat', False)
      LOG ('__get_user_hivefiles (4)', localservice)
      if localservice:
        entries.append (localservice)
                  
      networkservice = config_dir.get_child_by_path ('networkservice/ntuser.dat', False)
      LOG ('__get_user_hivefiles (5)', networkservice)
      if networkservice:
        entries.append (networkservice)

    # system profile
    config_dir = root.get_child_by_path ('windows/system32/config/systemprofile', False)
    LOG ('__get_user_hivefiles (6)', config_dir)
    if config_dir:
      f = config_dir.get_child_by_name ("ntuser.dat", False)
      if f:
        entries.append (f)

    # create hivefiles
    hivefiles = []

    for entry in entries:
      parent = entry.get_parent ()

      hivefile = dataholder ()
      hivefile.path = entry.path.replace ('/', '\\')
      hivefile.role = 'NTUSER'
      hivefile.filename = 'NTUSER-%s.dat' % parent.name.upper ()
      hivefile.entry = entry
      hivefiles.append (hivefile)

    return hivefiles

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief retrieve file
  # @param entry entry
  # @param path fullpath
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_file (self, entry, idx, filename):
    path = mobius.deprecated.create_item_data_path (self.__item, 'registry', '%04d' % idx, filename)
    reader = entry.new_reader ()
    fp = open (path, 'w')
    h = hashlib.md5 ()
    BLOCK_SIZE = 65536

    data = reader.read (BLOCK_SIZE)
    while data:
      fp.write (data)
      h.update (data)
      data = reader.read (BLOCK_SIZE)

    fp.close ()
    
    return h.hexdigest ()
