// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/crypt/cipher_rc4.h>
#include <mobius/exception.inc>
#include <stdexcept>

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief constructor
//!\param passwd encryption/decryption password
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
cipher_rc4::cipher_rc4 (const bytearray& passwd)
  : passwd_ (passwd)
{
  if (passwd.empty () || passwd.size () > 255)
    throw std::out_of_range (MOBIUS_EXCEPTION_MSG ("invalid password size"));

  reset ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief reset state
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
cipher_rc4::reset ()
{
  for (std::size_t i = 0; i < 256; i++)
    S_[i] = i;

  std::uint8_t j = 0;
  std::uint8_t tmp;

  for (std::size_t i = 0; i < 256; i++)
    {
      j += S_[i] + passwd_[i % passwd_.size ()];
      tmp = S_[i];
      S_[i] = S_[j];
      S_[j] = tmp;
    }

  i_ = j_ = 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief encrypt/decrypt one byte
//!\param c byte
//!\return a new byte
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint8_t
cipher_rc4::_crypt_byte (std::uint8_t c) noexcept
{
  i_++;
  j_ += S_[i_];

  std::uint8_t tmp = S_[i_];
  S_[i_] = S_[j_];
  S_[j_] = tmp;

  std::uint8_t k = S_[(S_[i_] + S_[j_]) & 0xff];

  return c ^ k;
}

} // namespace crypt
} // namespace mobius
