// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "case.h"
#include "item.h"
#include <mobius/exception.inc>
#include <mobius/exception_posix.inc>
#include <mobius/database/database.h>
#include <mobius/io/path.h>
#include <stdexcept>
#include <sys/stat.h>
#include <sys/types.h>

namespace mobius
{
namespace model
{
void case_schema (mobius::database::database);

namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Constants
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr char DIR_SEPARATOR = '/';

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create directory recursively
//! \param path directory path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
make_directory (const std::string& path)
{
  if (mkdir (path.c_str (), 0700) == -1)
    {
      if (errno == ENOENT)
        {
          auto idx = path.rfind (DIR_SEPARATOR);

          if (idx == std::string::npos)
            throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("Cannot create directory"));

          std::string head = path.substr (0, idx);
          make_directory (head);
          make_directory (path);
        }

      else if (errno != EEXIST)
        throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Join paths
//! \param abs_path Absolute path
//! \param rel_path Relative path
//! \return Full path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static std::string
join_path (const std::string& abs_path, const std::string& rel_path)
{
  auto path = mobius::io::path (abs_path);
  return to_string (path.get_child_by_path (rel_path));
}

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Case implementation class
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Case::impl
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // constructors
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  impl (const impl&) = delete;
  impl (impl&&) = delete;
  impl (const std::string&, std::uint32_t);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // function prototypes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::string get_path (const std::string&) const;
  std::string create_path (const std::string&) const;
  mobius::database::connection new_connection ();
  mobius::database::transaction new_transaction ();
  mobius::database::database get_database () const;
  bool has_item_by_uid (std::int64_t) const;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get root item
  //! \return root item
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::int64_t
  get_root_item_uid () const
  {
    return root_item_uid_;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief get uid
  //! \return uid
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::uint32_t
  get_uid () const
  {
    return uid_;
  }

private:
  //! \brief case UID
  std::uint32_t uid_;

  //! \brief root item UID
  std::int64_t root_item_uid_;

  //! \brief base directory
  std::string base_dir_;

  //! \brief database connection pool
  mobius::database::connection_pool pool_;
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param path case folder path
//! \param uid case UID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
Case::impl::impl (const std::string& path, std::uint32_t uid)
  : uid_ (uid),
    base_dir_ (path),
    pool_ (join_path (path, "case.sqlite"))

{
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // create tables
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto db = pool_.get_database ();
  db.execute ("pragma foreign_keys=ON");

  auto transaction = db.new_transaction ();
  case_schema (db);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // get root item UID, if any
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  auto stmt = db.new_statement (
           "SELECT uid "
             "FROM item "
            "WHERE parent_uid IS NULL");

  if (stmt.fetch_row ())
    root_item_uid_ = stmt.get_column_int64 (0);

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // otherwise, populate case and root item
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  else
    {
      // create case row
      stmt = db.new_statement (
               "INSERT INTO 'case'"
                    "VALUES (1, DATETIME ('NOW'))");

      stmt.execute ();


      // create root item
      stmt = db.new_statement (
               "INSERT INTO item "
                    "VALUES (NULL, NULL, 1, 'case', DATETIME ('NOW'))");

      stmt.execute ();

      root_item_uid_ = db.get_last_insert_row_id ();
    }

  transaction.commit ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get path inside case folder
//! \param rpath Relative path
//! \return Full path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
Case::impl::get_path (const std::string& rpath) const
{
  return join_path (base_dir_, rpath);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create path inside case folder
//! \param rpath relative path
//! \return full path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
Case::impl::create_path (const std::string& rpath) const
{
  const std::string fullpath = join_path (base_dir_, rpath);

  mobius::io::path path (fullpath);
  make_directory (path.get_dirname ());

  return fullpath;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new connection for case database
//! \return new connection object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::database::connection
Case::impl::new_connection ()
{
  return pool_.acquire ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new transaction
//! \return new transaction object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::database::transaction
Case::impl::new_transaction ()
{
  auto db = pool_.get_database ();
  return db.new_transaction ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get database object
//! \return database object for current thread
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::database::database
Case::impl::get_database () const
{
  return pool_.get_database ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get item by UID
//! \param uid item UID
//! \return item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
Case::impl::has_item_by_uid (std::int64_t uid) const
{
  auto db = pool_.get_database ();

  auto stmt = db.new_statement (
                "SELECT 1 "
                  "FROM item "
                 "WHERE uid = ?");

  stmt.bind (1, uid);

  return stmt.fetch_row ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param path case folder path
//! \param uid case UID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
Case::Case (const std::string& path, std::uint32_t uid)
  : impl_ (std::make_shared <impl> (path, uid))
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get path inside case folder
//! \param rpath relative path
//! \return full path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
Case::get_path (const std::string& rpath) const
{
  return impl_->get_path (rpath);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create path inside case folder
//! \param rpath relative path
//! \return full path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
Case::create_path (const std::string& rpath) const
{
  return impl_->create_path (rpath);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new connection for case database
//! \return new connection object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::database::connection
Case::new_connection ()
{
  return impl_->new_connection ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new transaction
//! \return new transaction object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::database::transaction
Case::new_transaction ()
{
  return impl_->new_transaction ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get database object
//! \return database object for current thread
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::database::database
Case::get_database () const
{
  return impl_->get_database ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get item by UID
//! \param uid item UID
//! \return item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item
Case::get_item_by_uid (std::int64_t uid) const
{
  item it;

  if (impl_->has_item_by_uid (uid))
    it = item (*this, uid);

  return it;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get root item
//! \return root item
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
item
Case::get_root_item () const
{
  return item (*this, impl_->get_root_item_uid ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get uid
//! \return uid
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t
Case::get_uid () const
{
  return impl_->get_uid ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new application object
//! \param id ID
//! \param name Name
//! \return Application object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
application
Case::new_application (const std::string& id, const std::string& name)
{
  auto db = impl_->get_database ();
  mobius::database::statement stmt;
  std::int64_t uid = 0;

  // check if application already exists
  stmt = db.new_statement (
                "SELECT uid "
                  "FROM application "
                 "WHERE id = ?");

  stmt.bind (1, id);

  if (stmt.fetch_row ())
    uid = stmt.get_column_int64 (0);

  // create new application, if not  
  else
    {
      stmt = db.new_statement (
              "INSERT INTO application "
                   "VALUES (NULL, ?, ?)");

      stmt.bind (1, id);
      stmt.bind (2, name);
      stmt.execute ();
  
      uid = db.get_last_insert_row_id ();
    }

  // return object
  return application (*this, uid);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get passwords
//! \return Passwords
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <password>
Case::get_passwords () const
{
  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                  "FROM password "
              "ORDER BY item_uid, type");

  std::vector <password> passwords;

  while (stmt.fetch_row ())
    {
      auto uid = stmt.get_column_int64 (0);
      passwords.emplace_back (*this, uid);
    }

  return passwords;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get password hashes
//! \return Password hashes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <password_hash>
Case::get_password_hashes () const
{
  auto db = impl_->get_database ();

  auto stmt = db.new_statement (
                "SELECT uid "
                  "FROM password_hash "
              "ORDER BY item_uid, type");

  std::vector <password_hash> password_hashes;

  while (stmt.fetch_row ())
    {
      auto uid = stmt.get_column_int64 (0);
      password_hashes.emplace_back (*this, uid);
    }

  return password_hashes;
}

} // namespace model
} // namespace mobius
