#ifndef MOBIUS_PYTHON_PYMOBIUS_H
#define MOBIUS_PYTHON_PYMOBIUS_H

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#define PY_SSIZE_T_CLEAN        // PEP 353

#include <Python.h>
#include <mobius/bytearray.h>
#include <mobius/datetime/datetime.h>
#include <cstdint>
#include <string>

namespace mobius
{
namespace py
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief GIL auto release/acquire class
//! \author Eduardo Aguiar
//
// Use: auto v = mobius::py::GIL () (expression)
// Example: auto data = mobius::py::GIL () (self->obj->get_bytearray ())
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class GIL
{
public:
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Constructor
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  GIL () noexcept
    : state_ (PyEval_SaveThread ())
  {
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Destructor
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  ~GIL ()
  {
    PyEval_RestoreThread (state_);
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Return value passed
  //! \param value Any value
  //! \return value
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  template <typename T>
  T operator () (T&& value)
  {
    return std::move (value);
  }
  
private:
  PyThreadState *state_;
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Parse function arguments
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::uint32_t get_arg_size (PyObject *) noexcept;
PyObject *get_arg (PyObject *, std::uint32_t);
std::string get_arg_as_std_string (PyObject *, std::uint32_t);
mobius::bytearray get_arg_as_bytearray (PyObject *, std::uint32_t);
mobius::datetime::datetime get_arg_as_datetime (PyObject *, std::uint32_t);
bool get_arg_as_bool (PyObject *, std::uint32_t);
std::uint32_t get_arg_as_uint32_t (PyObject *, std::uint32_t);
std::uint64_t get_arg_as_uint64_t (PyObject *, std::uint32_t);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Parse function arguments (with default value)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string get_arg_as_std_string (PyObject *, std::uint32_t, const std::string&);
mobius::bytearray get_arg_as_bytearray (PyObject *, std::uint32_t, const mobius::bytearray&);
mobius::datetime::datetime get_arg_as_datetime (PyObject *, std::uint32_t, const mobius::datetime::datetime&);
bool get_arg_as_bool (PyObject *, std::uint32_t, bool);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// C++ <-> Python type conversion
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool pystring_check (PyObject *);
PyObject *pystring_from_std_string (const std::string&);
std::string pystring_as_std_string (PyObject *);

bool pybytes_check (PyObject *);
PyObject *pybytes_from_bytearray (const mobius::bytearray&);
mobius::bytearray pybytes_as_bytearray (PyObject *);

bool pydatetime_check (PyObject *);
PyObject *pydatetime_from_datetime (const mobius::datetime::datetime&);
mobius::datetime::datetime pydatetime_as_datetime (PyObject *);

bool pybool_check (PyObject *);
PyObject *pybool_from_bool (bool);
bool pybool_as_bool (PyObject *);

PyObject *pylong_from_std_uint64_t (std::uint64_t);
std::uint64_t pylong_as_std_uint64_t (PyObject *);

PyObject *pylong_from_int (int);
int pylong_as_int (PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new Python Tuple from C++ pair
//! \param p C++ pair
//! \param pyf1 Function to convert first value
//! \param pyf2 Function to convert second value
//! \return Python Tuple
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
template <typename T, typename F1, typename F2> PyObject *
pytuple_from_cpp_pair (const T& p, F1 pyf1, F2 pyf2)
{
  PyObject *ret = PyTuple_New (2);

  if (ret)
    {
      PyTuple_SetItem (ret, 0, pyf1 (p.first));
      PyTuple_SetItem (ret, 1, pyf2 (p.second));
    }

  return ret;  
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new Python list from C++ container
//! \param container C++ container
//! \param pyfunc Function to convert C++ items to Python objects
//! \return Python list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
template <typename C, typename F> PyObject *
pylist_from_cpp_container (const C& container, F pyfunc)
{
  PyObject *ret = PyList_New (0);

  if (!ret)
    return nullptr;

  for (const auto& item : container)
    {
      PyObject *py_item = pyfunc (item);

      if (!py_item)
        {
          Py_CLEAR (ret);
          return nullptr;
        }

      PyList_Append (ret, py_item);
      Py_DECREF (py_item);
    }

  return ret;  
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create new Python dict from C++ container
//! \param container C++ container
//! \param pyfk Function to convert C++ keys to Python objects
//! \param pyfv Function to convert C++ values to Python objects
//! \return Python dict
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
template <typename C, typename FK, typename FV> PyObject *
pydict_from_cpp_container (const C& container, FK pyfk, FV pyfv)
{
  PyObject *ret = PyDict_New ();

  if (!ret)
    return nullptr;

  for (const auto& p : container)
    {
      PyObject *py_key = pyfk (p.first);

      if (!py_key)
        {
          Py_CLEAR (ret);
          return nullptr;
        }

      PyObject *py_value = pyfv (p.second);

      if (!py_value)
        {
          Py_DECREF (py_key);
          Py_CLEAR (ret);
          return nullptr;
        }

      PyDict_SetItem (ret, py_key, py_value);
      Py_DECREF (py_key);
      Py_DECREF (py_value);
    }

  return ret;
}

} // namespace py
} // namespace mobius

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Deprecated (must use mobius::py::pystring...)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *PyString2_from_std_string (const std::string&);

#endif
