# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import mobius
import string
import json

try:
   # Python 2.6-2.7 
   from HTMLParser import HTMLParser
except ImportError:
   # Python 3
   from html.parser import HTMLParser

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Format Account ID
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def format_account_id (text):
  if text and text[0] in string.digits and ':' in text:
    text = text.split (':', 1)[1]
  return text

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Emoji representation as Unicode char
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
EMOJI_CHARS = {
  'angel'                : '👼',
  'anger'                : '🤯',
  'angry'                : '😠',
  'bear'                 : '🧸',
  'bike'                 : '🚴',
  'blush'                : '😳',
  'bomb'                 : '💣',
  'brokenheart'          : '💔',
  'cake'                 : '🎂',
  'cash'                 : '💰',
  'cat'                  : '🐈',
  'coffee'               : '☕',
  'cool'                 : '😎',
  'cry'                  : '😢',
  'cwl'                  : '😂',
  'dance'                : '🕺',
  'devil'                : '😈',
  'dog'                  : '🐕',
  'drink'                : '🍸',
  'drunk'                : '🥴',
  'dull'                 : '🙄',
  'explode'              : '💣',
  'explosion'            : '💣',
  'facepalm'             : '🤦',
  'ghost'                : '👻',
  'giggle'               : '🤭',
  'happy'                : '🙂',
  'heart'                : '❤',
  'hearteyes'            : '😍',
  'heidy'                : '🐿',
  'hug'                  : '🧸',
  'inlove'               : '🥰',
  'kiss'                 : '😗',
  'ladyvamp'             : '🧛',
  'ladyvampire'          : '🧛',
  'laugh'                : '😃',
  'lips'                 : '💋',
  'lipssealed'           : '🤐',
  'loudlycrying'         : '😭',
  'mail'                 : '✉',
  'mmm'                  : '😋',
  'monkey'               : '🐒',
  'muscle'               : '💪',
  'music'                : '🎶',
  'nerd'                 : '🤓',
  'nerdy'                : '🤓',
  'ninja'                : '🥷',
  'no'                   : '👎',
  'ok'                   : '👌',
  'party'                : '🥳',
  'pizza'                : '🍕',
  'praying'              : '🙏',
  'puke'                 : '🤮',
  'rain'                 : '🌧',
  'rofl'                 : '🤣',
  'sad'                  : '😧',
  'skull'                : '💀',
  'sleepy'               : '😪',
  'smile'                : '😄',
  'smirk'                : '😏',
  'speechless'           : '😐',
  'squirrel'             : '🐿',
  'star'                 : '⭐',
  'stareyes'             : '🤩',
  'sun'                  : '🌞',
  'surprised'            : '😲',
  'swear'                : '🤬',
  'sweat'                : '😓',
  'think'                : '🤔',
  'time'                 : '⏲',
  'tongueout'            : '😛',
  'unamused'             : '😒',
  'vampire'              : '🧛',
  'victory'              : '✌',
  'wasntme'              : '🙄',
  'wink'                 : '😉',
  'worry'                : '😟',
  'xd'                   : '😆',
  'yawn'                 : '🥱',
  'yes'                  : '👍',
  'yoga'                 : '🧘',
}

UNKNOWN_EMOJIS = set ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Flags representation as Unicode char
# @see ISO-3166
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
FLAG_CHARS = {
  "ad" : "🇦🇩",
  "ae" : "🇦🇪",
  "af" : "🇦🇫",
  "ag" : "🇦🇬",
  "ai" : "🇦🇮",
  "al" : "🇦🇱",
  "am" : "🇦🇲",
  "ao" : "🇦🇴",
  "aq" : "🇦🇶",
  "ar" : "🇦🇷",
  "as" : "🇦🇸",
  "at" : "🇦🇹",
  "au" : "🇦🇺",
  "aw" : "🇦🇼",
  "ax" : "🇦🇽",
  "az" : "🇦🇿",
  "ba" : "🇧🇦",
  "bb" : "🇧🇧",
  "bd" : "🇧🇩",
  "be" : "🇧🇪",
  "bf" : "🇧🇫",
  "bg" : "🇧🇬",
  "bh" : "🇧🇭",
  "bi" : "🇧🇮",
  "bj" : "🇧🇯",
  "bl" : "🇧🇱",
  "bm" : "🇧🇲",
  "bn" : "🇧🇳",
  "bo" : "🇧🇴",
  "bq" : "🇧🇶",
  "br" : "🇧🇷",
  "bs" : "🇧🇸",
  "bt" : "🇧🇹",
  "bv" : "🇧🇻",
  "bw" : "🇧🇼",
  "by" : "🇧🇾",
  "bz" : "🇧🇿",
  "ca" : "🇨🇦",
  "cc" : "🇨🇨",
  "cd" : "🇨🇩",
  "cf" : "🇨🇫",
  "cg" : "🇨🇬",
  "ch" : "🇨🇭",
  "ci" : "🇨🇮",
  "ck" : "🇨🇰",
  "cl" : "🇨🇱",
  "cm" : "🇨🇲",
  "cn" : "🇨🇳",
  "co" : "🇨🇴",
  "cr" : "🇨🇷",
  "cu" : "🇨🇺",
  "cv" : "🇨🇻",
  "cw" : "🇨🇼",
  "cx" : "🇨🇽",
  "cy" : "🇨🇾",
  "cz" : "🇨🇿",
  "de" : "🇩🇪",
  "dj" : "🇩🇯",
  "dk" : "🇩🇰",
  "dm" : "🇩🇲",
  "do" : "🇩🇴",
  "dz" : "🇩🇿",
  "ec" : "🇪🇨",
  "ee" : "🇪🇪",
  "eg" : "🇪🇬",
  "eh" : "🇪🇭",
  "er" : "🇪🇷",
  "es" : "🇪🇸",
  "et" : "🇪🇹",
  "fi" : "🇫🇮",
  "fj" : "🇫🇯",
  "fk" : "🇫🇰",
  "fm" : "🇫🇲",
  "fo" : "🇫🇴",
  "fr" : "🇫🇷",
  "ga" : "🇬🇦",
  "gb" : "🇬🇧",
  "gd" : "🇬🇩",
  "ge" : "🇬🇪",
  "gf" : "🇬🇫",
  "gg" : "🇬🇬",
  "gh" : "🇬🇭",
  "gi" : "🇬🇮",
  "gl" : "🇬🇱",
  "gm" : "🇬🇲",
  "gn" : "🇬🇳",
  "gp" : "🇬🇵",
  "gq" : "🇬🇶",
  "gr" : "🇬🇷",
  "gs" : "🇬🇸",
  "gt" : "🇬🇹",
  "gu" : "🇬🇺",
  "gw" : "🇬🇼",
  "gy" : "🇬🇾",
  "hk" : "🇭🇰",
  "hm" : "🇭🇲",
  "hn" : "🇭🇳",
  "hr" : "🇭🇷",
  "ht" : "🇭🇹",
  "hu" : "🇭🇺",
  "id" : "🇮🇩",
  "ie" : "🇮🇪",
  "il" : "🇮🇱",
  "im" : "🇮🇲",
  "in" : "🇮🇳",
  "io" : "🇮🇴",
  "iq" : "🇮🇶",
  "ir" : "🇮🇷",
  "is" : "🇮🇸",
  "it" : "🇮🇹",
  "je" : "🇯🇪",
  "jm" : "🇯🇲",
  "jo" : "🇯🇴",
  "jp" : "🇯🇵",
  "ke" : "🇰🇪",
  "kg" : "🇰🇬",
  "kh" : "🇰🇭",
  "ki" : "🇰🇮",
  "km" : "🇰🇲",
  "kn" : "🇰🇳",
  "kp" : "🇰🇵",
  "kr" : "🇰🇷",
  "kw" : "🇰🇼",
  "ky" : "🇰🇾",
  "kz" : "🇰🇿",
  "la" : "🇱🇦",
  "lb" : "🇱🇧",
  "lc" : "🇱🇨",
  "li" : "🇱🇮",
  "lk" : "🇱🇰",
  "lr" : "🇱🇷",
  "ls" : "🇱🇸",
  "lt" : "🇱🇹",
  "lu" : "🇱🇺",
  "lv" : "🇱🇻",
  "ly" : "🇱🇾",
  "ma" : "🇲🇦",
  "mc" : "🇲🇨",
  "md" : "🇲🇩",
  "me" : "🇲🇪",
  "mf" : "🇲🇫",
  "mg" : "🇲🇬",
  "mh" : "🇲🇭",
  "mk" : "🇲🇰",
  "ml" : "🇲🇱",
  "mm" : "🇲🇲",
  "mn" : "🇲🇳",
  "mo" : "🇲🇴",
  "mp" : "🇲🇵",
  "mq" : "🇲🇶",
  "mr" : "🇲🇷",
  "ms" : "🇲🇸",
  "mt" : "🇲🇹",
  "mu" : "🇲🇺",
  "mv" : "🇲🇻",
  "mw" : "🇲🇼",
  "mx" : "🇲🇽",
  "my" : "🇲🇾",
  "mz" : "🇲🇿",
  "na" : "🇳🇦",
  "nc" : "🇳🇨",
  "ne" : "🇳🇪",
  "nf" : "🇳🇫",
  "ng" : "🇳🇬",
  "ni" : "🇳🇮",
  "nl" : "🇳🇱",
  "no" : "🇳🇴",
  "np" : "🇳🇵",
  "nr" : "🇳🇷",
  "nu" : "🇳🇺",
  "nz" : "🇳🇿",
  "om" : "🇴🇲",
  "pa" : "🇵🇦",
  "pe" : "🇵🇪",
  "pf" : "🇵🇫",
  "pg" : "🇵🇬",
  "ph" : "🇵🇭",
  "pk" : "🇵🇰",
  "pl" : "🇵🇱",
  "pm" : "🇵🇲",
  "pn" : "🇵🇳",
  "pr" : "🇵🇷",
  "ps" : "🇵🇸",
  "pt" : "🇵🇹",
  "pw" : "🇵🇼",
  "py" : "🇵🇾",
  "qa" : "🇶🇦",
  "re" : "🇷🇪",
  "ro" : "🇷🇴",
  "rs" : "🇷🇸",
  "ru" : "🇷🇺",
  "rw" : "🇷🇼",
  "sa" : "🇸🇦",
  "sb" : "🇸🇧",
  "sc" : "🇸🇨",
  "sd" : "🇸🇩",
  "se" : "🇸🇪",
  "sg" : "🇸🇬",
  "sh" : "🇸🇭",
  "si" : "🇸🇮",
  "sj" : "🇸🇯",
  "sk" : "🇸🇰",
  "sl" : "🇸🇱",
  "sm" : "🇸🇲",
  "sn" : "🇸🇳",
  "so" : "🇸🇴",
  "sr" : "🇸🇷",
  "ss" : "🇸🇸",
  "st" : "🇸🇹",
  "sv" : "🇸🇻",
  "sx" : "🇸🇽",
  "sy" : "🇸🇾",
  "sz" : "🇸🇿",
  "tc" : "🇹🇨",
  "td" : "🇹🇩",
  "tf" : "🇹🇫",
  "tg" : "🇹🇬",
  "th" : "🇹🇭",
  "tj" : "🇹🇯",
  "tk" : "🇹🇰",
  "tl" : "🇹🇱",
  "tm" : "🇹🇲",
  "tn" : "🇹🇳",
  "to" : "🇹🇴",
  "tr" : "🇹🇷",
  "tt" : "🇹🇹",
  "tv" : "🇹🇻",
  "tw" : "🇹🇼",
  "tz" : "🇹🇿",
  "ua" : "🇺🇦",
  "ug" : "🇺🇬",
  "um" : "🇺🇲",
  "us" : "🇺🇸",
  "uy" : "🇺🇾",
  "uz" : "🇺🇿",
  "va" : "🇻🇦",
  "vc" : "🇻🇨",
  "ve" : "🇻🇪",
  "vg" : "🇻🇬",
  "vi" : "🇻🇮",
  "vn" : "🇻🇳",
  "vu" : "🇻🇺",
  "wf" : "🇼🇫",
  "ws" : "🇼🇸",
  "ye" : "🇾🇪",
  "yt" : "🇾🇹",
  "za" : "🇿🇦",
  "zm" : "🇿🇲",
  "zw" : "🇿🇼"
}

UNKNOWN_FLAGS = set ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Skype entities
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ENTITIES = {
   'amp' : '&',
   'lt' : '<',
   'gt' : '>',
   'apos' : "'",
   'quot' : '"'
}

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Unescape entities
# @param text Escaped text (with &amp;, ...)
# @return Unescaped text
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def unescape (text):
  html = HTMLParser ()
  return html.unescape (text)
  
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Skype message parser
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class MessageParser (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, text):
    self.__elements = []
    self.__text = text
    
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse message
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def parse (self):
    self.__parser = mobius.decoder.sgml.parser (self.__text)
    e = self.__parser.get ()

    while e.type != 0:

      if e.type == 1:		# text
        self.__parse_text (e)

      elif e.type == 2:		# <tag>
        self.__parse_tag_open (e)

      elif e.type == 3:		# </tag>
        self.__parse_tag_close (e)

      elif e.type == 4:		# <tag/>
        self.__parse_tag_empty (e)

      elif e.type == 5:		# &entity;
        self.__parse_entity (e)

      e = self.__parser.get ()

    return self.__elements

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse popcard
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def parse_popcard (self):
    l = json.loads (self.__text)

    if len (l) > 0:
      c = l[0]
      text = c.get ('content', {}).get (u'text')

      if text:
        element = {u'type' : u'system', u'text' : text}
        self.add_element (element)

    return self.__elements

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <text>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __parse_text (self, e):
    element = {u'type' : u'text', u'text' : unicode (e.text, 'utf-8')}
    self.add_element (element)    

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <tag>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __parse_tag_open (self, e):
    tag = unicode (e.text, 'utf-8')

    f = getattr (self, '_parse_%s' % tag, None)
    if f:
      f (e)

    else:
      mobius.core.logf ('DEV app.skype: unknown tag open <%s>' % tag.encode ('utf-8'))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse </tag>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __parse_tag_close (self, e):
    element = None
    tag = unicode (e.text, 'utf-8')

    if tag == u'b':
      element = {u'type' : u'end/b'}

    elif tag == u'i':
      element = {u'type' : u'end/i'}

    elif tag == u's':
      element = {u'type' : u'end/s'}

    elif tag == u'quote':
      element = {u'type' : u'end/quote'}

    else:
      mobius.core.logf ('DEV app.skype: unknown tag close </%s>' % tag.encode ('utf-8'))

    if element:
      self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <tag/>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __parse_tag_empty (self, e):
    tag = unicode (e.text, 'utf-8')
    mobius.core.logf ('DEV app.skype: unknown tag empty <%s/>' % tag.encode ('utf-8'))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse entity
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __parse_entity (self, e):
    element = {
       u'type' : u'text',
       u'text' : unescape (u'&' + unicode (e.text, 'utf-8') + u';')
    }
    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get value from <tag>value</tag>
  # @param e Current element
  # @return Value
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_tag_value (self, e):
    tag = unicode (e.text, 'utf-8')
    value = None
    
    while e.type != 0 and (e.type != 3 or unicode (e.text, 'utf-8') != tag):
      if e.type == 1:
        value = unicode (e.text, 'utf-8')
      e = self.__parser.get ()

    return value
  
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add text element to list
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_element (self, element):

    # get previous element
    if len (self.__elements) > 0:
      p_element = self.__elements[-1]
    else:
      p_element = None

    # merge two consecutive 'text' or 'system'
    if p_element and element[u'type'] == p_element[u'type'] == u'text':
        p_element[u'text'] = p_element[u'text'] + element[u'text']

    elif p_element and element[u'type'] == p_element[u'type'] == u'system':
        p_element[u'text'] = p_element[u'text'] + '. ' + element[u'text']

    # otherwise, create new element
    else:
      self.__elements.append (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <b> node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_b (self, e):
    element = {u'type' : u'start/b'}
    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <i> node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_i (self, e):
    element = {u'type' : u'start/i'}
    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <s> node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_s (self, e):
    element = {u'type' : u'start/s'}
    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <a> node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_a (self, e):
    element = {u'type' : u'href', u'url' : unescape (e.attributes.get (u'href'))}
    self.add_element (element)

    value = self.__get_tag_value (e)  # ignore text and </a>

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <ss></ss> tag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_ss (self, e):
    element = {u'type' : u'emoji'}

    # get ss_type
    ss_type = e.attributes.get (u'type')
  
    if ss_type in EMOJI_CHARS:
      element[u'code'] = EMOJI_CHARS.get (ss_type)

    elif ss_type not in UNKNOWN_EMOJIS:
      mobius.core.logf ('DEV app.skype: unknown emoji "%s"' % ss_type)
      UNKNOWN_EMOJIS.add (ss_type)

    # get text
    text = self.__get_tag_value (e)
    if text:
      element[u'text'] = text

    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <flag></flag> tag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_flag (self, e):
    element = {u'type' : u'flag'}

    # get country
    country = e.attributes.get (u'country')
  
    if country in FLAG_CHARS:
      element[u'code'] = FLAG_CHARS.get (country)

    elif country not in UNKNOWN_FLAGS:
      mobius.core.logf ('DEV app.skype: unknown flag "%s"' % country)
      UNKNOWN_FLAGS.add (country)

    # get text
    text = self.__get_tag_value (e)
    if text:
      element[u'text'] = text

    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <partlist> node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_partlist (self, e):
    parts = []

    # root node = <partlist>
    if e.type != 2 or e.text != 'partlist':
      mobius.core.logf ('WRN app.skype: invalid <partlist> ' + text)
      return

    # get all <part> nodes
    e = self.__parser.get ()

    while e.type != 0 and (e.type != 3 or e.text != 'partlist'):

      if e.type == 2 and e.text == 'part':	# <part>
        part_id = e.attributes.get ('identity')
        part_name = u''

      elif e.type == 2 and e.text == 'name':	# <name>
        part_name = self.__get_tag_value (e)
        
      elif e.type == 3 and e.text == 'part':	# </part>
        parts.append ((part_id, part_name))

      e = self.__parser.get ()

    # create element
    text = u'Participants: ' + u', '.join (account + u' (' + name + u')' if name else account for (account, name) in sorted (parts))

    element = {u'type' : u'system', u'text' : text}
    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <contacts> node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_contacts (self, e):
    contacts = []

    # root node = <contacts>
    if e.type != 2 or e.text != 'contacts':
      mobius.core.logf ('WRN app.skype: invalid <contacts> ' + e.text)
      return

    # get all <c> nodes
    e = self.__parser.get ()

    while e.type != 0 and (e.type != 3 or e.text != 'contacts'):

      if e.type in (2, 4) and e.text == 'c':		# <c> or <c/>
        t = e.attributes.get ('t')

        if t != 's':
          mobius.core.logf ('DEV app.skype: new contact "t" value: ' + t)

        account_id = e.attributes.get ('s')
        account_name = e.attributes.get ('f')
        contacts.append ((account_id, account_name))

      e = self.__parser.get ()

    # create element
    text = u'Contacts: ' + u', '.join (account + u' (' + name + u')' if name else account for (account, name) in sorted (contacts))

    element = {u'type' : u'system', u'text' : text}
    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <files> node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_files (self, e):
    filelist = []

    e = self.__parser.get ()
    
    # get all <file> nodes
    while e.type != 0 and (e.type != 3 or e.text != 'files'):

      if e.type in (2, 4) and e.text == 'file':	# <file> or <file/>
        size = int (e.attributes.get ('size', 0))
        status = e.attributes.get ('status')
        timestamp = e.attributes.get ('tid')

        if e.type == 2:
          name = self.__get_tag_value (e)
        else:
          name = None
    
        filelist.append ((size, status, timestamp, name))

      e = self.__parser.get ()

    # create element
    text = None

    if len (filelist) == 1:
      size, status, timestamp, name = filelist[0]
      text = u'File "%s" sent (size: %d bytes)' % (name, size)

    elif len (filelist) > 1:
      text = u'Files %s sent' % ', '.join ('"%s"' % name for size, status, timestamp, name in filelist)

    if text:
      element = {u'type' : u'system', u'text' : text}
      self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <quote> node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_quote (self, e):
    timestamp = e.attributes.get ('timestamp')
    author_id = e.attributes.get ('author')
    author_name = e.attributes.get ('authorname')

    author = author_id
    if author_name:
      author += ' (' + author_name + ')'

    element = {u'type' : u'start/quote', u'author' : author}
    if timestamp:
      element['timestamp'] = mobius.datetime.new_datetime_from_unix_timestamp (int (timestamp))

    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <legacyquote></legacyquote> tag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_legacyquote (self, e):
    self.__get_tag_value (e)	# ignore text and </legacyquote>

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <sms> tag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_sms (self, e):
    msg = e.attributes.get ('alt')
    timestamp = None
    targets = []
    
    # process until </sms>
    e = self.__parser.get ()

    while e.type != 0 and (e.type != 3 or e.text != 'sms'):
      etext = unicode (e.text, 'utf-8')

      if e.type == 2 and etext == u'sendtimestamp':
        sendtimestamp = self.__get_tag_value (e)

      elif e.type == 2 and etext == u'target':
        target = self.__get_tag_value (e)
        targets.append (target)

      e = self.__parser.get ()

    # create element
    if timestamp:
      timestamp = mobius.datetime.new_datetime_from_unix_timestamp (int (timestamp))
      text = u'SMS message sent at %s to %s: %s' % (timestamp, ', '.join (targets), msg)

    else:
      text = u'SMS message sent to %s: %s' % (', '.join (targets), msg)

    element = {u'type' : u'system', u'text' : text}
    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <URIObject> tag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_URIObject (self, e):
    type = e.attributes.get ('type')
    uri = e.attributes.get ('uri')
    url_thumbnail = e.attributes.get ('url_thumbnail')
    ams_id = e.attributes.get ('ams_id')
    name = None
    meta_type = None
    meta_name = None
    filesize = None
    description = None
    title = None

    # process until </URIObject>
    e = self.__parser.get ()

    while e.type != 0 and (e.type != 3 or e.text != 'URIObject'):
      etext = unicode (e.text, 'utf-8')

      if e.type == 2:		# <tag>
        if etext == u'a':
          pass

        elif etext == u'Description':
          description = self.__get_tag_value (e)

        elif etext == u'Title':
          title = self.__get_tag_value (e)

        elif etext == u'FileSize':
          value = self.__get_tag_value (e)
          if value:
            filesize = int (value)

        elif etext == u'OriginalName':
          name = e.attributes.get ('v')

        elif etext == u'target':
          target = self.__get_tag_value (e)
          targets.append (target)

        elif etext == u'meta':
          meta_type = e.attributes.get ('type')
          meta_name = e.attributes.get ('originalName')

        else:
          mobius.core.logf ('DEV app.skype: unhandled URIObject sub-tag <%s>: %s' % (e.text, e.attributes.value))

      elif e.type == 4:		# <tag/>
        if etext == u'Description':
          pass
  
        elif etext == u'FileSize':
          filesize = int (e.attributes.get ('v'))

        elif etext == u'meta':
          meta_type = e.attributes.get ('type')
          meta_name = e.attributes.get ('originalName')

        elif etext == u'OriginalName':
          name = e.attributes.get ('v')

        elif etext == u'Title':
          pass
  
        else:
          mobius.core.logf ('DEV app.skype: unhandled URIObject sub-tag <%s/>: %s' % (e.text, e.attributes.value))

      e = self.__parser.get ()

    # create element
    text = u'File "%s" shared' % (name or meta_name)

    if type != None:
      text += u'\nType: %s' % type

    if filesize != None:
      text += u'\nSize: %d bytes' % filesize

    if title:
      text += u'\nTitle: %s' % title

    if description:
      text += u'\nDescription: %s' % description

    text += u'\nURL: %s' % uri

    element = {u'type' : u'system', u'text' : text}
    self.add_element (element)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Parse <deletemember> tag
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def _parse_deletemember (self, e):
    timestamp = None
    initiator = None
    target = None

    # process until </deletemember>
    e = self.__parser.get ()

    while e.type != 0 and (e.type != 3 or e.text != 'deletemember'):
      etext = unicode (e.text, 'utf-8')
      
      if e.type == 2:		# <tag>

        if etext == u'eventtime':
          value = self.__get_tag_value (e)
          timestamp = mobius.datetime.new_datetime_from_unix_timestamp (int (value) // 1000)

        elif etext == u'initiator':
          initiator = format_account_id (self.__get_tag_value (e))

        elif etext == u'target':
          target = format_account_id (self.__get_tag_value (e))

      e = self.__parser.get ()

    # create element
    text = u'Member "%s" deleted from chat' % target
    
    if initiator:
      text += u' by user "%s"' % initiator
      
    if timestamp:
      text += u' in %s' % timestamp

    element = {u'type' : u'system', u'text' : text}
    self.add_element (element)
