// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \file C++ API <i>mobius.decoder</i> module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <pymobius.h>
#include "data_decoder.h"
#include "mfc_decoder.h"
#include "qdatastream.h"
#include "sourcecode.h"
#include "sgml/module.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Module methods
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *decoder_func_base16 (PyObject *, PyObject *);
PyObject *decoder_func_base32 (PyObject *, PyObject *);
PyObject *decoder_func_base32hex (PyObject *, PyObject *);
PyObject *decoder_func_base64 (PyObject *, PyObject *);
PyObject *decoder_func_base64url (PyObject *, PyObject *);

static PyMethodDef module_methods[] =
{
  {
    "base16",
    decoder_func_base16,
    METH_VARARGS,
    "Decode base16 string"
  },
  {
    "base32",
    decoder_func_base32,
    METH_VARARGS,
    "Decode base32 string"
  },
  {
    "base32hex",
    decoder_func_base32hex,
    METH_VARARGS,
    "Decode base32hex string"
  },
  {
    "base64",
    decoder_func_base64,
    METH_VARARGS,
    "Decode base64 string"
  },
  {
    "base64url",
    decoder_func_base64url,
    METH_VARARGS,
    "Decode base64url string"
  },
  {
    NULL,
    NULL,
    0,
    NULL
  } // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Module datatypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr struct
{
  const char *name;
  PyTypeObject *type;
} types[] =

{
  {"data_decoder", &decoder_data_decoder_t},
  {"mfc_decoder", &decoder_mfc_decoder_t},
  {"qdatastream", &decoder_qdatastream_t},
  {"sourcecode", &decoder_sourcecode_t},
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create mobius.decoder module
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
new_decoder_module ()
{
  // initialize module mobius.decoder
  PyObject* module = Py_InitModule3 (
                       "mobius.decoder",
                       module_methods,
                       "Mobius Forensic Toolkit mobius.decoder module"
                     );

  // create submodules
  PyObject *sgml_module = new_decoder_sgml_module ();
  PyModule_AddObject (module, "sgml", sgml_module);

  // build types
  for (const auto& t : types)
    {
      if (PyType_Ready (t.type) < 0)
        return nullptr;
    }

  // add types
  for (const auto& t : types)
    {
      Py_INCREF (t.type);
      PyModule_AddObject (module, t.name, (PyObject *) t.type);
    }

  // return module
  return module;
}
