// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "hash_sha1.h"

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// initialization constants (RFC 3174 - section 6.1)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t A_VALUE = 0x67452301;
static constexpr std::uint32_t B_VALUE = 0xefcdab89;
static constexpr std::uint32_t C_VALUE = 0x98badcfe;
static constexpr std::uint32_t D_VALUE = 0x10325476;
static constexpr std::uint32_t E_VALUE = 0xc3d2e1f0;
static constexpr std::uint32_t BLOCK_SIZE = 64;  // (RFC 3174 - section 4)

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief rotate left a 32-bit value
//! \param v value
//! \param n number of bits to rotate
//! \return value rotated left by n bits
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint32_t
left_rotate (std::uint32_t v, std::uint32_t n) noexcept
{
  return (v << n) | (v >> (32 - n));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief default constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
hash_sha1::hash_sha1 () noexcept
  : hash_block (BLOCK_SIZE),
    a_ (A_VALUE),
    b_ (B_VALUE),
    c_ (C_VALUE),
    d_ (D_VALUE),
    e_ (E_VALUE),
    size_ (0)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief reset hash value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_sha1::_reset () noexcept
{
  a_ = A_VALUE;
  b_ = B_VALUE;
  c_ = C_VALUE;
  d_ = D_VALUE;
  e_ = E_VALUE;
  size_ = 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief update hash value
//! \param data data block
//! \see RFC 3174 - section 6.2 (Method 2)
//! \see http://software.intel.com/en-us/articles/improving-the-performance-of-the-secure-hash-algorithm-1/
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_sha1::_update_block (const mobius::bytearray& data) noexcept
{
  std::uint32_t a = a_;
  std::uint32_t b = b_;
  std::uint32_t c = c_;
  std::uint32_t d = d_;
  std::uint32_t e = e_;
  std::uint32_t temp;

  // W[0..79]
  const std::uint32_t W [] =
  {
    (std::uint32_t (data[0]) << 24) |
    (std::uint32_t (data[1]) << 16) |
    (std::uint32_t (data[2]) << 8) |
    (std::uint32_t (data[3])),
    (std::uint32_t (data[4]) << 24) |
    (std::uint32_t (data[5]) << 16) |
    (std::uint32_t (data[6]) << 8) |
    (std::uint32_t (data[7])),
    (std::uint32_t (data[8]) << 24) |
    (std::uint32_t (data[9]) << 16) |
    (std::uint32_t (data[10]) << 8) |
    (std::uint32_t (data[11])),
    (std::uint32_t (data[12]) << 24) |
    (std::uint32_t (data[13]) << 16) |
    (std::uint32_t (data[14]) << 8) |
    (std::uint32_t (data[15])),
    (std::uint32_t (data[16]) << 24) |
    (std::uint32_t (data[17]) << 16) |
    (std::uint32_t (data[18]) << 8) |
    (std::uint32_t (data[19])),
    (std::uint32_t (data[20]) << 24) |
    (std::uint32_t (data[21]) << 16) |
    (std::uint32_t (data[22]) << 8) |
    (std::uint32_t (data[23])),
    (std::uint32_t (data[24]) << 24) |
    (std::uint32_t (data[25]) << 16) |
    (std::uint32_t (data[26]) << 8) |
    (std::uint32_t (data[27])),
    (std::uint32_t (data[28]) << 24) |
    (std::uint32_t (data[29]) << 16) |
    (std::uint32_t (data[30]) << 8) |
    (std::uint32_t (data[31])),
    (std::uint32_t (data[32]) << 24) |
    (std::uint32_t (data[33]) << 16) |
    (std::uint32_t (data[34]) << 8) |
    (std::uint32_t (data[35])),
    (std::uint32_t (data[36]) << 24) |
    (std::uint32_t (data[37]) << 16) |
    (std::uint32_t (data[38]) << 8) |
    (std::uint32_t (data[39])),
    (std::uint32_t (data[40]) << 24) |
    (std::uint32_t (data[41]) << 16) |
    (std::uint32_t (data[42]) << 8) |
    (std::uint32_t (data[43])),
    (std::uint32_t (data[44]) << 24) |
    (std::uint32_t (data[45]) << 16) |
    (std::uint32_t (data[46]) << 8) |
    (std::uint32_t (data[47])),
    (std::uint32_t (data[48]) << 24) |
    (std::uint32_t (data[49]) << 16) |
    (std::uint32_t (data[50]) << 8) |
    (std::uint32_t (data[51])),
    (std::uint32_t (data[52]) << 24) |
    (std::uint32_t (data[53]) << 16) |
    (std::uint32_t (data[54]) << 8) |
    (std::uint32_t (data[55])),
    (std::uint32_t (data[56]) << 24) |
    (std::uint32_t (data[57]) << 16) |
    (std::uint32_t (data[58]) << 8) |
    (std::uint32_t (data[59])),
    (std::uint32_t (data[60]) << 24) |
    (std::uint32_t (data[61]) << 16) |
    (std::uint32_t (data[62]) << 8) |
    (std::uint32_t (data[63])),
    left_rotate (W[13] ^ W[8] ^ W[2] ^ W[0], 1),
    left_rotate (W[14] ^ W[9] ^ W[3] ^ W[1], 1),
    left_rotate (W[15] ^ W[10] ^ W[4] ^ W[2], 1),
    left_rotate (W[16] ^ W[11] ^ W[5] ^ W[3], 1),
    left_rotate (W[17] ^ W[12] ^ W[6] ^ W[4], 1),
    left_rotate (W[18] ^ W[13] ^ W[7] ^ W[5], 1),
    left_rotate (W[19] ^ W[14] ^ W[8] ^ W[6], 1),
    left_rotate (W[20] ^ W[15] ^ W[9] ^ W[7], 1),
    left_rotate (W[21] ^ W[16] ^ W[10] ^ W[8], 1),
    left_rotate (W[22] ^ W[17] ^ W[11] ^ W[9], 1),
    left_rotate (W[23] ^ W[18] ^ W[12] ^ W[10], 1),
    left_rotate (W[24] ^ W[19] ^ W[13] ^ W[11], 1),
    left_rotate (W[25] ^ W[20] ^ W[14] ^ W[12], 1),
    left_rotate (W[26] ^ W[21] ^ W[15] ^ W[13], 1),
    left_rotate (W[27] ^ W[22] ^ W[16] ^ W[14], 1),
    left_rotate (W[28] ^ W[23] ^ W[17] ^ W[15], 1),
    left_rotate (W[26] ^ W[16] ^ W[4] ^ W[0], 2),
    left_rotate (W[27] ^ W[17] ^ W[5] ^ W[1], 2),
    left_rotate (W[28] ^ W[18] ^ W[6] ^ W[2], 2),
    left_rotate (W[29] ^ W[19] ^ W[7] ^ W[3], 2),
    left_rotate (W[30] ^ W[20] ^ W[8] ^ W[4], 2),
    left_rotate (W[31] ^ W[21] ^ W[9] ^ W[5], 2),
    left_rotate (W[32] ^ W[22] ^ W[10] ^ W[6], 2),
    left_rotate (W[33] ^ W[23] ^ W[11] ^ W[7], 2),
    left_rotate (W[34] ^ W[24] ^ W[12] ^ W[8], 2),
    left_rotate (W[35] ^ W[25] ^ W[13] ^ W[9], 2),
    left_rotate (W[36] ^ W[26] ^ W[14] ^ W[10], 2),
    left_rotate (W[37] ^ W[27] ^ W[15] ^ W[11], 2),
    left_rotate (W[38] ^ W[28] ^ W[16] ^ W[12], 2),
    left_rotate (W[39] ^ W[29] ^ W[17] ^ W[13], 2),
    left_rotate (W[40] ^ W[30] ^ W[18] ^ W[14], 2),
    left_rotate (W[41] ^ W[31] ^ W[19] ^ W[15], 2),
    left_rotate (W[42] ^ W[32] ^ W[20] ^ W[16], 2),
    left_rotate (W[43] ^ W[33] ^ W[21] ^ W[17], 2),
    left_rotate (W[44] ^ W[34] ^ W[22] ^ W[18], 2),
    left_rotate (W[45] ^ W[35] ^ W[23] ^ W[19], 2),
    left_rotate (W[46] ^ W[36] ^ W[24] ^ W[20], 2),
    left_rotate (W[47] ^ W[37] ^ W[25] ^ W[21], 2),
    left_rotate (W[48] ^ W[38] ^ W[26] ^ W[22], 2),
    left_rotate (W[49] ^ W[39] ^ W[27] ^ W[23], 2),
    left_rotate (W[50] ^ W[40] ^ W[28] ^ W[24], 2),
    left_rotate (W[51] ^ W[41] ^ W[29] ^ W[25], 2),
    left_rotate (W[52] ^ W[42] ^ W[30] ^ W[26], 2),
    left_rotate (W[53] ^ W[43] ^ W[31] ^ W[27], 2),
    left_rotate (W[54] ^ W[44] ^ W[32] ^ W[28], 2),
    left_rotate (W[55] ^ W[45] ^ W[33] ^ W[29], 2),
    left_rotate (W[56] ^ W[46] ^ W[34] ^ W[30], 2),
    left_rotate (W[57] ^ W[47] ^ W[35] ^ W[31], 2),
    left_rotate (W[58] ^ W[48] ^ W[36] ^ W[32], 2),
    left_rotate (W[59] ^ W[49] ^ W[37] ^ W[33], 2),
    left_rotate (W[60] ^ W[50] ^ W[38] ^ W[34], 2),
    left_rotate (W[61] ^ W[51] ^ W[39] ^ W[35], 2),
    left_rotate (W[62] ^ W[52] ^ W[40] ^ W[36], 2),
    left_rotate (W[63] ^ W[53] ^ W[41] ^ W[37], 2),
    left_rotate (W[64] ^ W[54] ^ W[42] ^ W[38], 2),
    left_rotate (W[65] ^ W[55] ^ W[43] ^ W[39], 2),
    left_rotate (W[66] ^ W[56] ^ W[44] ^ W[40], 2),
    left_rotate (W[67] ^ W[57] ^ W[45] ^ W[41], 2),
    left_rotate (W[68] ^ W[58] ^ W[46] ^ W[42], 2),
    left_rotate (W[69] ^ W[59] ^ W[47] ^ W[43], 2),
    left_rotate (W[70] ^ W[60] ^ W[48] ^ W[44], 2),
    left_rotate (W[71] ^ W[61] ^ W[49] ^ W[45], 2),
    left_rotate (W[72] ^ W[62] ^ W[50] ^ W[46], 2),
    left_rotate (W[73] ^ W[63] ^ W[51] ^ W[47], 2)
  };

  // unrolled loop 0..19
  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[0];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[1];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[2];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[3];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[4];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[5];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[6];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[7];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[8];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[9];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[10];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[11];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[12];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[13];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[14];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[15];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[16];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[17];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[18];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (~b & d)) + e + 0x5a827999 + W[19];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  // unrolled loop 20..39
  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[20];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[21];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[22];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[23];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[24];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[25];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[26];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[27];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[28];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[29];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[30];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[31];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[32];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[33];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[34];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[35];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[36];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[37];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[38];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0x6ed9eba1 + W[39];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  // unrolled loop 40..59
  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[40];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[41];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[42];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[43];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[44];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[45];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[46];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[47];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[48];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[49];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[50];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[51];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[52];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[53];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[54];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[55];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[56];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[57];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[58];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + ((b & c) | (b & d) | (c & d)) + e + 0x8f1bbcdc + W[59];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  // unrolled loop 60..79
  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[60];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[61];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[62];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[63];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[64];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[65];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[66];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[67];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[68];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[69];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[70];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[71];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[72];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[73];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[74];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[75];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[76];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[77];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[78];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;

  temp = left_rotate (a, 5) + (b ^ c ^ d) + e + 0xca62c1d6 + W[79];
  e = d;
  d = c;
  c = left_rotate (b, 30);
  b = a;
  a = temp;


  // update keys
  a_ += a;
  b_ += b;
  c_ += c;
  d_ += d;
  e_ += e;

  // update size
  size_ += data.size ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate final hash value
//! \param data remaining data
//! \return digest
// The final block is padded with a 1 bit, \0 bytes and a 64-bit value
// representing the message size in bits.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
hash_sha1::_evaluate (const mobius::bytearray& data)
{
  hash_sha1 tmp (*this);

  // represent message size in bits as a bytearray
  std::uint64_t bits = ((size_ + data.size ()) << 3);

  mobius::bytearray dsize =
  {
    std::uint8_t (bits >> 56),
    std::uint8_t (bits >> 48),
    std::uint8_t (bits >> 40),
    std::uint8_t (bits >> 32),
    std::uint8_t (bits >> 24),
    std::uint8_t (bits >> 16),
    std::uint8_t (bits >> 8),
    std::uint8_t (bits)
  };

  // if the length of the last block size >= 56, generate two blocks:
  // one: data + '1' bit + '0' bits (padding)
  // two: 56 bytes (0) + message size
  if (data.size () >= 56)
    {
      mobius::bytearray padding (64 - data.size ());
      padding.fill (0);
      padding[0] = 0x80;                // 1st bit = "1"
      _update_block (data + padding);

      mobius::bytearray padding2 (56);
      padding2.fill (0);
      _update_block (padding2 + dsize);
    }

  // otherwise, generate just one block:
  // data + '1' bit + '0' bits (padding) + message size
  else
    {
      mobius::bytearray padding (56 - data.size ());
      padding.fill (0);
      padding[0] = 0x80;                // 1st bit = "1"
      _update_block (data + padding + dsize);
    }

  // build digest
  mobius::bytearray digest
  {
    std::uint8_t (a_ >> 24),
    std::uint8_t (a_ >> 16),
    std::uint8_t (a_ >> 8),
    std::uint8_t (a_),
    std::uint8_t (b_ >> 24),
    std::uint8_t (b_ >> 16),
    std::uint8_t (b_ >> 8),
    std::uint8_t (b_),
    std::uint8_t (c_ >> 24),
    std::uint8_t (c_ >> 16),
    std::uint8_t (c_ >> 8),
    std::uint8_t (c_),
    std::uint8_t (d_ >> 24),
    std::uint8_t (d_ >> 16),
    std::uint8_t (d_ >> 8),
    std::uint8_t (d_),
    std::uint8_t (e_ >> 24),
    std::uint8_t (e_ >> 16),
    std::uint8_t (e_ >> 8),
    std::uint8_t (e_)
  };

  *this = tmp;
  return digest;
}

} // namespace crypt
} // namespace mobius
