// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "hash_sha2_224.h"

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// initialization constants (FIPS.180-4 - section 5.3.2)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static constexpr std::uint32_t H0_VALUE = 0xc1059ed8;
static constexpr std::uint32_t H1_VALUE = 0x367cd507;
static constexpr std::uint32_t H2_VALUE = 0x3070dd17;
static constexpr std::uint32_t H3_VALUE = 0xf70e5939;
static constexpr std::uint32_t H4_VALUE = 0xffc00b31;
static constexpr std::uint32_t H5_VALUE = 0x68581511;
static constexpr std::uint32_t H6_VALUE = 0x64f98fa7;
static constexpr std::uint32_t H7_VALUE = 0xbefa4fa4;

//! \brief block size = 64 bytes (512 bits) - FIPS.180-4 - section 1
static constexpr int BLOCK_SIZE = 64;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief rotate right a 32-bit value
//! \param v value
//! \param n number of bits to rotate
//! \return value rotated right by n bits
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
inline constexpr std::uint32_t
rrot (std::uint32_t v, std::uint32_t n) noexcept
{
  return (v >> n) | (v << (32 - n));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief default constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
hash_sha2_224::hash_sha2_224 () noexcept
  : hash_block (BLOCK_SIZE),
    a_ (H0_VALUE),
    b_ (H1_VALUE),
    c_ (H2_VALUE),
    d_ (H3_VALUE),
    e_ (H4_VALUE),
    f_ (H5_VALUE),
    g_ (H6_VALUE),
    h_ (H7_VALUE),
    size_ (0)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief reset hash value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_sha2_224::_reset () noexcept
{
  hash_block::reset ();
  a_ = H0_VALUE;
  b_ = H1_VALUE;
  c_ = H2_VALUE;
  d_ = H3_VALUE;
  e_ = H4_VALUE;
  f_ = H5_VALUE;
  g_ = H6_VALUE;
  h_ = H7_VALUE;
  size_ = 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief update hash value
//! \param data data block
//! \see FIPS.180-4 - section 6.2
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_sha2_224::_update_block (const mobius::bytearray& data) noexcept
{
  std::uint32_t a = a_;
  std::uint32_t b = b_;
  std::uint32_t c = c_;
  std::uint32_t d = d_;
  std::uint32_t e = e_;
  std::uint32_t f = f_;
  std::uint32_t g = g_;
  std::uint32_t h = h_;
  std::uint32_t temp1;
  std::uint32_t temp2;

  // W[0..63]
  const std::uint32_t W [] =
  {
    (std::uint32_t (data[0]) << 24) |
    (std::uint32_t (data[1]) << 16) |
    (std::uint32_t (data[2]) << 8) |
    (std::uint32_t (data[3])),
    (std::uint32_t (data[4]) << 24) |
    (std::uint32_t (data[5]) << 16) |
    (std::uint32_t (data[6]) << 8) |
    (std::uint32_t (data[7])),
    (std::uint32_t (data[8]) << 24) |
    (std::uint32_t (data[9]) << 16) |
    (std::uint32_t (data[10]) << 8) |
    (std::uint32_t (data[11])),
    (std::uint32_t (data[12]) << 24) |
    (std::uint32_t (data[13]) << 16) |
    (std::uint32_t (data[14]) << 8) |
    (std::uint32_t (data[15])),
    (std::uint32_t (data[16]) << 24) |
    (std::uint32_t (data[17]) << 16) |
    (std::uint32_t (data[18]) << 8) |
    (std::uint32_t (data[19])),
    (std::uint32_t (data[20]) << 24) |
    (std::uint32_t (data[21]) << 16) |
    (std::uint32_t (data[22]) << 8) |
    (std::uint32_t (data[23])),
    (std::uint32_t (data[24]) << 24) |
    (std::uint32_t (data[25]) << 16) |
    (std::uint32_t (data[26]) << 8) |
    (std::uint32_t (data[27])),
    (std::uint32_t (data[28]) << 24) |
    (std::uint32_t (data[29]) << 16) |
    (std::uint32_t (data[30]) << 8) |
    (std::uint32_t (data[31])),
    (std::uint32_t (data[32]) << 24) |
    (std::uint32_t (data[33]) << 16) |
    (std::uint32_t (data[34]) << 8) |
    (std::uint32_t (data[35])),
    (std::uint32_t (data[36]) << 24) |
    (std::uint32_t (data[37]) << 16) |
    (std::uint32_t (data[38]) << 8) |
    (std::uint32_t (data[39])),
    (std::uint32_t (data[40]) << 24) |
    (std::uint32_t (data[41]) << 16) |
    (std::uint32_t (data[42]) << 8) |
    (std::uint32_t (data[43])),
    (std::uint32_t (data[44]) << 24) |
    (std::uint32_t (data[45]) << 16) |
    (std::uint32_t (data[46]) << 8) |
    (std::uint32_t (data[47])),
    (std::uint32_t (data[48]) << 24) |
    (std::uint32_t (data[49]) << 16) |
    (std::uint32_t (data[50]) << 8) |
    (std::uint32_t (data[51])),
    (std::uint32_t (data[52]) << 24) |
    (std::uint32_t (data[53]) << 16) |
    (std::uint32_t (data[54]) << 8) |
    (std::uint32_t (data[55])),
    (std::uint32_t (data[56]) << 24) |
    (std::uint32_t (data[57]) << 16) |
    (std::uint32_t (data[58]) << 8) |
    (std::uint32_t (data[59])),
    (std::uint32_t (data[60]) << 24) |
    (std::uint32_t (data[61]) << 16) |
    (std::uint32_t (data[62]) << 8) |
    (std::uint32_t (data[63])),
    W[0] + W[9] + (rrot (W[1], 7) ^ rrot (W[1], 18) ^ (W[1] >> 3)) + (rrot (W[14], 17) ^ rrot (W[14], 19) ^ (W[14] >> 10)),
    W[1] + W[10] + (rrot (W[2], 7) ^ rrot (W[2], 18) ^ (W[2] >> 3)) + (rrot (W[15], 17) ^ rrot (W[15], 19) ^ (W[15] >> 10)),
    W[2] + W[11] + (rrot (W[3], 7) ^ rrot (W[3], 18) ^ (W[3] >> 3)) + (rrot (W[16], 17) ^ rrot (W[16], 19) ^ (W[16] >> 10)),
    W[3] + W[12] + (rrot (W[4], 7) ^ rrot (W[4], 18) ^ (W[4] >> 3)) + (rrot (W[17], 17) ^ rrot (W[17], 19) ^ (W[17] >> 10)),
    W[4] + W[13] + (rrot (W[5], 7) ^ rrot (W[5], 18) ^ (W[5] >> 3)) + (rrot (W[18], 17) ^ rrot (W[18], 19) ^ (W[18] >> 10)),
    W[5] + W[14] + (rrot (W[6], 7) ^ rrot (W[6], 18) ^ (W[6] >> 3)) + (rrot (W[19], 17) ^ rrot (W[19], 19) ^ (W[19] >> 10)),
    W[6] + W[15] + (rrot (W[7], 7) ^ rrot (W[7], 18) ^ (W[7] >> 3)) + (rrot (W[20], 17) ^ rrot (W[20], 19) ^ (W[20] >> 10)),
    W[7] + W[16] + (rrot (W[8], 7) ^ rrot (W[8], 18) ^ (W[8] >> 3)) + (rrot (W[21], 17) ^ rrot (W[21], 19) ^ (W[21] >> 10)),
    W[8] + W[17] + (rrot (W[9], 7) ^ rrot (W[9], 18) ^ (W[9] >> 3)) + (rrot (W[22], 17) ^ rrot (W[22], 19) ^ (W[22] >> 10)),
    W[9] + W[18] + (rrot (W[10], 7) ^ rrot (W[10], 18) ^ (W[10] >> 3)) + (rrot (W[23], 17) ^ rrot (W[23], 19) ^ (W[23] >> 10)),
    W[10] + W[19] + (rrot (W[11], 7) ^ rrot (W[11], 18) ^ (W[11] >> 3)) + (rrot (W[24], 17) ^ rrot (W[24], 19) ^ (W[24] >> 10)),
    W[11] + W[20] + (rrot (W[12], 7) ^ rrot (W[12], 18) ^ (W[12] >> 3)) + (rrot (W[25], 17) ^ rrot (W[25], 19) ^ (W[25] >> 10)),
    W[12] + W[21] + (rrot (W[13], 7) ^ rrot (W[13], 18) ^ (W[13] >> 3)) + (rrot (W[26], 17) ^ rrot (W[26], 19) ^ (W[26] >> 10)),
    W[13] + W[22] + (rrot (W[14], 7) ^ rrot (W[14], 18) ^ (W[14] >> 3)) + (rrot (W[27], 17) ^ rrot (W[27], 19) ^ (W[27] >> 10)),
    W[14] + W[23] + (rrot (W[15], 7) ^ rrot (W[15], 18) ^ (W[15] >> 3)) + (rrot (W[28], 17) ^ rrot (W[28], 19) ^ (W[28] >> 10)),
    W[15] + W[24] + (rrot (W[16], 7) ^ rrot (W[16], 18) ^ (W[16] >> 3)) + (rrot (W[29], 17) ^ rrot (W[29], 19) ^ (W[29] >> 10)),
    W[16] + W[25] + (rrot (W[17], 7) ^ rrot (W[17], 18) ^ (W[17] >> 3)) + (rrot (W[30], 17) ^ rrot (W[30], 19) ^ (W[30] >> 10)),
    W[17] + W[26] + (rrot (W[18], 7) ^ rrot (W[18], 18) ^ (W[18] >> 3)) + (rrot (W[31], 17) ^ rrot (W[31], 19) ^ (W[31] >> 10)),
    W[18] + W[27] + (rrot (W[19], 7) ^ rrot (W[19], 18) ^ (W[19] >> 3)) + (rrot (W[32], 17) ^ rrot (W[32], 19) ^ (W[32] >> 10)),
    W[19] + W[28] + (rrot (W[20], 7) ^ rrot (W[20], 18) ^ (W[20] >> 3)) + (rrot (W[33], 17) ^ rrot (W[33], 19) ^ (W[33] >> 10)),
    W[20] + W[29] + (rrot (W[21], 7) ^ rrot (W[21], 18) ^ (W[21] >> 3)) + (rrot (W[34], 17) ^ rrot (W[34], 19) ^ (W[34] >> 10)),
    W[21] + W[30] + (rrot (W[22], 7) ^ rrot (W[22], 18) ^ (W[22] >> 3)) + (rrot (W[35], 17) ^ rrot (W[35], 19) ^ (W[35] >> 10)),
    W[22] + W[31] + (rrot (W[23], 7) ^ rrot (W[23], 18) ^ (W[23] >> 3)) + (rrot (W[36], 17) ^ rrot (W[36], 19) ^ (W[36] >> 10)),
    W[23] + W[32] + (rrot (W[24], 7) ^ rrot (W[24], 18) ^ (W[24] >> 3)) + (rrot (W[37], 17) ^ rrot (W[37], 19) ^ (W[37] >> 10)),
    W[24] + W[33] + (rrot (W[25], 7) ^ rrot (W[25], 18) ^ (W[25] >> 3)) + (rrot (W[38], 17) ^ rrot (W[38], 19) ^ (W[38] >> 10)),
    W[25] + W[34] + (rrot (W[26], 7) ^ rrot (W[26], 18) ^ (W[26] >> 3)) + (rrot (W[39], 17) ^ rrot (W[39], 19) ^ (W[39] >> 10)),
    W[26] + W[35] + (rrot (W[27], 7) ^ rrot (W[27], 18) ^ (W[27] >> 3)) + (rrot (W[40], 17) ^ rrot (W[40], 19) ^ (W[40] >> 10)),
    W[27] + W[36] + (rrot (W[28], 7) ^ rrot (W[28], 18) ^ (W[28] >> 3)) + (rrot (W[41], 17) ^ rrot (W[41], 19) ^ (W[41] >> 10)),
    W[28] + W[37] + (rrot (W[29], 7) ^ rrot (W[29], 18) ^ (W[29] >> 3)) + (rrot (W[42], 17) ^ rrot (W[42], 19) ^ (W[42] >> 10)),
    W[29] + W[38] + (rrot (W[30], 7) ^ rrot (W[30], 18) ^ (W[30] >> 3)) + (rrot (W[43], 17) ^ rrot (W[43], 19) ^ (W[43] >> 10)),
    W[30] + W[39] + (rrot (W[31], 7) ^ rrot (W[31], 18) ^ (W[31] >> 3)) + (rrot (W[44], 17) ^ rrot (W[44], 19) ^ (W[44] >> 10)),
    W[31] + W[40] + (rrot (W[32], 7) ^ rrot (W[32], 18) ^ (W[32] >> 3)) + (rrot (W[45], 17) ^ rrot (W[45], 19) ^ (W[45] >> 10)),
    W[32] + W[41] + (rrot (W[33], 7) ^ rrot (W[33], 18) ^ (W[33] >> 3)) + (rrot (W[46], 17) ^ rrot (W[46], 19) ^ (W[46] >> 10)),
    W[33] + W[42] + (rrot (W[34], 7) ^ rrot (W[34], 18) ^ (W[34] >> 3)) + (rrot (W[47], 17) ^ rrot (W[47], 19) ^ (W[47] >> 10)),
    W[34] + W[43] + (rrot (W[35], 7) ^ rrot (W[35], 18) ^ (W[35] >> 3)) + (rrot (W[48], 17) ^ rrot (W[48], 19) ^ (W[48] >> 10)),
    W[35] + W[44] + (rrot (W[36], 7) ^ rrot (W[36], 18) ^ (W[36] >> 3)) + (rrot (W[49], 17) ^ rrot (W[49], 19) ^ (W[49] >> 10)),
    W[36] + W[45] + (rrot (W[37], 7) ^ rrot (W[37], 18) ^ (W[37] >> 3)) + (rrot (W[50], 17) ^ rrot (W[50], 19) ^ (W[50] >> 10)),
    W[37] + W[46] + (rrot (W[38], 7) ^ rrot (W[38], 18) ^ (W[38] >> 3)) + (rrot (W[51], 17) ^ rrot (W[51], 19) ^ (W[51] >> 10)),
    W[38] + W[47] + (rrot (W[39], 7) ^ rrot (W[39], 18) ^ (W[39] >> 3)) + (rrot (W[52], 17) ^ rrot (W[52], 19) ^ (W[52] >> 10)),
    W[39] + W[48] + (rrot (W[40], 7) ^ rrot (W[40], 18) ^ (W[40] >> 3)) + (rrot (W[53], 17) ^ rrot (W[53], 19) ^ (W[53] >> 10)),
    W[40] + W[49] + (rrot (W[41], 7) ^ rrot (W[41], 18) ^ (W[41] >> 3)) + (rrot (W[54], 17) ^ rrot (W[54], 19) ^ (W[54] >> 10)),
    W[41] + W[50] + (rrot (W[42], 7) ^ rrot (W[42], 18) ^ (W[42] >> 3)) + (rrot (W[55], 17) ^ rrot (W[55], 19) ^ (W[55] >> 10)),
    W[42] + W[51] + (rrot (W[43], 7) ^ rrot (W[43], 18) ^ (W[43] >> 3)) + (rrot (W[56], 17) ^ rrot (W[56], 19) ^ (W[56] >> 10)),
    W[43] + W[52] + (rrot (W[44], 7) ^ rrot (W[44], 18) ^ (W[44] >> 3)) + (rrot (W[57], 17) ^ rrot (W[57], 19) ^ (W[57] >> 10)),
    W[44] + W[53] + (rrot (W[45], 7) ^ rrot (W[45], 18) ^ (W[45] >> 3)) + (rrot (W[58], 17) ^ rrot (W[58], 19) ^ (W[58] >> 10)),
    W[45] + W[54] + (rrot (W[46], 7) ^ rrot (W[46], 18) ^ (W[46] >> 3)) + (rrot (W[59], 17) ^ rrot (W[59], 19) ^ (W[59] >> 10)),
    W[46] + W[55] + (rrot (W[47], 7) ^ rrot (W[47], 18) ^ (W[47] >> 3)) + (rrot (W[60], 17) ^ rrot (W[60], 19) ^ (W[60] >> 10)),
    W[47] + W[56] + (rrot (W[48], 7) ^ rrot (W[48], 18) ^ (W[48] >> 3)) + (rrot (W[61], 17) ^ rrot (W[61], 19) ^ (W[61] >> 10)),
  };

  // unrolled loop 0..63
  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x428a2f98 + W[0];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x71374491 + W[1];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xb5c0fbcf + W[2];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xe9b5dba5 + W[3];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x3956c25b + W[4];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x59f111f1 + W[5];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x923f82a4 + W[6];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xab1c5ed5 + W[7];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xd807aa98 + W[8];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x12835b01 + W[9];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x243185be + W[10];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x550c7dc3 + W[11];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x72be5d74 + W[12];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x80deb1fe + W[13];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x9bdc06a7 + W[14];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xc19bf174 + W[15];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xe49b69c1 + W[16];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xefbe4786 + W[17];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x0fc19dc6 + W[18];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x240ca1cc + W[19];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x2de92c6f + W[20];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x4a7484aa + W[21];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x5cb0a9dc + W[22];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x76f988da + W[23];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x983e5152 + W[24];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xa831c66d + W[25];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xb00327c8 + W[26];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xbf597fc7 + W[27];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xc6e00bf3 + W[28];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xd5a79147 + W[29];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x06ca6351 + W[30];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x14292967 + W[31];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x27b70a85 + W[32];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x2e1b2138 + W[33];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x4d2c6dfc + W[34];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x53380d13 + W[35];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x650a7354 + W[36];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x766a0abb + W[37];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x81c2c92e + W[38];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x92722c85 + W[39];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xa2bfe8a1 + W[40];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xa81a664b + W[41];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xc24b8b70 + W[42];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xc76c51a3 + W[43];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xd192e819 + W[44];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xd6990624 + W[45];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xf40e3585 + W[46];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x106aa070 + W[47];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x19a4c116 + W[48];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x1e376c08 + W[49];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x2748774c + W[50];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x34b0bcb5 + W[51];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x391c0cb3 + W[52];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x4ed8aa4a + W[53];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x5b9cca4f + W[54];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x682e6ff3 + W[55];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x748f82ee + W[56];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x78a5636f + W[57];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x84c87814 + W[58];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x8cc70208 + W[59];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0x90befffa + W[60];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xa4506ceb + W[61];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xbef9a3f7 + W[62];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  temp1 = h + (rrot (e, 6) ^ rrot (e, 11) ^ rrot (e, 25)) + ((e & f) ^ ((~e) & g)) + 0xc67178f2 + W[63];
  temp2 = (rrot (a, 2) ^ rrot (a, 13) ^ rrot (a, 22)) + ((a & b) ^ (a & c) ^ (b & c));
  h = g;
  g = f;
  f = e;
  e = d + temp1;
  d = c;
  c = b;
  b = a;
  a = temp1 + temp2;

  // update keys
  a_ += a;
  b_ += b;
  c_ += c;
  d_ += d;
  e_ += e;
  f_ += f;
  g_ += g;
  h_ += h;

  // update size
  size_ += data.size ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate final hash value
//! \param data remaining data
//! \return digest
// The final block is padded with a 1 bit, \0 bytes and a 64-bit value
// representing the message size in bits (FIPS.180-4 - section 5.1.1)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const mobius::bytearray
hash_sha2_224::_evaluate (const mobius::bytearray& data)
{
  // represent message size in bits as a bytearray
  std::uint64_t bits = ((size_ + data.size ()) << 3);

  mobius::bytearray dsize =
  {
    std::uint8_t (bits >> 56),
    std::uint8_t (bits >> 48),
    std::uint8_t (bits >> 40),
    std::uint8_t (bits >> 32),
    std::uint8_t (bits >> 24),
    std::uint8_t (bits >> 16),
    std::uint8_t (bits >> 8),
    std::uint8_t (bits)
  };

  // if the length of the last block size >= 56, generate two blocks:
  // one: data + '1' bit + '0' bits (padding)
  // two: 56 bytes (0) + message size
  if (data.size () >= 56)
    {
      mobius::bytearray padding (64 - data.size ());
      padding.fill (0);
      padding[0] = 0x80;                // 1st bit = "1"
      _update_block (data + padding);

      mobius::bytearray padding2 (56);
      padding2.fill (0);
      _update_block (padding2 + dsize);
    }

  // otherwise, generate just one block:
  // data + '1' bit + '0' bits (padding) + message size
  else
    {
      mobius::bytearray padding (56 - data.size ());
      padding.fill (0);
      padding[0] = 0x80;                // 1st bit = "1"
      _update_block (data + padding + dsize);
    }

  // build digest
  return mobius::bytearray
  {
    std::uint8_t (a_ >> 24),
    std::uint8_t (a_ >> 16),
    std::uint8_t (a_ >> 8),
    std::uint8_t (a_),
    std::uint8_t (b_ >> 24),
    std::uint8_t (b_ >> 16),
    std::uint8_t (b_ >> 8),
    std::uint8_t (b_),
    std::uint8_t (c_ >> 24),
    std::uint8_t (c_ >> 16),
    std::uint8_t (c_ >> 8),
    std::uint8_t (c_),
    std::uint8_t (d_ >> 24),
    std::uint8_t (d_ >> 16),
    std::uint8_t (d_ >> 8),
    std::uint8_t (d_),
    std::uint8_t (e_ >> 24),
    std::uint8_t (e_ >> 16),
    std::uint8_t (e_ >> 8),
    std::uint8_t (e_),
    std::uint8_t (f_ >> 24),
    std::uint8_t (f_ >> 16),
    std::uint8_t (f_ >> 8),
    std::uint8_t (f_),
    std::uint8_t (g_ >> 24),
    std::uint8_t (g_ >> 16),
    std::uint8_t (g_ >> 8),
    std::uint8_t (g_)
  };
}

} // crypt namespace
} // mobius namespace
