# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import os.path

import mobius
import pymobius
from gi.repository import Gtk

from metadata import *


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief View: Preferences
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class PreferencesView(object):

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Initialize widget
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __init__(self, control):
        self.__mediator = pymobius.mediator.copy()
        self.__control = control
        self.__iped_path = None
        self.name = 'Preferences'

        path = self.__mediator.call('extension.get-resource-path', EXTENSION_ID, 'preferences.png')
        self.icon_data = open(path, 'rb').read()

        # build widget
        self.__widget = mobius.ui.container()
        self.__widget.show()

        vbox = Gtk.VBox()
        vbox.set_border_width(10)
        vbox.set_spacing(5)
        vbox.show()
        self.__widget.set_content(vbox)

        grid = Gtk.Grid.new()
        grid.set_row_spacing(10)
        grid.set_column_spacing(5)
        grid.show()
        vbox.pack_start(grid, True, True, 0)

        label = Gtk.Label.new()
        label.set_markup('<b>IPED installation folder</b>')
        label.set_xalign(1.0)
        label.show()
        grid.attach(label, 0, 0, 1, 1)

        self.__iped_folder_button = Gtk.FileChooserButton(title="Select IPED installation folder")
        self.__iped_folder_button.set_hexpand(True)
        self.__iped_folder_button.set_action(Gtk.FileChooserAction.SELECT_FOLDER)
        self.__iped_folder_button.set_current_folder(self.__iped_path or '/opt')
        self.__iped_folder_button.connect('file-set', self.__file_set)
        self.__iped_folder_button.show()
        grid.attach(self.__iped_folder_button, 1, 0, 4, 1)

        # XMM (max memory size)
        label = Gtk.Label.new()
        label.set_markup('<b>Max. Memory (XMX, in GB):</b>')
        label.set_xalign(1.0)
        label.show()
        grid.attach(label, 0, 1, 1, 1)

        xmx = mobius.framework.get_config('iped.xmx') or 8

        self.__xmx_spinbutton = Gtk.SpinButton.new_with_range(1, 4096, 1)
        self.__xmx_spinbutton.set_value(xmx)
        self.__xmx_spinbutton.connect('value-changed', self.__on_xmx_value_changed)

        self.__xmx_spinbutton.show()

        grid.attach(self.__xmx_spinbutton, 1, 1, 1, 1)

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief get ui widget
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def get_ui_widget(self):
        return self.__widget.get_widget().get_ui_widget()

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief set data to be viewed
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def set_data(self, itemlist):
        pass

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Set IPED path
    # @param path Full path
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def set_iped_path(self, path):
        self.__iped_path = path
        self.__widget.show_content()

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Save current state
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def on_destroy(self):
        self.__mediator.clear()

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Handle 'value-changed' spinbutton event
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __on_xmx_value_changed(self, spinbutton, *args):
        transaction = mobius.framework.new_config_transaction()
        mobius.framework.set_config('iped.xmx', spinbutton.get_value_as_int())
        transaction.commit()

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Handle 'file-set' filechooserbutton event
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __file_set(self, filechooserbutton, *args):
        path = filechooserbutton.get_filename()
        jar_path = os.path.join(path, 'iped.jar')

        # check if folder contains 'iped.jar' file
        if os.path.exists(jar_path):
            transaction = mobius.framework.new_config_transaction()
            mobius.framework.set_config('iped.folder_path', path)
            transaction.commit()

            self.__control.set_iped_path(path)

        # Show error dialog
        else:
            dialog = mobius.ui.message_dialog(mobius.ui.message_dialog.type_error)
            dialog.text = 'Folder does not contain "iped.jar" file'
            dialog.add_button(mobius.ui.message_dialog.button_ok)
            dialog.run()

            self.__iped_folder_button.set_filename('/opt')
            self.__control.set_iped_path(None)
