/* This file is part of the MPFRCPP Library.

  Copyright (c) 2006 -- 2007 Alexey V. Beshenov <bav.272304@gmail.com>.

  The MPFRCPP Library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  The MPFRCPP Library is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with the MPFRCPP Library; see the file COPYING.LIB. If
  not, write to the Free Software Foundation, Inc., 51 Franklin Street,
  Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file comparison.hpp
 * @date 2007-04-08
 * Comparison operator overloads for mpfrcpp::Real.
 * INTERNAL HEADER, NOT TO BE USED DIRECTLY.
 */

#ifndef INCLUDED_BY_MPFRCPP
    #error THIS IS AN INTERNAL HEADER ONLY, NOT TO BE USED DIRECTLY
#endif

#ifndef MPFRCPP_REAL_COMPARISON
#define MPFRCPP_REAL_COMPARISON

namespace mpfrcpp {

    //------------------------------------------------------------

    /**
     * Signum
     */

    int Real::sign () const throw( ComparisonWithNaNError ) {
        if ( isNaN () )
            throw ComparisonWithNaNError();

        int n = mpfr_sgn( getMpfrT () );

        if ( n < 0 )
            return -1;
        else if ( n > 0 )
            return 1;
        else //  n == 0
            return 0;
    }

    //------------------------------------------------------------

    /**
     * Comparison
     */

    bool Real::operator> ( const Real& x ) const throw( ComparisonWithNaNError ) {
        if ( isNaN () || x.isNaN() )
            throw ComparisonWithNaNError();

        return mpfr_greater_p( getMpfrT(), x.getMpfrT() ) != 0;
    }

    bool Real::operator>= ( const Real& x ) const throw( ComparisonWithNaNError ) {
        if ( isNaN () || x.isNaN() )
            throw ComparisonWithNaNError();

        return mpfr_greaterequal_p( getMpfrT(), x.getMpfrT() ) != 0;
    }

    bool Real::operator< ( const Real& x ) const throw( ComparisonWithNaNError ) {
        if ( isNaN () || x.isNaN() )
            throw ComparisonWithNaNError();

        return mpfr_less_p( getMpfrT(), x.getMpfrT() ) != 0;
    }

    bool Real::operator<= ( const Real& x ) const throw( ComparisonWithNaNError ) {
        if ( isNaN () || x.isNaN() )
            throw ComparisonWithNaNError();

        return mpfr_lessequal_p( getMpfrT(), x.getMpfrT() ) != 0;
    }

    bool Real::operator== ( const Real& x ) const throw( ComparisonWithNaNError ) {
        if ( isNaN () || x.isNaN() )
            throw ComparisonWithNaNError();

        return mpfr_equal_p( getMpfrT(), x.getMpfrT() ) != 0;
    }

    bool Real::operator!= ( const Real& x ) const throw( ComparisonWithNaNError ) {
        return !( *this == x );
    }

    //------------------------------------------------------------

}    // namespace mpfrcpp

#endif    // MPFRCPP_REAL_COMPARISON
