/* This file is part of the MPFRCPP Library.

  Copyright (c) 2006 -- 2007 Alexey V. Beshenov <bav.272304@gmail.com>.

  The MPFRCPP Library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  The MPFRCPP Library is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with the MPFRCPP Library; see the file COPYING.LIB. If
  not, write to the Free Software Foundation, Inc., 51 Franklin Street,
  Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file constructors.hpp
 * @date 2007-03-30
 * mpfrcpp::Real constructors.
 * INTERNAL HEADER, NOT TO BE USED DIRECTLY.
 */

#ifndef INCLUDED_BY_MPFRCPP
    #error THIS IS AN INTERNAL HEADER ONLY, NOT TO BE USED DIRECTLY
#endif

#ifndef MPFRCPP_REAL_CONSTRUCTORS
#define MPFRCPP_REAL_CONSTRUCTORS

namespace mpfrcpp {

    //------------------------------------------------------------

    Real::Real ( const Precision& pr ) throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        if ( getParameters().isInitializedByZero() )
            mpfr_set_ui( getMpfrT(), 0,
                         getParameters().getDefaultRoundMode().getMpfrRndT() );
        /* set number to NaN else */
    }

    Real::Real ( const Real& x ) throw() {
        mpfr_init2 ( getMpfrT(), x.getPrecision().getMpfrPrecT() );
        mpfr_set ( getMpfrT(), x.getMpfrT(),
                   getParameters().getDefaultRoundMode().getMpfrRndT() );
    }

    Real::Real ( const Real& x, const Precision& pr, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set ( getMpfrT(), x.getMpfrT(), rm.getMpfrRndT () );
    }

    Real::Real ( const Real& x, const Precision& pr ) throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set ( getMpfrT(), x.getMpfrT(),
                   getParameters().getDefaultRoundMode().getMpfrRndT() );
    }

    Real::Real ( const Real& x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(), getParameters().getDefaultPrecision() );
        mpfr_set ( getMpfrT(), x.getMpfrT(), rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const mpfr_t& x, const Precision& pr, const RoundMode& rm )
    throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const unsigned long int x, const Precision& pr,
                 const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_ui ( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const unsigned long int x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_ui ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    Real::Real ( const unsigned int x, const Precision& pr, const RoundMode& rm )
    throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_ui ( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const unsigned int x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_ui ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    Real::Real ( const unsigned short int x, const Precision& pr,
                 const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_ui( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const unsigned short int x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_ui ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const long int x, const Precision& pr, const RoundMode& rm )
    throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_si ( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const long int x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_si ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    Real::Real ( const int x, const Precision& pr, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_si ( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const int x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_si ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    Real::Real ( const short int x, const Precision& pr, const RoundMode& rm )
    throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_si ( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const short int x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_si ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const double& x, const Precision& pr, const RoundMode& rm )
    throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_d ( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const double& x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_d ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    Real::Real ( const long double& x, const Precision& pr, const RoundMode& rm )
    throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_ld ( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const long double& x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_ld ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const mpz_t& x, const Precision& pr, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_z ( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const mpz_t& x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_z ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const mpq_t& x, const Precision& pr, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_q ( getMpfrT(), x, rm.getMpfrRndT () );
    }
    Real::Real ( const mpq_t& x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_q ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const mpf_t& x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(), mpf_get_prec( x ) );
        mpfr_set_f ( getMpfrT(), x, rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const std::string& x, const Base& base, const Precision& pr,
                 const RoundMode& rm ) throw( InvalidNumberStringError ) {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        int c = mpfr_set_str( getMpfrT(), x.c_str(), base.getInt(),
                              rm.getMpfrRndT () );
        if ( c == -1 ) {
            throw InvalidNumberStringError();
        }
    }

    Real::Real ( const std::string& x, const Base& base, const RoundMode& rm )
    throw( InvalidNumberStringError ) {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        int c = mpfr_set_str ( getMpfrT(), x.c_str(), base.getInt(),
                               rm.getMpfrRndT () );
        if ( c == -1 ) {
            throw InvalidNumberStringError();
        }
    }

    Real::Real ( const std::string& x, const Precision& pr)
    throw( InvalidNumberStringError ) {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        int c = mpfr_set_str( getMpfrT(), x.c_str(),
                              getParameters().getDefaultBase().getInt(),
                              getParameters().getDefaultRoundMode().getMpfrRndT () );
        if ( c == -1 ) {
            throw InvalidNumberStringError();
        }
    }

    Real::Real ( const std::string& x, const RoundMode& rm)
    throw( InvalidNumberStringError ) {
        mpfr_init2 ( getMpfrT(), getParameters().getDefaultPrecision().getMpfrPrecT() );
        int c = mpfr_set_str( getMpfrT(), x.c_str(),
                              getParameters().getDefaultBase().getInt(),
                              rm.getMpfrRndT () );
        if ( c == -1 ) {
            throw InvalidNumberStringError();
        }
    }

    //------------------------------------------------------------

#ifdef GMP_CPP_INTERFACE
    Real::Real ( const mpz_class& x, const Precision& pr, const RoundMode& rm )
    throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_z( getMpfrT(), x.get_mpz_t(), rm.getMpfrRndT () );
    }
    Real::Real ( const mpz_class& x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_z ( getMpfrT(), x.get_mpz_t(), rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const mpq_class& x, const Precision& pr, const RoundMode& rm )
    throw() {
        mpfr_init2 ( getMpfrT(), pr.getMpfrPrecT() );
        mpfr_set_q( getMpfrT(), x.get_mpq_t(), rm.getMpfrRndT () );
    }
    Real::Real ( const mpq_class& x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(),
                     getParameters().getDefaultPrecision().getMpfrPrecT() );
        mpfr_set_q ( getMpfrT(), x.get_mpq_t(), rm.getMpfrRndT () );
    }

    //------------------------------------------------------------

    Real::Real ( const mpf_class& x, const RoundMode& rm ) throw() {
        mpfr_init2 ( getMpfrT(), x.get_prec() );
        mpfr_set_f ( getMpfrT(), x.get_mpf_t(), rm.getMpfrRndT () );
    }
#endif    // GMP_CPP_INTERFACE

    //------------------------------------------------------------

    /**
     * Destructor
     */

    Real::~Real() throw() {
        mpfr_clear( getMpfrT() );
    }

    //------------------------------------------------------------

}    // namespace mpfrcpp

#endif    // MPFRCPP_REAL_CONSTRUCTORS
