/************************************************************************
 *                                                                      *
 * Open File Manager - ncurses file manager for GNU/Linux               *
 * (c) 2001, 2002 Slawomir Strumecki, Raphael Bugajewski                *
 *                                                                      *
 * This program is free software, you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to the Free Software          *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 * If you want to contact us, please use the following address(es):     *
 *                                                                      *
 *     Raphael Bugajewski               Slawomir Strumecki              *
 *     Kl. Mittelstr. 1                 mailto: <logospam@poczta.fm>    *
 *     13585 Berlin / Germany                                           *
 *     Tel.: +49 (175) 331 93 92                                        *
 *     mailto: <born@bugajewski.de>                                     *
 *                                                                      *
 ************************************************************************/
#include "iface.h"

#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>

#include "misc.h"
#include "stack.h"
#include "menu.h"
#include "subshell.h"

/* constans */

const char *xb[]={"","kB","MB","GB","TB"};
const char *button[]={"CANCEL","NO","YES","OK"};

/* we could read the strings from an external file */

const char *strings[]={
"Error while reading dir.",\
"Not enough memory.",\
"Unable to delete",\
"Delete"};

char *menubar_names[] = 
{
    "File", "Utilities", (char *)NULL
};

char *men1_names[] =
{
    "View", "Copy", "MkDir", "Delete", "Rename", "Exit", (char *)NULL
};
char *men1_desc[] =
{
    "F3","F5","F7","F8","C-n" ,"C-x" , (char *)NULL
};
char *men2_names[] =
{
    "Clock",  (char *)NULL
};
char *men2_desc[] = 
{
    "F2",  (char *)NULL
};

//SCREEN *ofm_term;

/* console size */

int scrn_width,scrn_height;

/* which panel is active? */

int opanelActive;

/* panels */

OPANEL* opanels[2];

/* windows with information */

WINDOW *up_info, *down_info;
OMENU *ofm_menu[2];
OSTACK * stack[2];
char workdir[2][PATH_MAX];

/*input buffer vars*/
char cmd_buffer[PATH_MAX+1];
int cmd_size=0;

/*bookmarks stuff*/
int bookmark_is_active=0;
/*pointers replaced by bookmarks*/
DirStruct *bb_dir, *bb_sel_ds, *bb_first ;
char *bb_sel_name;
/*bookmarks pointers*/
DirStruct *b_sel_ds=NULL, *b_first=NULL ;
char *b_sel_name=NULL;
DirStruct *Bookmarks=NULL;


/* ***** ***** ***** */

DirStruct *bookmarkDelete(DirStruct *bm){
    DirStruct *tmp;
    DirStruct *new_sel;

    new_sel=NULL;
    if (bm!=NULL) {
	if (bm->prev==NULL && bm->next==NULL) {
	    Bookmarks=NULL;
	    b_sel_ds=NULL;
	    b_first=NULL;
	    b_sel_name=NULL;
	}
	else {
	    if (bm->next!=NULL) bm->next->prev=bm->prev;
	    else Bookmarks->last=bm->prev;
	    if (bm->prev!=NULL) {
		bm->prev->next=bm->next;
		new_sel=bm->prev;
	    } else {
		new_sel=bm->next;
		Bookmarks=bm->next;
		Bookmarks->index=1;
		Bookmarks->first=Bookmarks;
	    }
	    bm->next=bm->prev=NULL;
	    
	    tmp=Bookmarks;
	    if (bm->dir) tmp->dir_count--;
	    else tmp->files_count--;
	    
	    while (tmp->next!=NULL) {
		tmp->next->dir_count=tmp->dir_count;
		tmp->next->files_count=tmp->files_count;
		tmp->next->index=tmp->index+1;
		tmp->next->first=tmp->first;
		tmp->next->last=tmp->last;
		tmp=tmp->next;
	    }
	    
	}
	dirFree(bm);
    }

    return new_sel;
}

DirStruct *bookmarkInsert(DirStruct *bm){
    DirStruct *tmp;
    int ti;
    
    tmp=NULL;
    if (bm!=NULL) {
	tmp=dirDuplicate(bm);
	if (tmp!=NULL) {
	    tmp->path=NULL;
	    ti=strlen(workdir[opanelActive]);
	    tmp->path=(char *)xmalloc(ti+1);
	    if (tmp->path!=NULL) strcpy(tmp->path,workdir[opanelActive]);
	    if (Bookmarks==NULL) {
		tmp->last=tmp;
		tmp->first=tmp;
		tmp->index=1;
		Bookmarks=tmp;
	    } else {
		tmp->prev=Bookmarks->last;
		tmp->index=Bookmarks->last->index+1;
		tmp->last=tmp;
		tmp->first=Bookmarks->first;
		Bookmarks->last->next=tmp;
		Bookmarks->last=tmp;
	    }
	    if (tmp->dir) Bookmarks->dir_count++;
	    else Bookmarks->files_count++;
	    tmp=Bookmarks;
	    while (tmp->next!=NULL) {
		tmp->next->dir_count=tmp->dir_count;
		tmp->next->files_count=tmp->files_count;
		tmp=tmp->next;
	    }
	    tmp=Bookmarks->last;
	}
	
    }
    return tmp;
}

int bookmarkSaveState(OPANEL *op)
{
    bb_dir=op->dir;
    bb_sel_ds=op->sel_ds;
    bb_sel_name=op->sel_name;
    bb_first=op->first;
    op->dir=Bookmarks;
    op->sel_ds=b_sel_ds;
    op->sel_name=b_sel_name;
    op->first=b_first;
    op->virtual_dir=1;
    return 0;
}

int bookmarkRestoreState(OPANEL *op)
{
    bookmark_is_active=0;
    b_sel_ds=op->sel_ds;
    b_sel_name=op->sel_name;
    b_first=op->first;
    op->dir=bb_dir;
    op->sel_ds=bb_sel_ds;
    op->sel_name=bb_sel_name;
    op->first=bb_first;
    op->virtual_dir=0;
    return 0;
}


/* creates two panels, windows with information, gets the path of the inactive
   panel (the path of the active panel is read from opanelRefresh */

int viewportCreate(){	
    int pwidth,pheight;
    
    pheight=LINES-4;
    pwidth=COLS/2-1;

	/* you need to divide it by six,
	   because there could be two or three columns */
	
    while (((pwidth)%6)!=0) pwidth--;
    pwidth++;		
    opanels[0]=opanelCreate(0,1,pwidth,pheight);
    opanels[1]=opanelCreate(pwidth,1,pwidth,pheight);
    if ((opanels[0]==NULL)||(opanels[1]==NULL)) return -1;
    opanelActive=0;
    opanels[0]->active=1;
    opanels[1]->active=0;
    up_info=newwin(1,COLS,0,0);
    down_info=newwin(3,COLS,LINES-3,0);
    getcwd(workdir[opanelActive],PATH_MAX);
    getcwd(workdir[! opanelActive],PATH_MAX);
    stack[0]=stackInit(50);
    stack[1]=stackInit(50);
    ofm_menu[0]=menuCreate(0,1,men1_names,men1_desc);
    ofm_menu[1]=menuCreate(12,1,men2_names,men2_desc);
    menuBarCreate(menubar_names);
    if (ofm_menu[0]==NULL || ofm_menu[1]==NULL) {
	fprintf(stderr,"Could't initialize menu (it will be unaccessable) !\n");
	sleep(1000);
    }

    return 0;
}

void viewportRefresh(){
    refresh();
    opanelRefresh(opanels[0]);
    opanelRefresh(opanels[1]);
    chdir(workdir[opanelActive]);
}

void viewportDelete()
{
    opanelDelete(opanels[0]);
    opanelDelete(opanels[1]);
    if (bookmark_is_active) Bookmarks=bb_dir;  //opanelDelete cleared bookmarks' dir so we clear real dir
    delwin(up_info);
    delwin(down_info);
    menuDelete(ofm_menu[0]);
    menuDelete(ofm_menu[1]);
    menuBarDelete();
    stackFree(stack[0]);
    stackFree(stack[1]);
    if (Bookmarks!=NULL) dirFree(Bookmarks);    
    clear();
}

/* ***** ***** ***** */

OPANEL *opanelCreate(int x,int y,int width,int height)
{
    OPANEL *tmp;
    
    tmp=(OPANEL*)xmalloc(sizeof(OPANEL));
    if (tmp!=NULL) {
	//tmp->path="./";
	tmp->dir=(DirStruct*)NULL;
	tmp->virtual_dir=0;
	tmp->columns=2;
	tmp->width=width;
	tmp->height=height;
	tmp->posx=x;
	tmp->posy=y;  
	tmp->sel_ds=(DirStruct*)NULL;
	tmp->sel_name=NULL;
	tmp->win=newwin(height,width,y,x);
	tmp->reread=0;
	tmp->sort_mode=SORT_DIR_FIRST;
    }
    return tmp;
}

void opanelDelete(OPANEL *p)
{
    if (p!=NULL) {
	if (p->sel_name!=NULL) xfree(p->sel_name);
	delwin(p->win);
	dirFree(p->dir);
	xfree(p);
    }
}

/* clears the panel, draws the frame,
   choses the right colors for the panel,
   if we didn't read the directory yet, it'll be read,
   if reread == 1, then the directory is read again,
   the inode names are written */

int opanelRefresh(OPANEL *p)
{

	/* artificial variable */
    int tmp;

	/* char before the name */
    char prefix;

	/* string dependent on the column length */
    char *tstr;

	/* artificial string */
    char tname[FILENAME_MAX];

	/* current cursor position */
    int x=1,y=1;

	/* current structure */
    DirStruct *ds, *dstmp;

	/* column width */
    int colw;

	/* color */
    int col;

	/* number of current panel */
    int opanel_nr;

	/* artificial variable */
    unsigned long tsize;

	/* artificial variable */
    int t;

    if (p==NULL) return 0;
    
    /* draws windows and seperators */

	/* color */
    col=5;

	/* if the panel is active... */
    if (p->active) col=1;
    wbkgd(p->win,COLOR_PAIR(col));
    wattrset(p->win,COLOR_PAIR(col));	
    werase(p->win);
    wborder(p->win,0,0,0,0,0,0,0,0);
    if (p->columns>1) {
	tmp=p->width/p->columns;
	wmove(p->win,1,tmp);
	wvline(p->win,0,p->height-2);
	if (p->columns>2) {
	    wmove(p->win,1,2*tmp);
	    wvline(p->win,0,p->height-2); 
	}
    }

    if (p==opanels[opanelActive]) opanel_nr=opanelActive;
    else opanel_nr=! opanelActive;
    tmp=chdir(workdir[opanel_nr]);
    if (tmp) {
//	fprintf(stderr,"Error changing to workdir[%d]: %s\n",opanel_nr,workdir[opanel_nr]);
	tmp=chdir("/");
	p->reread=1;
    }
/*    else {
	fprintf(stderr,"PANELREFRESH: Refreshing panel %d\n",opanel_nr);
	fprintf(stderr,"\t\tworkdir[0]=%s\n",workdir[0]);
	fprintf(stderr,"\t\tworkdir[1]=%s\n",workdir[1]);
    }*/

	/* if the directory isn't read yet, then we should do it */

    if (p->dir==NULL) {
	//while ( ((p->dir=dirSuck("./"))==NULL) && (! chdir("..")) );
	p->dir=dirSuck("./");
	if (p->dir==NULL) return 1;
	p->dir->next=dirSort(p->dir->next,p->sort_mode,1);
	p->first=p->dir->first;
	if (p->sel_name!=NULL) p->reread=1;
	getcwd(workdir[opanel_nr],PATH_MAX);
    }

	/* repeated read in of the directory */

    if (p->reread) {
	p->reread=0;
	ds=NULL;
	if (p->virtual_dir) ds=p->dir;
	else ds=dirSuck("./");
	if (ds!=NULL){
	    if (p->virtual_dir) {
		ds=dirSort(ds,p->sort_mode,0);
		p->first=ds;
		p->dir=ds;
		Bookmarks=ds;
	    }
	    else ds->next=dirSort(ds->next,p->sort_mode,1);
	    if (p->sel_ds!=NULL) strcpy(tname,p->sel_ds->name);
	    else if (p->sel_name!=NULL) strcpy(tname,p->sel_name);

		/* we need to check if an active inode exists and we need to setup the
		   first visible inode in the panel */

	    if ((dstmp=dirFind(ds,tname))==NULL){
		p->first=ds->first;
		p->sel_ds=ds->first;
	    }
	    else {   //we found selected node
		p->sel_ds=dstmp;
		if ((dstmp=dirFind(ds,p->first->name))==NULL)  //if our previous first node disappear
		    p->first=p->sel_ds;
		else 
		    if (dstmp->index>p->sel_ds->index) //if previous node is  after selected node
			p->first=p->sel_ds;
		    else {
			tmp=p->sel_ds->index - dstmp->index - p->columns*(p->height - 2) + 1;
			if (tmp>0) while ((dstmp->next!=NULL) && tmp--) dstmp=dstmp->next;
			p->first=dstmp;
		    }
	    }
	    
	    p->sel_size=0;
	    p->sel_count=0;
	    if (! p->virtual_dir) {
		dirFree(p->dir);
		p->dir=ds;
		getcwd(workdir[opanel_nr],PATH_MAX);
	    } else {
		ds=p->dir;
		while (ds!=NULL) {
		    if (ds->selected) {
			p->sel_size+=ds->size;
			p->sel_count++;
		    }
		    ds=ds->next;
		}
	    }
	} else return 1;
    }
    
	/* the first inode shown in the panel */
	
    if (p->first==NULL) p->first=p->dir->first;

	/* column width */
	
    colw=p->width/p->columns;
    if (p->columns==1) colw--;
	
	/* variable for the inode name */
	
    tstr=(char *)xmalloc(colw+1);
    if (tstr==NULL) return 2;
    wmove(p->win,y,x);
    ds=p->first;
    //selcount=selsize=0;

    tmp=0;

	/* in this loop there are entered the names of the inodes */ 
	
    while( (tmp<=(p->columns*(p->height-2)-1))&&(ds!=NULL) ) {
	tmp++;

	/* if we read the directory for the first time */
	
	if (p->sel_name==NULL) p->sel_name=strdup(ds->name);
	if ( (p->sel_ds==NULL)&&(strcmp(p->sel_name,ds->name)==0) ) p->sel_ds=ds;

	/* determine the prefix */

	if (ds->dir) {
	    wattrset(p->win,COLOR_PAIR(col)|A_BOLD);
	    if (ds->link) prefix='~';
	    else prefix='/';     
	}
	else {
	    wattrset(p->win,COLOR_PAIR(col));
	    prefix=' ';
	    if (ds->exe) {
		wattrset(p->win,COLOR_PAIR(col+3));
		prefix='*'; 
	    } 
	    if ((ds->mode&S_IFCHR)==S_IFCHR) {
		prefix='-';
		wattrset(p->win,COLOR_PAIR(11+(col+1)/2)|A_BOLD); 
	    }
	    if ((ds->mode&S_IFIFO)==S_IFIFO) prefix='|';
	    if ((ds->mode&S_IFBLK)==S_IFBLK) prefix='+';
	    if (ds->link) {
		wattrset(p->win,COLOR_PAIR(col));
		prefix='@'; 
	    }
	}

	/* highlight the chosen file */

	if (ds->corrupt) {
	    prefix='!';
	    wattrset(p->win,COLOR_PAIR(col+2)); 
	}
	if (ds->selected) wattrset(p->win,COLOR_PAIR(col+8)|A_BOLD);
	if (p->sel_ds==ds) wattrset(p->win,COLOR_PAIR(col+1));
	if ((p->sel_ds==ds) && (ds->selected)) wattrset(p->win,COLOR_PAIR(15));
	
	
	/* write out the prefix and the filename */

	t=colw-2;   //maximum name length
	if ((tsize=strlen(ds->name))<=t) sprintf(tstr,"%-*.*s",t,t,ds->name); 
	else {
	    t/=2;
	    sprintf(tstr,"%.*s~%s",t,ds->name,ds->name+tsize-(colw-t-3)); 
	}
	mvwprintw(p->win,y,x,"%c%.*s",prefix,colw-2,tstr);

	y++;	//next row
	if (y>p->height-2) {  	//if need to jump to next column
	    y=1;
	    x+=colw;
	    wmove(p->win,y,x); 
	}
	ds=ds->next; 		//next node
    }  
    ds=p->first->first;		//get first node
    xfree(tstr);

	/* arrow */

    wattrset(p->win,COLOR_PAIR(col));
    t=ds->dir_count+ds->files_count;
    if ((p->first->index-1+p->columns*(p->height-2))<t) 
	mvwprintw(p->win,p->height-1,p->width-2,"=>");

    wrefresh(p->win);
    if (p->active) writeinfo(1);
    return 0;
}

/* ***** ***** ***** */

/* initializing the  screen */

int cursesInit() 
{
//	ofm_term=newterm(NULL,stdout,stdin);
//	if (ofm_term==NULL) printf("Couldn't initialize!");
//	set_term(ofm_term);    

	initscr();
	start_color();
	
	/* active!? - FIXME */

	/* normal */
	
	init_pair(1,COLOR_WHITE,COLOR_BLUE);
	
	/* highlighted */
	
	init_pair(2,COLOR_BLACK,COLOR_WHITE);
	
	/* damaged */		
	
	init_pair(3,COLOR_RED,COLOR_BLUE);

	/* executable */
	
	init_pair(4,COLOR_CYAN,COLOR_BLUE);

	/* non-active */

	/* normal */

	init_pair(5,COLOR_WHITE,COLOR_BLACK);
	
	/* highlighted */
	
	init_pair(6,COLOR_BLACK,COLOR_YELLOW);
	
	/* damaged */
	
	init_pair(7,COLOR_RED,COLOR_BLACK);
	
	/* executable */
	
	init_pair(8,COLOR_CYAN,COLOR_BLACK);

	/* active */

	init_pair(9,COLOR_YELLOW,COLOR_BLUE);
	
	/* colors */
	
	init_pair(10,COLOR_WHITE,COLOR_RED);

	/* for the messagebox */
	
	init_pair(11,COLOR_RED,COLOR_WHITE);

	/* char/block file (active) */
	
	init_pair(12,COLOR_MAGENTA,COLOR_BLUE);

	/* highlighted (non-active) */
	
	init_pair(13,COLOR_YELLOW,COLOR_BLACK);

	/* char/block file (non-active) */
	
	init_pair(14,COLOR_MAGENTA,COLOR_BLACK);
//	init_pair(15,COLOR_BLACK,COLOR_BLUE);
	init_pair(15,COLOR_YELLOW,COLOR_WHITE);
	
	
	noecho();cbreak();keypad(stdscr,TRUE);curs_set(0);
	scrn_width=COLS;
	scrn_height=LINES;
	memset(cmd_buffer,0,sizeof(cmd_buffer));
	return 0;
}

void cursesDestroy()
{
	clear();
	refresh();
	endwin(); 
//	delscreen(ofm_term);   
}

/* ***** ***** ***** */

int writeinfo(int mode)  {
    int t;
    unsigned int tsize;
    OPANEL *tmp;
    char msg[PATH_MAX];

    tmp=opanels[opanelActive];
	
    /* upper info */

    if (mode<2) {
	wattrset(up_info,COLOR_PAIR(0));
	wmove(up_info,0,0);wclrtoeol(up_info);
	wprintw(up_info,"DIRS: %d FILES: %d",tmp->sel_ds->first->dir_count,tmp->sel_ds->first->files_count);

	wrefresh(up_info); 
    }
	
    /* lower info */

    if (mode>0) {
	/* write out the current path */
	wattrset(down_info,COLOR_PAIR(0)|A_BOLD);
	wmove(down_info,1,1);wclrtoeol(down_info);
	if (opanelActive==bookmark_is_active-1) 
	    wprintw(down_info,"Bookmarks' directory (%s)",tmp->sel_ds->path);
	else {
	    wprintw(down_info,"%s",workdir[opanelActive]);
	    t=strlen(workdir[opanelActive]);
	    if (t!=1) {
		wprintw(down_info,"/");
		t++;
	    }
	    wattron(down_info,A_REVERSE);
	    if (cmd_size>0) {
		t+=cmd_size+2;
		if (t<COLS) 
		    wprintw(down_info,"%s",cmd_buffer);
		else
		    wprintw(down_info,"%s",cmd_buffer+(t-COLS));
	    }
	    wattroff(down_info,A_REVERSE);
	}

	wattrset(down_info,COLOR_PAIR(0));
	wmove(down_info,0,1);wclrtoeol(down_info);
	t=0;tsize=tmp->sel_size;
	while(tsize>1023) {
		tsize>>=10;
		t++; 
	}
	if (tmp->sel_count>0) wprintw(down_info,"%d%s in %d directory node%c",tsize,xb[t],tmp->sel_count,(tmp->sel_count>1) ? 's':' ');
	else if ((! tmp->sel_ds->corrupt)&&(! tmp->sel_ds->link)){
	    t=tmp->sel_ds->mode;
	    wprintw(down_info,"%c%c%c%c%c%c%c%c%c%c (%.6o)\t%s",\
		((t&S_IFBLK)==S_IFBLK) ? 'b': (((t&S_IFCHR)==S_IFCHR) ? 'c': (((t&S_IFDIR)==S_IFDIR) ? 'd': (((t&S_IFIFO)==S_IFIFO) ? 'p':'-'))),\
		((t&S_IRUSR)==S_IRUSR) ? 'r':'-',\
		((t&S_IWUSR)==S_IWUSR) ? 'w':'-',\
		((t&S_IXUSR)==S_IXUSR) ? 'x':'-',\
		((t&S_IRGRP)==S_IRGRP) ? 'r':'-',\
		((t&S_IWGRP)==S_IWGRP) ? 'w':'-',\
		((t&S_IXGRP)==S_IXGRP) ? 'x':'-',\
		((t&S_IROTH)==S_IROTH) ? 'r':'-',\
		((t&S_IWOTH)==S_IWOTH) ? 'w':'-',\
		((t&S_IXOTH)==S_IXOTH) ? 'x':'-',\
		tmp->sel_ds->mode&0xffff,ctime(&(tmp->sel_ds->mtime))); 
	}
		
	/* write out the filename and the filesize */

	wmove(down_info,2,1);wclrtoeol(down_info);
	if (! tmp->sel_ds->dir) attrset(COLOR_PAIR(0)); 
	if (tmp->sel_ds->exe) attrset(COLOR_PAIR(8));
	if (tmp->sel_ds->dir) attrset(COLOR_PAIR(0));
	if (tmp->sel_ds->corrupt) {
	    wattrset(down_info,COLOR_PAIR(7));
	    sprintf(msg,"This file is corrupted"); 
	}
	if (tmp->sel_ds->link) {
	    t=readlink(tmp->sel_ds->name,msg,PATH_MAX);
	    if (t>=0) {
		msg[t]='\0';
		wprintw(down_info,"%s -> %s",tmp->sel_ds->name,msg); 
	    }		
	    else wprintw(down_info,"Unable to read the symbolic link!"); 

	/* added index to the tests */
		
	} else wprintw(down_info,"%s (%u)",tmp->sel_ds->name,tmp->sel_ds->size);
    }
    wrefresh(down_info);
    return 0;
}

/* MakeDir dialog and handler */
int OMakeDir()
{
    char 	buff[PATH_MAX+1];

    memset(buff,0,sizeof(buff));
    dlgMessageBox(B_OK,buff,"MakeDir","Input Name:");
    if (strlen(buff)>0 && ! mkdir(buff,040755)){
        opanels[opanelActive]->reread=1;
        if (strcmp(workdir[0],workdir[1])==0) opanels[! opanelActive]->reread=1;
    }

    return 0;
}

/* Delete file/dir */
int ODelete()
{
    OPANEL 	*tmp=opanels[opanelActive];
    int 	result=0;
    char 	*wd;
    
    if (strcmp(tmp->sel_ds->name,"..") || tmp->sel_count>0) {
        if (dlgMessageBox(B_YES|B_NO,NULL,"Delete","Are You Sure?")==B_YES) {
	    if (tmp->sel_ds->path!=NULL) wd=tmp->sel_ds->path;
	    else wd=workdir[opanelActive];
	    dlgDelete(tmp);
	    tmp->reread=1;
	    if (strcmp(wd,workdir[! opanelActive])==0) {
	        opanels[! opanelActive]->reread=1;
	    }		    
	}
	result=1;
    }
    return result;
}

/* Copy file/dir */
int OCopy()
{
    OPANEL 	*tmp=opanels[opanelActive];
    int 	result=0;
    char 	buff[PATH_MAX+1];
    
    if (strcmp(tmp->sel_ds->name,"..") || tmp->sel_count) {
	sprintf(buff,"%s/",workdir[! opanelActive]);
	if (dlgMessageBox(B_OK|B_CANCEL,buff,"CopyTo","Destination:")==B_OK){
	    dlgCopy(tmp,buff);
	    tmp->reread=1;	//have to check it
	    opanels[! opanelActive]->reread=1;
	}
	result=1;;	
    }
    return result;
}

/* Rename */
int ORename()
{
    OPANEL	*tmp=opanels[opanelActive];
    int 	result=0;
    char 	buff[PATH_MAX+1];
    int 	itmp;
    char 	*wd;

    if (strcmp(tmp->sel_ds->name,"..")) {
	result=1;
	if (tmp->sel_ds->path!=NULL) wd=tmp->sel_ds->path;
	else wd=workdir[opanelActive];
	if (strlen(wd)==1)	//if root dir	
	    sprintf(buff,"/%s",tmp->sel_ds->name);
	else 
	    sprintf(buff,"%s/%s",wd,tmp->sel_ds->name);
	if (dlgMessageBox(B_OK|B_CANCEL,buff,"Rename","New Name:")==B_OK){
		itmp=rename(tmp->sel_ds->name,buff);
		if (itmp) dlgMessageBox(B_OK,NULL,"Error",strerror(errno));
		/* we should really check this! rename could be into another directory -
		   FIXME */
		tmp->reread=1;
		if (strcmp(wd,workdir[! opanelActive])==0) opanels[! opanelActive]->reread=1;
	}
    }
    return result;
}

/* messagebox with current time */
int OClock()
{
    time_t 	tm1;
    struct tm 	*lt;
    char 	buf[30];

    tm1=time(NULL);
    lt=localtime(&tm1);
    sprintf(buf,"%d:%02d:%02d %02d.%02d.%04d",lt->tm_hour,lt->tm_min,lt->tm_sec,lt->tm_mday,1+lt->tm_mon,1900+lt->tm_year);
    dlgMessageBox(B_OK,NULL,"Clock",buf);

    return 0;
}

/* subshell; need to be out of the curses */
int OShell()
{
    subshell_run(CTRL('o'),workdir[opanelActive],NULL);
    return 0;
}

/* ***** KEY-BINDINGS ***** */

/* reacts on a keypress */
 
int keyReact(int i)
{
    OPANEL *tmp;
    DirStruct *dstmp;
    int itmp;
    int m_sel;
    int activem=0;
    char *tmpch;
    
    tmp=opanels[opanelActive];
    
    if (i==KEY_F(9)) {     		//MENU PULL DOWN
	menuPost(ofm_menu[activem]);
	box(ofm_menu[activem]->win,0,0);
	menuBarRefresh(activem);
	m_sel=0;
	
	while((i=wgetch(ofm_menu[activem]->win))!=KEY_F(9) && i!=KEY_ESC && i!='\n') {
	    if (i==KEY_RIGHT) {
		menuUnpost(ofm_menu[activem]);
		viewportRefresh();
		activem=(activem+1)%2;		//choose next menu
		menuPost(ofm_menu[activem]);
		box(ofm_menu[activem]->win,0,0);
		menuBarRefresh(activem);
		m_sel=menuCurrentItem(ofm_menu[activem]);
	    } else if (i==KEY_LEFT) {
		menuUnpost(ofm_menu[activem]);
		viewportRefresh();
		activem=(activem+2+1)%2;	//choose previous menu
		menuPost(ofm_menu[activem]);
		box(ofm_menu[activem]->win,0,0);
		menuBarRefresh(activem);
		m_sel=menuCurrentItem(ofm_menu[activem]);
	    } else {
		m_sel=menuProcess(ofm_menu[activem],i);
		wrefresh(ofm_menu[activem]->subwin);
	    }
	}
	
	if (i==KEY_F(9) || i==KEY_ESC) m_sel=-1;
	menuUnpost(ofm_menu[activem]);
	viewportRefresh();
	if (m_sel>=0) {
	    if (activem==0) {
		if (m_sel==1) OCopy();
		if (m_sel==2) OMakeDir();
		if (m_sel==3) ODelete();
		if (m_sel==4) ORename();
		if (m_sel==5) i=CTRL('x');
	    } else if (activem==1) {
		if (m_sel==0) OClock();
	    }
	    viewportRefresh();
	}    
    }
    
    else if (i==KEY_DOWN) {
	if ( (tmp->sel_ds!=NULL)&&(tmp->sel_ds->next!=NULL) ) {
	    tmp->sel_ds=tmp->sel_ds->next;
	    itmp=tmp->first->index+tmp->columns*(tmp->height-2)-1;
	    if (tmp->sel_ds->index>itmp) tmp->first=tmp->sel_ds;
	    opanelRefresh(opanels[opanelActive]);
	}
    }
    else if (i==KEY_UP) {

	/* similar to the previous... */

	if ((tmp->sel_ds!=NULL)&&(tmp->sel_ds->prev!=NULL)) tmp->sel_ds=tmp->sel_ds->prev;
	if (tmp->sel_ds->index<tmp->first->index) 
	    if (tmp->first->index>1) {
		itmp=tmp->height-2;
		while ((tmp->first->prev!=NULL)&&(itmp--)) tmp->first=tmp->first->prev;
	    }
	opanelRefresh(opanels[opanelActive]);
    }

    else if (i=='\n') {
	if (cmd_size>0) {  //if there is some command to execute
	    erase();
	    refresh();
	    def_prog_mode();
	    endwin();
	    subshell_run(0,workdir[opanelActive],cmd_buffer);
	    reset_prog_mode();
	    cmd_size=0;
	    memset(cmd_buffer,0,sizeof(cmd_buffer));
	    opanels[0]->reread=1;
	    opanels[1]->reread=1;
	    viewportRefresh();
	    curs_set(0);
	} else
	if (tmp->sel_ds->dir){	//if we want to change dir
	    if ( strcmp(workdir[opanelActive],"/") || strcmp(tmp->sel_ds->name,"..") ){
		itmp=0;
		if (tmp->virtual_dir && tmp->sel_ds->path!=NULL) {
		    itmp=chdir(tmp->sel_ds->path); 
		}
		itmp+=chdir(tmp->sel_ds->name);
		if (itmp) {
		    dlgMessageBox(B_OK,NULL,"Error",strerror(errno));
		    opanelRefresh(opanels[! opanelActive]);
		}
		else {	//if we changed dir
		    if (tmp->sel_name!=NULL) xfree(tmp->sel_name);
		    tmp->sel_name=NULL;
		    if (strcmp(tmp->sel_ds->name,"..")) //if we're going down in dir tree
			stackPush(stack[opanelActive],tmp->sel_ds->name);
		    else tmp->sel_name=stackPop(stack[opanelActive]);

			/* resets the stack, if link */
			
		    if (tmp->sel_ds->link || tmp->virtual_dir) {
			tmpch=NULL;
			while ((tmpch=stackPop(stack[opanelActive]))!=NULL) xfree(tmpch);
		    }
		    if (tmp->virtual_dir) bookmarkRestoreState(tmp);
		    getcwd(workdir[opanelActive],PATH_MAX);
		    tmp->sel_ds=NULL;
		    tmp->first=NULL;
		    tmp->sel_count=0;
		    tmp->sel_size=0;
		    tmp->reread=0;
		    dirFree(tmp->dir);
		    tmp->dir=NULL;
		}
	    }
	    opanelRefresh(tmp);
	}
    }
    else if (i=='\t') {
	opanelActive=! opanelActive;
	opanels[0]->active=opanels[1]->active;
	opanels[1]->active=! opanels[1]->active;
	opanels[opanelActive]->reread=1;
	viewportRefresh();
    }
    else if (i==KEY_RIGHT){
	itmp=tmp->height-2;
	while((tmp->sel_ds->next!=NULL)&&(itmp--)) tmp->sel_ds=tmp->sel_ds->next;
	itmp=tmp->height-2;
	if ( tmp->sel_ds->index>(itmp*tmp->columns+tmp->first->index-1) )
	    while((tmp->first->next!=NULL)&&(itmp--)) tmp->first=tmp->first->next;
	opanelRefresh(tmp);
    }
    else if (i==KEY_LEFT){
	itmp=tmp->height-2;
	while ((tmp->sel_ds->prev!=NULL)&&(itmp--)) tmp->sel_ds=tmp->sel_ds->prev;
	if (tmp->sel_ds->index<tmp->first->index){
	    itmp=tmp->height-2;
	    while ((tmp->first->prev!=NULL)&&(itmp--)) tmp->first=tmp->first->prev;
	}	
	opanelRefresh(tmp);
    }
	
    /* home */
	
    else if (i==KEY_HOME){
	
	/* should we change the field names? */
	
	tmp->first=tmp->first->first;
	tmp->sel_ds=tmp->first;
	opanelRefresh(tmp);
    }
	
    /* end */
    
    else if(i==KEY_END){
	dstmp=tmp->first;
	itmp=dstmp->first->files_count+dstmp->first->dir_count-(tmp->columns*(tmp->height-2))+2;
	if ((itmp>0) && (dstmp->index<itmp)) {
	    itmp-=dstmp->index;

		/* here we could determine prev_first */
		
	    while((dstmp!=NULL)&&(itmp--)) dstmp=dstmp->next;
	    tmp->first=dstmp;
	}
	dstmp=tmp->first;
	while(dstmp->next!=NULL) dstmp=dstmp->next;
	tmp->sel_ds=dstmp;
	opanelRefresh(tmp);
    }
	
    /* insert */
    
	else if (i==KEY_IC) {
	if (strcmp(tmp->sel_ds->name,"..")!=0) {
	    tmp->sel_ds->selected=! tmp->sel_ds->selected;
	    if (tmp->sel_ds->selected) {
		tmp->sel_count++;
		if(! tmp->sel_ds->dir) tmp->sel_size+=tmp->sel_ds->size;
	    }
	    else {
		tmp->sel_count--;
		if(! tmp->sel_ds->dir) tmp->sel_size-=tmp->sel_ds->size;
	    }
	    if (tmp->sel_ds->next!=NULL) tmp->sel_ds=tmp->sel_ds->next;
	    if (tmp->sel_ds->index>=(tmp->first->index+(tmp->height-2)*tmp->columns)) 
		tmp->first=tmp->first->next;
	    opanelRefresh(tmp);
        }
    }
    else if (i=='*'){
	dstmp=tmp->first->first->next;
	while (dstmp!=NULL) {
	    
		/* we do not highlight the directories */
		
		if (! dstmp->dir) {	
		dstmp->selected=! dstmp->selected;
		if (dstmp->selected) {
		    tmp->sel_count++;
		    tmp->sel_size+=dstmp->size;
		}
		else {
		    tmp->sel_count--;
		    tmp->sel_size-=dstmp->size;
		}	
	    }
	    dstmp=dstmp->next;
	}
	opanelRefresh(tmp);
    }
	
    /* delete */
    
    else if(i==KEY_F(8)){
	if (ODelete()>0) viewportRefresh();
    }
	
    /* copy */
	
    else if(i==KEY_F(5)){
	if (OCopy()>0) viewportRefresh();
    }
	
    /* mkdir */
	
    else if (i==KEY_F(7)){
	OMakeDir();
        viewportRefresh();
    }
	
    /* rename */
	
    else if(i==CTRL('n')){
	if (ORename()>0) viewportRefresh();	
    }
	
    /* refresh */
    
    else if(i==CTRL('r')){
	tmp->reread=1;
	opanelRefresh(tmp);
    }
    else if ((i<'4')&&(i>'0')) {
	tmp->columns=i-'0'; 
	itmp=(tmp->sel_ds->index-tmp->first->index)/(tmp->height-2);
	itmp*=(tmp->height-2);
        while((tmp->first->next!=NULL)&&(itmp--)) tmp->first=tmp->first->next; 
	opanelRefresh(tmp);
    }
	
    /* time */
	
    else if(i==KEY_F(2)){
	OClock();
	viewportRefresh();
    }
    
    /* change of sort mode */
    else if(i==CTRL('u')){
	tmp->sort_mode=(++tmp->sort_mode)%SORT_MODES;
	tmp->reread=1;
	opanelRefresh(tmp);
    }
    
    /* switch to subshell */
    else if (i==CTRL('o')) {
	erase();
	refresh();
	def_prog_mode();
	endwin();
	OShell();
	reset_prog_mode();
	opanels[0]->reread=1;
	opanels[1]->reread=1;
	viewportRefresh();
	curs_set(0);
    }
    
    /* switch to/from bookmarks' dir displaying in active panel */
    else if (i==CTRL('b') && Bookmarks!=NULL) {
	if (bookmark_is_active-1==opanelActive) {
	    bookmarkRestoreState(tmp);
	    tmp->reread=1;
	    bookmark_is_active=0;
	} else {
	    if (bookmark_is_active) {
		bookmarkRestoreState(opanels[! opanelActive]);
		opanels[! opanelActive]->reread=1;
		opanelRefresh(opanels[! opanelActive]);
	    }
	    
	    bookmark_is_active=opanelActive+1;
	    bookmarkSaveState(tmp);
	    tmp->reread=1;
	}

	opanelRefresh(tmp);
    }
    
    /* add bookmark */
    else if (i==CTRL('a')) {
	if (opanelActive+1!=bookmark_is_active) {
	    bookmarkInsert(tmp->sel_ds);
	    if (bookmark_is_active) {
		opanels[bookmark_is_active-1]->reread=1;
		opanelRefresh(opanels[bookmark_is_active-1]);
	    }
	}
    }
    
    /* delete selected bookmark */
    else if (i==CTRL('d')) {
	if (bookmark_is_active==opanelActive+1) {
	    tmp->sel_ds=bookmarkDelete(tmp->sel_ds);
	    if (Bookmarks==NULL) {
		bookmarkRestoreState(tmp);
		tmp->reread=1;
	    } else {
		tmp->dir=Bookmarks;
		tmp->first=Bookmarks->first;
		tmp->sel_name=NULL;
	    }
	    opanelRefresh(tmp);
	}
    }
    
    /* add ascii character to command's string */
    else if (isascii(i)) {
	if (cmd_size<sizeof(cmd_buffer)) {
	    cmd_buffer[cmd_size++]=(char)i;
	    writeinfo(2); //refresh bottom info
	}    
    }
    
    /* delete last character in command's string */
    else if (i==KEY_BACKSPACE && cmd_size>0) {
	cmd_buffer[--cmd_size]='\0';
	writeinfo(2);
    }
    return i;
}

/* ***** ***** ***** */
