/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */

package opale.mathtools;
import opale.tools.*;

import java.io.*;
import java.util.*;

/**
* This class represents an array of double. Its dimension must be bracketed between 1 and 3.
* @since Opale-Mathtools 0.18
*/
public  class DArray implements Cloneable
{
private DVect tab;
private int[] dim;
private int nb_dim;


public DArray()
{
nb_dim=0;
}

public DArray(int i0)
{
nb_dim = 1;
dim = new int[nb_dim];
dim[0] = i0;
tab = new DVect(i0);
}


public DArray(int i0, int i1)
{
nb_dim = 2;
dim = new int[nb_dim];
dim[0] = i0;
dim[1] = i1;
tab = new DVect(i0*i1);
}



public DArray(int i0, int i1, int i2)
{
nb_dim = 3;
dim = new int[nb_dim];
dim[0] = i0;
dim[1] = i1;
dim[2] = i2;
tab = new DVect(i0*i1*i2);
}


public double get(int i)
{
if (Assert.enabled) Assert.check(nb_dim==1);
if (Assert.enabled) Assert.check(i>=0 && i<dim[0]);
return tab.get(i);
}


public double get(int i, int j)
{
if (Assert.enabled) Assert.check(nb_dim==2);
if (Assert.enabled) Assert.check( i>=0 && i<dim[0] && j>=0 && j<dim[1]);
return tab.get(i*dim[0]+j);
}



public double get(int i, int j, int k)
{
if (Assert.enabled) Assert.check(nb_dim==3);
if (Assert.enabled) Assert.check( i>=0 && i<dim[0] && j>=0 && j<dim[1] && k>=0 && k<dim[2]);
return tab.get(i*dim[0]*dim[1]+j*dim[1]+k);
}



public void readFile(String name, int ic) throws FileNotFoundException, IOException, InvalidFormatException
{
BufferedReader buf = new BufferedReader(new FileReader(name));
String read;
Vector doub = new Vector();
int nl = 0; // nb de lignes dans le fichier
int ndouble = 0; // nb de double

read = buf.readLine();
while (read != null) 
	{
	nl++;
	StringTokenizer st = new StringTokenizer(read);
	int compt=0; // nb de colonnes
	boolean ok=false;
	
	while (st.hasMoreTokens()) 
		{
		compt++;
		String tok = st.nextToken();
		if (compt == ic)
			{
			Double d = new Double(tok);
			doub.addElement(d);
			ndouble++;
			ok=true;
			break;
			}
 	 	}	
//	 if (!ok) throw new InvalidFormatException("The column "+ic+" doesn't exist !!!");
	 read = buf.readLine();
	}

buf.close();

if (ndouble==0) throw new InvalidFormatException("The column "+ic+" doesn't exist !!!");

nb_dim=1;
dim = new int[nb_dim];
dim[0]=ndouble;
tab = new DVect(dim[0]);
for (int i=0;i<ndouble;i++)
	{
	tab.set(i,((Double)doub.get(i)).doubleValue());
	}


}



public void readFile(String name, int ic, int jc) throws FileNotFoundException, IOException, InvalidFormatException
{
BufferedReader buf = new BufferedReader(new FileReader(name));
String read;
Vector doubi = new Vector();
Vector doubj = new Vector();
int nl = 0; // nb de lignes dans le fichier
int ndoublei = 0; // nb de double
int ndoublej = 0; // nb de double

read = buf.readLine();
while (read != null) 
	{
	nl++;
	StringTokenizer st = new StringTokenizer(read);
	int compt=0; // nb de colonnes
	
	while (st.hasMoreTokens()) 
		{
		compt++;
		String tok = st.nextToken();
		if (compt == ic)
			{
			Double d = new Double(tok);
			doubi.addElement(d);
			ndoublei++;
			}
		else if (compt == jc)
			{
			Double d = new Double(tok);
			doubj.addElement(d);
			ndoublej++;
			}
 	 	}	
	 read = buf.readLine();
	}

buf.close();

if (ndoublei==0) throw new InvalidFormatException("The column "+ic+" doesn't exist !!!");
if (ndoublej==0) throw new InvalidFormatException("The column "+jc+" doesn't exist !!!");
if (ndoublei!=ndoublej) throw new InvalidFormatException("The columns "+ic+" and "+jc+" have not the same dimension !!!");

nb_dim=2;
dim = new int[nb_dim];
dim[1]=2;
dim[0]=ndoublei;
tab = new DVect(dim[0]*dim[1]);
for (int j=0;j<ndoublei;j++)
	{
	tab.set(2*j,((Double)doubi.get(j)).doubleValue());
	tab.set(2*j+1,((Double)doubj.get(j)).doubleValue());
	}

}


public String toString()
{
String s = "Dimension = " + nb_dim+"\n\t";
for (int i = 0; i<nb_dim; i++) s+=dim[i]+" ";
s+="\n"+tab.toString();

return s;
} 

	
/**
* Cre un nouvel objet de mme classe et de mme contenu.
* @return Object un clone de l'objet.
* @exception  OutOfMemoryError s'il n'y a pas assez de mmoire.
* @see        java.lang.Cloneable
*/
public Object clone()
	{
	try
		{
		DArray v = (DArray) super.clone();
		v.nb_dim = nb_dim;
		v.dim = new int[nb_dim];
		for (int i=0;i<nb_dim;i++) v.dim[i] = dim[i];
		v.tab = new DVect(tab);
		return v;
		}
	catch (CloneNotSupportedException e) // ne devrait jamais arriver
		{
		throw new InternalError();
		}
   	}
	
public static void main(String[] arg)throws FileNotFoundException, IOException, InvalidFormatException
{
DArray ar = new DArray();
ar.readFile("test.coupe",4);
System.out.println(ar);

ar.readFile("test.coupe",1,4);
System.out.println(ar);
}



}
