-------------------------------------------------------------------
--           RAPID - RAPID ADA PORTABLE INTERFACE DESIGNER
--           MCC GUI PACKAGE LIBRARY
--           Copyright (C) 1999 Martin C. Carlisle.
--
-- RAPID is free software;  you can  redistribute it  and/or modify
-- it under terms of the  GNU General Public License as published
-- by the Free Software  Foundation;  either version 2,  or (at your
-- option) any later version.  RAPID is distributed in the hope that
-- it will be useful, but WITHOUT ANY WARRANTY;  without even the
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
-- PURPOSE.  See the GNU General Public License for more details.
-- You should have  received  a copy of the GNU General Public License
-- distributed with RAPID; see file COPYING.  If not, write to the
-- Free Software Foundation,  59 Temple Place - Suite 330,  Boston,
-- MA 02111-1307, USA.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
------------------------------------------------------------------------------
with Gtk;                       use Gtk;
with Glib;
with Gtk.Container;
with Gtk.Event_Box;
with Gtk.Object;
with Gtk.Radio_Button;
with Gtk.Widget;
with Lists_Generic;
with Lists_Generic.Generic_Key;
with mcc.Gui.Widget.Peer;
with System;
with Ada.Text_IO;
with mcc.Gtk_Signals;

package body mcc.Gui.Widget.Button.Radio is
   -------------------------------------------------------
   -- Here's a picture of the data structure
   --
   -- A Mcc Radio_Button has a My_Peer field, which points to
   -- a Mcc_Gtk_Radio_Record.  That is the Gtk Radio button,
   -- extended with an extra field, Group, which is a pointer
   -- to a Gtk_Group
   --
   -- A Mcc Radio_Group has a My_Peer field, which points to
   -- a Gtk_Group.  The Gtk_Group contains a list of pointers
   -- to Mcc Radio_buttons
   -------------------------------------------------------

   package Radio_List is new Lists_Generic (Radio_Pointer);

   type Gtk_Group is new Gtk.Object.Gtk_Object_Record with record
      Radios : Radio_List.List;
   end record;
   type Gtk_Group_Pointer is access all Gtk_Group'Class;

   type Mcc_Gtk_Radio_Record is new Gtk.Radio_Button.Gtk_Radio_Button_Record
   with
      record
         Group : Gtk_Group_Pointer;
      end record;
   type Mcc_Gtk_Radio is access all Mcc_Gtk_Radio_Record'Class;

   function Get_Mcc_Radio
     (Button : Radio_Button'Class)
      return   Mcc_Gtk_Radio
   is
   begin
      return Mcc_Gtk_Radio (Gtk.Widget.Widget_List.Get_Data
                               (Gtk.Widget.Widget_List.First
                                   (Gtk.Container.Children
                                       (Gtk.Container.Gtk_Container (
        Button.My_Peer)))));
   end Get_Mcc_Radio;

   -- The key in this list is the Mcc_Gtk_Radio;
   -- in order to use the generic list package I already have
   -- it was easiest just to add this function
   function Get_Key (Item : in Radio_Pointer) return Mcc_Gtk_Radio is
   begin
      return Get_Mcc_Radio (Item.all);
   end Get_Key;

   package Radio_Key_List is new Radio_List.Generic_Key (
      Mcc_Gtk_Radio,
      Get_Key);

   package Mcc_Object_Callback is new mcc.Gtk_Signals.Object_Callback_Void (
      Base_Type => Mcc_Gtk_Radio_Record,
      Widget_Type => Mcc_Gtk_Radio);

   procedure Remove_From_Group (The_Radio : Mcc_Gtk_Radio) is
      P : Radio_List.Position;
   begin
      if The_Radio.Group /= null then
         P :=
            Radio_Key_List.Find
              (Key => The_Radio,
               Ptr => The_Radio.Group.Radios);
         Radio_List.Delete (L => The_Radio.Group.Radios, P => P);
      end if;
   end Remove_From_Group;

   -- Since we never reinitialize a group, it is important to catch
   -- when a radio button is destroyed, and remove it from the group
   -- ourselves.  This is especially important in RAPID, as we
   -- reuse the group for multiple copies of the same window
   procedure Mcc_Destroy_Callback (Glib_Obj : System.Address) is
      Obj : Mcc_Gtk_Radio := Mcc_Object_Callback.Find (Glib_Obj);
   begin
      Remove_From_Group (Obj.all'Unchecked_Access);
   end Mcc_Destroy_Callback;

   ------------------
   -- Add_To_Group --
   ------------------
   procedure Add_To_Group
     (Group  : in out Radio_Group;
      Button : in Radio_Pointer)
   is
      use type Gtk.Object.Gtk_Object;
   begin
      -- Check to see if this radio belongs to another group
      -- if so, remove it from that list
      Remove_From_Group (Get_Mcc_Radio (Button.all));

      -- Check to see if this is the first thing being added to the group
      if Group.My_Peer /= null then
         -- So, what we're doing here is that the group is a Feldman
         -- list of Radio_Pointers.
         if not Radio_List.IsEmpty
                  (Gtk_Group_Pointer (Group.My_Peer).Radios)
         then
            -- If not empty, We get the first one by doing
            -- a retrieve on first.  To get the Gtk radio from that,
            -- we add a .My_Peer and a cast, and then set the group
            -- using that.
            Gtk.Radio_Button.Set_Group
              (Radio_Button =>
                 Gtk.Radio_Button.Gtk_Radio_Button (Get_Mcc_Radio
                                                       (Button.all)),
               Group        =>
                  Gtk.Radio_Button.Group
                    (Radio_Button =>
                       Gtk.Radio_Button.Gtk_Radio_Button (Get_Mcc_Radio
                                                             (
              Radio_List.Retrieve
  (L => Gtk_Group_Pointer (Group.My_Peer).Radios,
   P => Radio_List.First (Gtk_Group_Pointer (Group.My_Peer).Radios)).all))));
         end if;
         -- add this radio to the group's list
         Radio_List.AddToFront
           (L => Gtk_Group_Pointer (Group.My_Peer).Radios,
            X => Button);
      else
         Group.My_Peer := new Gtk_Group;
         Radio_List.Initialize (Gtk_Group_Pointer (Group.My_Peer).Radios);
         Radio_List.AddToFront
           (L => Gtk_Group_Pointer (Group.My_Peer).Radios,
            X => Button);
      end if;
      -- remember this radio button's group
      Get_Mcc_Radio (Button.all).Group := Gtk_Group_Pointer (Group.My_Peer);
   end Add_To_Group;

   ------------
   -- Create --
   ------------

   procedure Create
     (Obj    : in out Radio_Button;
      Parent : in mcc.Gui.Container.Container'Class;
      X      : in Integer;
      Y      : in Integer;
      Width  : in Natural;
      Height : in Natural;
      Text   : in String)
   is
      New_Button    : Gtk.Radio_Button.Gtk_Radio_Button;
      New_Event_Box : Gtk.Event_Box.Gtk_Event_Box;
   begin
      Obj.Push_Handler := null;

      Gtk.Event_Box.Gtk_New (New_Event_Box);
      Obj.My_Peer := Gtk.Object.Gtk_Object (New_Event_Box);

      mcc.Gui.Widget.Peer.Setup
        (Obj    => Obj,
         X      => X,
         Y      => Y,
         Width  => Width,
         Height => Height,
         Parent => Parent);

      New_Button := new Mcc_Gtk_Radio_Record;
      -- this seems a bit odd, as we want to pass null as a group
      -- so one gets created, but recall that the internal
      -- field of New_Button will be null until after this call
      Gtk.Radio_Button.Initialize
        (Radio_Button => New_Button,
         Group        => New_Button,
         Label        => Text);

      Gtk.Container.Add
        (Container => Gtk.Container.Gtk_Container (New_Event_Box),
         Widget    => New_Button);
      Gtk.Radio_Button.Show (New_Button);
      declare
         Cb_Id : Glib.Guint;
      begin
         Cb_Id :=
            Mcc_Object_Callback.Connect
              (Obj         => Gtk.Object.Gtk_Object (Obj.My_Peer),
               Name        => "destroy",
               Func        => Mcc_Destroy_Callback'Access,
               Slot_Object => Mcc_Gtk_Radio (New_Button));
      end;
   end Create;

   ------------------
   -- Get_Selected --
   ------------------

   function Get_Selected (Group : in Radio_Group) return Radio_Pointer is
      Traverse : Radio_List.Position;
      Item     : Radio_Pointer;
      Ptr      : Radio_List.List;
   begin
      Ptr      := Gtk_Group_Pointer (Group.My_Peer).Radios;
      Traverse := Radio_List.First (Ptr);
      while not Radio_List.IsPastEnd (L => Ptr, P => Traverse) loop
         Item := Radio_List.Retrieve (Ptr, Traverse);
         if Gtk.Radio_Button.Get_Active
              (Gtk.Radio_Button.Gtk_Radio_Button (Get_Mcc_Radio (Item.all)))
         then
            return Item;
         end if;
         Radio_List.GoAhead (Ptr, Traverse);
      end loop;
      return null;
   end Get_Selected;

   ------------------
   -- Select_Radio --
   ------------------

   procedure Select_Radio (Obj : in out Radio_Button) is
   begin
      Gtk.Radio_Button.Set_Active
        (Toggle_Button =>
           Gtk.Radio_Button.Gtk_Radio_Button (Get_Mcc_Radio (Obj)),
         Is_Active     => True);
   end Select_Radio;

end Mcc.Gui.Widget.Button.Radio;
