---------------------------------------------------------------
--
--  GENERATE_HELPERS
--  Description : Use to auto generate Ada code
--
--  Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
--
-- This is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- This is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Ada.Strings.Unbounded;
with Ada.Characters.Handling; use Ada.Characters.Handling;
with Ada.Text_IO;
with File_Helpers;

package body Generate_Helpers is

   procedure Generate_Subprogram_Name (Action : in String) is
      Start, Parenthesis, Ending : Integer;
   begin
      Start       := Action'First;
      Parenthesis := Start - 1;
      Ending      := Action'Last;
      while Start <= Ending and then Action (Start) = ' ' loop
         Start := Start + 1;
      end loop;

      Ending := Start;
      while Ending <= Action'Last
        and then (Action (Ending) /= ' ' and Action (Ending) /= '(')
      loop
         Parenthesis := Ending;
         Ending      := Ending + 1;
      end loop;

      if Parenthesis > Start then
         File_Helpers.Put (Action (Start .. Parenthesis));
      end if;
   end Generate_Subprogram_Name;

   subtype Withlist_Count_T is Natural range 0 .. 100;
   Withlist : array (1 .. Withlist_Count_T'Last) of
              Ada.Strings.Unbounded.Unbounded_String;
   Withlist_Count : Withlist_Count_T := 0;

   function Already_With (Pkg : Ada.Strings.Unbounded.Unbounded_String)
            return Boolean is
      use type Ada.Strings.Unbounded.Unbounded_String;
   begin
      for I in 1 .. Withlist_Count loop
         if Pkg = Withlist (I) then
            return True;
         end if;
      end loop;
      return False;
   end Already_With;

   procedure Generate_With (Pkg : in String) is
      LC_Name : constant String := To_Lower (Pkg);
      UBS_Name : constant Ada.Strings.Unbounded.Unbounded_String
               := Ada.Strings.Unbounded.To_Unbounded_String (LC_Name);
   begin
      if LC_Name = "standard" or else Already_With (UBS_Name) then
         return;
      end if;
      Withlist_Count := Withlist_Count + 1;
      Withlist (Withlist_Count) := UBS_Name;
      File_Helpers.P ("with " & Pkg & ";");
   end Generate_With;

   procedure Generate_With_For_FQN (FQN : in String) is
      Start, Last_Period, Ending : Integer;
   begin
      Start  := FQN'First;
      Ending := FQN'Last;
      -- start should point to the first non-space character
      while Start <= Ending and then FQN (Start) = ' ' loop
         Start := Start + 1;
      end loop;

      -- Last_Period should point to the last period before
      -- a space or parenthesis
      Last_Period := Start - 1;
      Ending      := Start;
      while Ending <= FQN'Last
        and then FQN (Ending) /= ' '
        and then FQN (Ending) /= '('
      loop
         if FQN (Ending) = '.' then
            Last_Period := Ending;
         end if;
         Ending := Ending + 1;
      end loop;

      if Last_Period <= Start then
         Ada.Text_IO.Put_Line
           ("Generate_Helpers: not generating with clause for " & FQN);
         return;
      end if;
      Generate_With (FQN (Start .. Last_Period - 1));
   end Generate_With_For_FQN;

   procedure Reset_Withlist is
   begin
      Withlist_Count := 0;
   end Reset_Withlist;

   -- return name with "." converted to "_"
   function Undot_Name (Name : in String) return String is
      Result : String := Name;
   begin
      for i in Result'Range loop
         if Result (i) = '.' or else Result (i) = ' ' then
            Result (i) := '_';
         end if;
      end loop;

      return Result;
   end Undot_Name;

   function Quote_String (Item : in String) return String is
      Result : String (1 .. 2 * Item'Length);
      Last   : Natural := 0;
   begin
      for i in Item'Range loop
         if Item (i) = '"' then
            Result (Last + 1) := '"';
            Result (Last + 2) := '"';
            Last              := Last + 2;
         else
            Result (Last + 1) := Item (i);
            Last              := Last + 1;
         end if;
      end loop;
      return Result (1 .. Last);
   end Quote_String;

end Generate_Helpers;
