---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  FILE_MENU.ADB
--  Description : Implements choices from the File menu
--
--  Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with state;
with gui.Window;                  use type gui.Window.Window_Pointer;
with mcc.Common_Dialogs;
with Limits;
with new_dialog_window;
with Text_IO;
with Ada.Exceptions;
with mcc.Gui.Container.Window;
with mcc.Gui.Widget.Text_Entry;
with mcc.Gui.Widget.Button.Check;
with mcc.Gui.Widget.Dropdown;
with Gui_Enum;

package body File_Menu is
   New_Running : Boolean := False;
   -- Is a new dialog running already?

   -- used in File Open/Save dialogs to select
   -- which files are displayed
   File_Types : constant String :=
      "{ ""Gui Files"" { .gui } } " & "{ ""All Files"" * }";

   -- Holds file name and Directory name returned from
   -- File dialogs
   Filename       : String (1 .. Limits.Max_Name_Length);
   Filename_Last  : Natural;
   Directory      : String (1 .. Limits.Max_Name_Length);
   Directory_Last : Natural;

   -----------------------------------------------------------
   -- procedure New_Choice
   --
   -- called when user pushes OK in New Dialog (.new_dialog)
   --
   -- 1) Read entry1 (name)
   -- 2) Change first letter to lower case (for Tcl)
   -- 3) Replace other spaces by underscores (for Tcl)
   -- 4) Replace main by "." (for Tcl)
   -- 5) Read width and height
   -- 6) Allocate window, setting current Filename to "Untitled"
   --    (see Save for use of this) and Directory to current (.)
   -- 7) Create and display window
   -- 8) Note that window is unchanged
   -----------------------------------------------------------
   procedure New_Done is
      Window_Name   : String          :=
         mcc.Gui.Widget.Text_Entry.Get_Text (new_dialog_window.entry1);
      Window_Last   : Natural         := Window_Name'Last;
      Title         : constant String :=
         mcc.Gui.Widget.Text_Entry.Get_Text (new_dialog_window.entry4);
      Width, Height : Integer;
      Novice_Mode   : Boolean;
      Frame         : Gui_Enum.Frame_T;
   begin
      Text_IO.Put_Line ("Here is File_Menu.New_Done");
      -- eliminate trailing spaces
      while Window_Last > Window_Name'First
        and then Window_Name (Window_Last) = ' '
      loop
         Window_Last := Window_Last - 1;
      end loop;

      -- get rid of other spaces
      for i in Window_Name'First .. Window_Last loop
         if Window_Name (i) = ' ' then
            Window_Name (i) := '_';
         end if;
      end loop;

      Width := mcc.Gui.Widget.Text_Entry.Get_Text (new_dialog_window.entry2);

      Height      :=
         mcc.Gui.Widget.Text_Entry.Get_Text (new_dialog_window.entry3);
      Novice_Mode :=
         mcc.Gui.Widget.Button.Check.Is_Checked (new_dialog_window.novice);

      Frame    :=
         Gui_Enum.Frame_T'Val
           (mcc.Gui.Widget.Dropdown.Get_Selected
               (new_dialog_window.Frame_DD) -
            1);

      mcc.Gui.Destroy (mcc.Gui.Object (new_dialog_window.new_dialog_window));

      state.Set_Current_Window (state.Allocate_Window);
      state.Set_Filename ("Untitled");
      state.Set_Directory (".");

      gui.Window.Create_Window
        (Window      => state.Get_Current_Window.all,
         Width       => Width,
         Height      => Height,
         Filename    => "Untitled",
         Title       => Title,
         Window_Name => Window_Name (Window_Name'First .. Window_Last),
         Novice_Mode => Novice_Mode);

      gui.Window.Display_Window (Window => state.Get_Current_Window.all);
      state.Set_Changed (False);
      state.Control_Released;
      New_Running := False;
   end New_Done;
   procedure New_Done (Obj : in out mcc.Gui.Widget.Button.Button'Class) is
   begin
      New_Done;
   end New_Done;

   -----------------------------------------------------------
   -- procedure Cancel_New
   --
   -- Set New_Running to false
   -----------------------------------------------------------
   procedure Cancel_New is
   begin
      mcc.Gui.Destroy (mcc.Gui.Object (new_dialog_window.new_dialog_window));
      New_Running := False;
   end Cancel_New;

   procedure Cancel_New
     (Obj : in out mcc.Gui.Container.Window.Window'Class)
   is
   begin
      Cancel_New;
   end Cancel_New;

   -----------------------------------------------------------
   -- procedure New_Choice
   --
   -- 0) if New already running, then return (set New_Running)
   -- 1) Close old file (remove if MDI implemented)
   -- 2) Generate new dialog
   -- 2b) Set window close to Cancel_New (see tcl_bindings.txt)
   -- 3) Set default size to 300x300 in dialog
   -- 4) Set focus on name (entry1)
   -- 5) Exit as rest is done in New_Done
   -----------------------------------------------------------
   procedure New_Choice (Obj : in out mcc.Gui.Widget.Button.Button'Class) is
   begin
      New_Choice;
   end New_Choice;

   procedure New_Choice is
   begin -- New_Choice
      if New_Running then
         return;
      end if;
      New_Running := True;

      Close_Choice;  -- remove for MDI

      new_dialog_window.Generate_Window;

      mcc.Gui.Container.Window.Set_Close_Handler
        (Obj     => new_dialog_window.new_dialog_window'Access,
         Handler => Cancel_New'Access);

      -- set initial width and height to 300
      mcc.Gui.Widget.Text_Entry.Set_Text
        (Obj  => new_dialog_window.entry2,
         Text => "300");

      mcc.Gui.Widget.Text_Entry.Set_Text
        (Obj  => new_dialog_window.entry3,
         Text => "300");

      mcc.Gui.Widget.Text_Entry.Highlight (Obj => new_dialog_window.entry1);

      mcc.Gui.Widget.Dropdown.Select_Item
        (Obj    => new_dialog_window.Frame_DD,
         Number => Gui_Enum.Frame_T'Pos (Gui_Enum.No_Frame) + 1);

   end New_Choice;

   -----------------------------------------------------------
   -- procedure Open_Choice
   --
   -- 1) Display dialog
   -- 2) If not cancelled, run close choice (remove for MDI)
   --    this gives opportunity to save previous
   -- 3) Allocate window, setting filename and directory
   --    to result from dialog
   -- 4) Call Read_Window from Gui_Window to read file
   -- 5) Display the Window
   -- 6) Note window is unchanged
   -----------------------------------------------------------
   procedure Open_Choice (Obj : in out mcc.Gui.Widget.Button.Button'Class) is
   begin
      Open_Choice;
   end Open_Choice;

   procedure Open_Choice is

   begin -- Open_Choice
      mcc.Common_Dialogs.Open_Dialog
        (File_Types => File_Types,
         Filename   => Filename,
         File_Last  => Filename_Last,
         Directory  => Directory,
         Dir_Last   => Directory_Last);
      state.Control_Released;
      if Filename_Last > 0 then
         -- remove for MDI
         Close_Choice;

         state.Set_Current_Window (state.Allocate_Window);
         state.Set_Filename (Filename (1 .. Filename_Last));
         state.Set_Directory (Directory (1 .. Directory_Last));
         gui.Window.Read_Window
           (Window   => state.Get_Current_Window.all,
            Filename => state.Get_Filename);
         gui.Window.Display_Window (Window => state.Get_Current_Window.all);
         state.Set_Changed (False);
      end if;
   exception
      when e : others =>
         state.Set_Current_Window (null);
         mcc.Common_Dialogs.Ok_Box
           ("Unable to open: " &
            Filename (1 .. Filename_Last) &
            ASCII.LF &
            "Not a valid GUI file?" &
            ASCII.LF &
            Ada.Exceptions.Exception_Information (e));
   end Open_Choice;

   -----------------------------------------------------------
   -- procedure Close_Choice
   --
   -- 1) If a window is not open, ignore
   -- 2) If window has changed, ask if they want to save
   --    2b) If so, run Save_Choice
   -- 3) Undisplay window, set current to null (change for MDI)
   -----------------------------------------------------------
   procedure Close_Choice (Obj : in out mcc.Gui.Widget.Button.Button'Class) is
   begin
      Close_Choice;
   end Close_Choice;

   procedure Close_Choice is
      Window : gui.Window.Window_Pointer;
   begin -- Close_Choice
      Window := state.Get_Current_Window;

      if Window /= null then
         if state.Get_Changed then
            if mcc.Common_Dialogs.Yesno_Dialog
                 (Title   => "Save work?",
                  Message => "Do you want to save changes to " &
                             state.Get_Filename &
                             "?")
            then
               Save_Choice;
            end if;
         end if;
         gui.Window.Undisplay_Window (Window => state.Get_Current_Window.all);
      end if;
      state.Set_Current_Window (null);
      state.Control_Released;
   end Close_Choice;

   -----------------------------------------------------------
   -- procedure Save_Choice
   --
   -- 1) If no current window, then ignore
   -- 2) If window name is "Untitled" (see New_Choice)
   --    then go to SaveAs_Choice
   -- 3) Otherwise, ask Window to write itself
   -- 4) Mark window as unchanged
   -----------------------------------------------------------
   procedure Save_Choice (Obj : in out mcc.Gui.Widget.Button.Button'Class) is
   begin
      Save_Choice;
   end Save_Choice;

   procedure Save_Choice is

   begin -- Save_Choice
      if state.Get_Current_Window /= null then
         if state.Get_Filename = "Untitled" then
            SaveAs_Choice;
         else
            gui.Window.Write_Window (Window => state.Get_Current_Window.all);
            state.Set_Changed (False);
         end if;
      end if;
   end Save_Choice;

   -----------------------------------------------------------
   -- procedure SaveAs_Choice
   --
   -- 1) if no Window, ignore
   -- 2) Put up Save dialog
   -- 3) if we get a filename (not pressed cancel) then
   --    3b) Change window name
   --    3c) Set new filename and directory
   --    3d) call Save to perform write
   -----------------------------------------------------------
   procedure SaveAs_Choice
     (Obj : in out mcc.Gui.Widget.Button.Button'Class)
   is
   begin
      SaveAs_Choice;
   end SaveAs_Choice;

   procedure SaveAs_Choice is

   begin -- SaveAs_Choice
      if state.Get_Current_Window /= null then
         mcc.Common_Dialogs.Save_Dialog
           (File_Types => File_Types,
            Filename   => Filename,
            File_Last  => Filename_Last,
            Directory  => Directory,
            Dir_Last   => Directory_Last);
         state.Control_Released;
         if Filename_Last > Filename'First then
            state.Set_Filename (Filename (1 .. Filename_Last));
            state.Set_Directory (Directory (1 .. Directory_Last));
            gui.Window.Change_Filename
              (Window   => state.Get_Current_Window.all,
               Filename => state.Get_Filename);
            Save_Choice;
         end if;
      end if;
   end SaveAs_Choice;

   -----------------------------------------------------------
   -- procedure Exit_Choice
   --
   -- 1) Close file (to allow possible save)
   -- 2) Quit (do not ask)
   -----------------------------------------------------------
   procedure Exit_Choice
     (Obj : in out mcc.Gui.Container.Window.Window'Class)
   is
   begin
      Exit_Choice;
   end Exit_Choice;

   procedure Exit_Choice is

   begin -- Exit_Choice
      Close_Choice;
      mcc.Common_Dialogs.Quit_Dialog (Verify => False);
      state.Control_Released;
   end Exit_Choice;

end File_Menu;
