/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/

package fr.gouv.culture.sdx.application;

import fr.gouv.culture.sdx.document.Document;
import fr.gouv.culture.sdx.document.XMLDocument;
import fr.gouv.culture.sdx.documentbase.DocumentBase;
import fr.gouv.culture.sdx.documentbase.IndexParameters;
import fr.gouv.culture.sdx.documentbase.LuceneDocumentBase;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.framework.Framework;
import fr.gouv.culture.sdx.framework.FrameworkImpl;
import fr.gouv.culture.sdx.pipeline.Transformation;
import fr.gouv.culture.sdx.repository.Repository;
import fr.gouv.culture.sdx.resolver.entity.SDXResolver;
import fr.gouv.culture.sdx.search.Searchable;
import fr.gouv.culture.sdx.search.lucene.FieldsDefinition;
import fr.gouv.culture.sdx.thesaurus.SDXThesaurus;
import fr.gouv.culture.sdx.thesaurus.Thesaurus;
import fr.gouv.culture.sdx.user.*;
import fr.gouv.culture.sdx.utils.Utilities;
import fr.gouv.culture.sdx.utils.constants.Node;
import fr.gouv.culture.sdx.utils.database.DatabaseBacked;
import fr.gouv.culture.sdx.utils.rdbms.hsql.HSQLDB;
import org.apache.avalon.excalibur.xml.EntityResolver;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.context.DefaultContext;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.xml.XMLizable;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import java.io.File;
import java.net.MalformedURLException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Locale;

/** An SDX application, from which searches can be done.
 *
 * <p>
 * This class represents an SDX 2 application. There can be many applications
 * within an SDX installation. The applications within the installation are referenced
 * and managed by the {@link fr.gouv.culture.sdx.framework.Framework}.
 * The aim of an application is to make documents available and searchable.
 * <p>
 * An application is created from an XML configuration file. This file and all other files
 * needed by the application must reside within one and only one directory hierarchy on the server.
 *
 * BTW, there is also a reference to the application in {TOMCAT_HOME}/webapps/sdx/WEB-INF/sdx/applications -pb
 *
 * This directory is called the base directory, a directory hierarchy, and must be directly within the SDX
 * installation directory.
 * <p>
 * All files created by SDX regarding the management of an application and it's users and
 * groups will be placed in subdirectories within the directory where the configuration
 * file is located. it could look like this (for an application whose id is 'sdxworld') :
 * <pre>
 *   -webapps
 *     -sdx
 *       -sdxworld
 *         -conf
 *           application.xconf
 *           -dbs ({@link fr.gouv.culture.sdx.documentbase databases} needed by SDX)
 *           -repos ({@link fr.gouv.culture.sdx.repository repositories} needed by SDX)
 *           -users
 *         index.xsp (or any valid 'welcome page')
 *         ...
 *         -xsl
 *           index.xsl
 *           ...
 *
 * </pre>
 * <p>
 * An application can be either public, private or restricted. A public application is
 * searchable from other applications. A private application is not searchable from other
 * applications. A restricted application may be searchable from other applications, if they
 * qualify. Qualification is done from hosts and public ids. This could change in the future.
 * <p>
 * An appplication has four identification properties :
 * <UL>
 * <LI>a {@link #getId id}, which should be unique amongst SDX applications anywhere as would be a Java package</LI>
 * <LI>a name, for humans</LI> //TODO : where is it ? -pb
 * <LI>a description, also for humans</LI> //TODO : where is it ? -pb
 * </UL>
 * <p>
 * Other than basic properties, an application manages {@link fr.gouv.culture.sdx.documentbase document bases},
 * {@link fr.gouv.culture.sdx.user.User users} and {@link fr.gouv.culture.sdx.user.Group groups}. It
 * keeps track of the objects using Database objects and a filesystem store.
 * <p>
 * An application can also be used to search document bases within other applications.
 * <p>
 * After creating an Application object, you must remember to :
 * 1) provide a logger
 * 2) configure the application
 * 3) initialize the application
 * The methods are {@link #enableLogging enableLogging()}, {@link #configure configure()} and {@link #init init()}.
 */
public class Application implements XMLizable, Configurable, LogEnabled, Composable, Contextualizable {


    /************
     Class members
     ************/

    /** Avalon logger to write information. */
    private org.apache.avalon.framework.logger.Logger logger;
    /** The framework's context. */
    private DefaultContext context;
    /** The ComponentManager. */
    private ComponentManager manager;
    /** The application's properties. */
    private Hashtable props;
    /** The application's path. */
    private String path;
    /** The application's id. */
    private String id;
    /** The application's locale (found either in the configuration file or by a call to <code>Locale.getDefault()</code>). */
    private Locale locale;
    /** The xml:lang attribute value which has been passed to this application's configuration. */
    private String xmlLang = ""; //TODO : why retain it ? what does it bring more to locale ? -pb
    /** The repositories owned by this application (i.e. application-level repositories). */
    private Hashtable repositories;
    /**The fieldList(s) defined at the application level*/
    private Hashtable fieldDefintions = null;
    /** The document bases owned by this application. */
    private Hashtable documentBases;
    /** The default document base. */
    private DocumentBase defaultDocumentBase = null;
    /** The user database. */
    private UserDatabase userDatabase;
    /** The list of thesauri*/
    private Hashtable thesauri = new Hashtable();
    /** The info relative to the users ; incrementally feeded as incoming users are identified. */
    private Hashtable userInformations;
    /** The id of the default admin group. */
    private String defaultAdminGroupId = null;
    /** The coniguration of the default admin group. */
    private Configuration defaultAdminGroupConf = null;
    /** The id of the default admin user. */
    private String defaultAdminUserId = null;
    /** The *fixed* id of the user document base for this application. */
    public static final String USER_DOCUMENT_BASE_ID = "sdxuserdb"; //TODO : move to document base ? -pb

    /**************************************************************
     String representation of the keys used by the properties object

     TODO :
     Most of these keys seem to be redundant with class members...
     I know that these keys may be used by other classes to access properties, but, IMHO
     we should offer accessors to this properties rather than :
     1) expose the properties object to other classes : it should be "protected"
     2) expose keys to other classes : they alos should be "protected"
     Wha't your mind about tihs? -pb
     **************************************************************/

    /** String representation for a key in the Properties object : application's id. */
    public static final String APPLICATION_ID = "appId";
    /** String representation for a key in the Properties object : application-level repositories. */
    public static final String APPLICATION_REPOSITORIES = "appRepos";
    /** String representation for a key in the Properties object : application-level repositories. */
    public static final String APPLICATION_FIELD_LISTS = "appRepos";
    /** String representation for a key in the Properties object : directory for the location of the DBs. */
    public static final String DOCUMENTBASES_DIR_PATH = "dbsDirPath";
    /** String representation for a key in the Properties object : directory for the location of the repositories. */
    public static final String REPOSITORIES_DIR_PATH = "reposDirPath";
    /** String representation for a key in the Properties object : directory for the location of the users info. */
    public static final String USERS_DIR_PATH = "usersDirPath";
    /** String representation for a key in the Properties object : directory for the location of the users' documents. */
    public static final String USERS_DOCUMENTBASE_DIR_PATH = "userDocBaseDirPath";
    /** String representation for a key in the Properties object : directory for the location of the thesauri. */
    public static final String THESAURI_DIR_PATH = "thesauriDirPath";

    /***************************************
     Directory names to be provided to Lucene
     ***************************************/

    /** The name for the directory that contains the application-level repositories. */
    private final String REPOSITORIES_DIR_NAME = "repos";
    /** The name for the directory that contains the document bases. */
    private final String DOCUMENTBASES_DIR_NAME = "dbs";
    /** The name for the directory that contains the user info. */
    private final String USERS_DIR_NAME = "users";
    /** The name for the directory that contains the users DB. */
    private final String USERS_DATABASE_DIR_NAME = "userDataBase";
    /** The name for the directory that contains the users document base. */
    private final String USERS_DOCUMENTBASE_DIR_NAME = "userDocBase";
    /**The name for the directory that contains the thesauri*/
    private final String THESAURI_DIR_NAME = "thesauri";

    /****************************************
     Directory names for dynamic class loading
     ****************************************/

    /** Directory name for dynamic class loading of library files. */
    private final String LIB_DIR_NAME = "lib" + File.separator;
    /** Directory name for dynamic loading of class files. */
    private final String CLASSES_DIR_NAME = "classes" + File.separator;

    /**********************************************************************
     Attribute names for the configuration element in the configuration file
     **********************************************************************/

    /** The required attribute giving the id of this application. */
    private final String ATTRIBUTE_ID = Node.Name.ID;

    /*************************************************************************
     Child element names of the configuration element in the configuration file
     *************************************************************************/

    /** The element used to define a user document base. */
    public static final String ELEMENT_NAME_USER_DOCUMENT_BASE = "userDocumentBase"; //TODO : move it to AbstractDocumentBase.java ? -pb
    /** The element used to define the default group for admin. */
    private String ELEMENT_NAME_ADMIN = "admin"; //TODO : use a dedicated configurable class here ? -pb
    /** The required attribute used for defining the id of the admin group. */
    private String ATTRIBUTE_GROUP_ID = "groupId"; //TODO : use a dedicated configurable class here ? -pb
    /** The required attribute used for defining the default user in the admin group. */
    private String ATTRIBUTE_USER_ID = "userId"; //TODO : use a dedicated configurable class here ? -pb
    /** The implied attribute used for defining the default user's password in the admin group. */
    private String ATTRIBUTE_USER_PASSWORD = "userPassword"; //TODO : use a dedicated configurable class here ? -pb
    /** The element used to define the catalogs. */
    private final String ELEMENT_NAME_CATALOGS = "catalogs";
    /** The element used to define a catalog. */
    private final String ELEMENT_NAME_CATALOG = "catalog"; //TODO : move to a catalog class ? -pb
    /** The required attribute used for defining the source of a catalog. */
    private final String ATTRIBUTE_CATALOG_SRC = "src"; //TODO : use a dedicated configurable class here ? -pb

    /** String representation for a default document base type in the config file. */
    private final String DEFAULT_DOCUMENTBASE_TYPE = "lucene"; //TODO : move to AbstractDocumentBase ? -pb
    /** String representation for a default thesaurus type in the config file. */
    private final String DEFAULT_THESAURUS_TYPE = "lucene"; //TODO : move to AbstractDocumentBase ? or an abstract thesaurus ? -pb

    /** The key name for the default database type this application in the "props" object. */
    public static final String DEFAULT_DATABASE_CONF = "defaultDatabaseType";
    /** The key name for the database directory of this application in the "props" object. */
    public static final String SDX_DATABASE_DIR_PATH = "sdxDatabaseDirPath";
    /** The key name for the potention hsql db of this application in the "props" object. */
    public static final String HSQL_DATABASE = "hsqldb";

    /**The name for the directory that contains the databases*/
    public String DATABASES_DIR_NAME = "databases";
    /**The number of results and terms objects that will be stored in session before
     * being subject our first-in first-out removal strategy*/
    protected int sessionObjectLimit = 5;//defaulted


    /** Builds an application that must be configured afterwards. */
    public Application() {
    }

    /** Sets the logger for this application.
     *
     * @param logger	The logger.
     */
    public void enableLogging(org.apache.avalon.framework.logger.Logger logger) {
        this.logger = logger;
    }

    /**
     * Contextualize this class.
     *
     * @param context   The context provided by Cocoon.
     * @throws org.apache.avalon.framework.context.ContextException
     */
    public void contextualize(Context context) throws ContextException {
        if (context == null) throw new ContextException("Context provided was null", null);

        this.context = (DefaultContext) context;
    }

    /** Sets the application ComponentManager.
     *
     * @param componentManager The componentManager. Provided by the {@link fr.gouv.culture.sdx.framework.Framework framework} component.
     * @throws ComponentException
     */
    public void compose(ComponentManager componentManager) throws ComponentException {
        this.manager = componentManager;
    }

    /** Sets the configuration options to build the application.
     *
     * @param configuration The configuration object, previously populated in
     * {@link fr.gouv.culture.sdx.framework.FrameworkImpl},
     * that will allow the application to configure itself.
     *
     * @see #documented_application.xconf we should link to this in the future when we have better documentation capabilities
     * @throws ConfigurationException
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        Utilities.checkConfiguration(configuration);
        //getting the path of the application provided by the FrameworkImpl
        path = (String) props.get(FrameworkImpl.APP_PATH_NAME);

        // TODO: enforce some constraints on the format of this name.
        // we will see
        //getting the id from the configuration file
        id = configuration.getAttribute(ATTRIBUTE_ID);

        //ensuring we have a valid id
        if (!Utilities.checkString(id)) {
            String[] args = new String[2];
            args[0] = path;
            args[1] = configuration.getLocation();
            SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_APP_PUBLIC_ID, args, null);
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }

        //setting the id in the application's properties
        props.put(APPLICATION_ID, id);

        //xml:lang attribute is now required for an application
        this.xmlLang = configuration.getAttribute(Node.Name.XML_LANG);
        Utilities.checkConfAttributeValue(Node.Name.XML_LANG, this.xmlLang, configuration.getLocation());
        //building the locale
        this.locale = Utilities.buildLocale(configuration, null);

        //creating paths for the application's configuration directories
        //using the FrameworkImpl path context
        final String appConfPath = Utilities.getStringFromHashtable(FrameworkImpl.APP_CONF_PATH, props);
        final String dbsDirPath = appConfPath + DOCUMENTBASES_DIR_NAME + File.separator;
        final String reposDirPath = appConfPath + REPOSITORIES_DIR_NAME + File.separator;
        final String usersDirPath = appConfPath + USERS_DIR_NAME + File.separator;
        final String usersDocbaseDirPath = usersDirPath + USERS_DOCUMENTBASE_DIR_NAME + File.separator;
        final String thesauriDbsPath = appConfPath + THESAURI_DIR_NAME + File.separator;
        final String sdxDatabaseDir = appConfPath + DATABASES_DIR_NAME + File.separator;
        //setting them in application's properties
        props.put(DOCUMENTBASES_DIR_PATH, dbsDirPath);
        props.put(REPOSITORIES_DIR_PATH, reposDirPath);
        props.put(USERS_DIR_PATH, usersDirPath);
        props.put(USERS_DOCUMENTBASE_DIR_PATH, usersDocbaseDirPath);
        props.put(THESAURI_DIR_PATH, thesauriDbsPath);
        props.put(SDX_DATABASE_DIR_PATH, sdxDatabaseDir);
        try {
            props.put(HSQL_DATABASE, new HSQLDB());
        } catch (ClassNotFoundException e) {
            throw new ConfigurationException(e.getMessage(), e);
        }

        //do dynamic class loading
        loadClasses(appConfPath);
        configureDefaultDatabaseType(configuration);
        configureSessionObjectLimit(configuration);

        //global configuration objects
        configureRepositories(configuration);
        configureFieldLists(configuration);


        configureUserDocumentBase(configuration);

        //retaining the default admin group configuration if it exists
        //TODO : comment using XML elements when everything is in place -pb
        defaultAdminGroupConf = configuration.getChild(ELEMENT_NAME_ADMIN, false);


        //getting entity catalogs if specified
        //TODO : comment using XML elements when everything is in place -pb
        Configuration catalogs = configuration.getChild(ELEMENT_NAME_CATALOGS, false);
        if (catalogs != null) {
            Configuration[] catalogConfList = catalogs.getChildren(ELEMENT_NAME_CATALOG);
            if (catalogConfList != null && catalogConfList.length > 0) {
                try {
                    //adding catalog files to the sdx entity resolver
                    addEntityCatalogs(catalogConfList);
                } catch (SDXException e) {
                    throw new ConfigurationException(e.getMessage(), e);
                }
            }
        }

        //at this point, we should have a <sdx:documentBases> element containing a list of documentBases
        //intializing the array
        Configuration[] dbConfList = new Configuration[configuration.getChild(DocumentBase.ELEMENT_NAME_DOCUMENT_BASES).getChildren(DocumentBase.ELEMENT_NAME_DOCUMENT_BASE).length];
        //getting an array of configuration objects from each of the <sdx:documentBase> subElements of <sdx:documentBases> element
        dbConfList = configuration.getChild(DocumentBase.ELEMENT_NAME_DOCUMENT_BASES).getChildren(DocumentBase.ELEMENT_NAME_DOCUMENT_BASE);
        //testing if we have something
        if (dbConfList == null || dbConfList.length == 0) {
            String[] args = new String[1];
            //getting the location of the configuration file
            args[0] = ((Configuration) props.get(FrameworkImpl.APP_CONF)).getLocation();
            SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_DOCUMENTBASES_IN_CONFIG, args, null);
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }

        //building and configuring the configuring the document bases
        configureDocumentBases(dbConfList);

        //configuring any thesauri
        configureThesauri(configuration);
    }

    protected void configureFieldLists(Configuration configuration) throws ConfigurationException {
        //at this point, we should have a <sdx:repositories> element containing a list of repositories
        //intializing the array
        /*adding the repositories to this documentbase, must be done before documentBases
        are configured as global repos need to be provided to documentbases*/
        String elemNameFieldLists = LuceneDocumentBase.ELEMENT_NAME_FIELD_LIST + "s";
        Configuration[] fieldListConfList = new Configuration[configuration.getChild(elemNameFieldLists).getChildren(LuceneDocumentBase.ELEMENT_NAME_FIELD_LIST).length];
        //getting an array of configuration objects from each of the <sdx:repository> subElements of <sdx:repositories> element
        fieldListConfList = configuration.getChild(elemNameFieldLists).getChildren(LuceneDocumentBase.ELEMENT_NAME_FIELD_LIST);
        //testing if we have something
        if (fieldListConfList == null || fieldListConfList.length == 0) {
            String[] args = new String[1];
            args[0] = this.id;
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_NO_APP_REPOS_DEFINED, args, null);
            Utilities.logInfo(logger, sdxE.getMessage());
            return;
        }

        //ensuring we have a hashtable to populate
        if (this.fieldDefintions == null) this.fieldDefintions = new Hashtable();
        //iterating over the list and creating the repositories
        for (int i = 0; i < fieldListConfList.length; i++) {
            FieldsDefinition fieldsDef = null;
            try {

                //creating the repository
                fieldsDef = Utilities.configureFieldList(logger, fieldListConfList[i], props);
                String id = fieldsDef.getId();
                Utilities.checkConfAttributeValue("id", id, fieldListConfList[i].getLocation());
                //populating the hashtable
                this.fieldDefintions.put(fieldsDef.getId(), fieldsDef);

            } catch (ConfigurationException e) {
                Utilities.logException(logger, e);
                //we don't want all repository configurations to fail, so we won't throw this farther out
            }
        }

        //adding the list of repositories to the properties object
        props.put(APPLICATION_FIELD_LISTS, this.fieldDefintions);
    }

    protected void configureUserDocumentBase(Configuration configuration) throws ConfigurationException {
        //getting the <sdx:userDocumentBase> element from application.xconf
        Configuration userDbConf = configuration.getChild(ELEMENT_NAME_USER_DOCUMENT_BASE, false);
        //testing if we have something
        if (userDbConf == null) {

            /*we don't want the default from the sdx.xconf anymore
            //getting default user setup from sdx.xconf as we couldn't get it from application.xconf
            Configuration sdxConf = (Configuration) props.get(FrameworkImpl.SDX_CONF);
            if (sdxConf == null) {
                String[] args = new String[2];
                args[0] = FrameworkImpl.CONFIGURATION_FILE_NAME;
                args[1] = props.getField(FrameworkImpl.SDX_CONF_PATH);
                SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_SDX_CONFIG_FILE, null, null);
                throw new ConfigurationException(sdxE.getMessage(), sdxE);
            }
            //using default user setup from sdx.xconf as we couldn't get it from application.xconf
            userDbConf = sdxConf.getChild(ELEMENT_NAME_USER_DOCUMENT_BASE, false);
            //testing to make sure we have something to work with
            if (userDbConf == null) {
            */
            //at this point, we have no user document base configuration to work with
            String[] args = new String[2];
            args[0] = configuration.getLocation();
            //args[1] = props.getField(FrameworkImpl.SDX_CONF_PATH) + FrameworkImpl.CONFIGURATION_FILE_NAME;
            args[1] = configuration.getLocation();
            SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_USER_DOCUMENTBASE_CONFIG, args, null);
            //log a warn message here indicating that we dont have the configuration data in the application.xconf
            Utilities.logWarn(logger, sdxE.getMessage(), sdxE);
            //throw new ConfigurationException(sdxE.getMessage(), sdxE);
            //}
        }

        //ensuring we have a hashtable in which to store the docBases
        if (documentBases == null) documentBases = new Hashtable(); //TODO : instantiate it when it is declared ? -pb

        try {
            /*not needed
            //editing the props object for the userDocumentBase
            Properties userDocBaseProps = (Properties)this.props.clone();
            //creating paths for the application's configuration directories
            //using the FrameworkImpl path context
            reposDirPath = usersDirPath + REPOSITORIES_DIR_NAME + File.separator;
            //setting them in application's properties
            //removing the old repositories dir path
            userDocBaseProps.remove(REPOSITORIES_DIR_PATH);
            //adding the new repositories dir path
            userDocBaseProps.setProperty(REPOSITORIES_DIR_PATH, reposDirPath);
            */

            //creating a DB for user management
            //for now, it is a lucene document base
            //TODO : eventually provide other engines ;-) -pb
            LuceneDocumentBase userDb = new LuceneDocumentBase();
            //setting the DB's logger
            userDb.enableLogging(this.logger);
            //passing component manager (actually Cocoon's one) to the DB
            userDb.compose(manager);
            //passing the properties object to the DB
            userDb.setProperties(new Hashtable(this.props));
            //configuring the DB
            userDb.setId(USER_DOCUMENT_BASE_ID);
            userDb.configure(userDbConf);
            //initializing the DB
            userDb.init();
            //adding the DB to the hashtable
            documentBases.put(USER_DOCUMENT_BASE_ID, userDb);
        } catch (ComponentException e) {
            throw new ConfigurationException(e.getMessage(), e.fillInStackTrace());
        } catch (SDXException e) {
            throw new ConfigurationException(e.getMessage(), e.fillInStackTrace());
        }
    }

    protected void configureSessionObjectLimit(Configuration configuration) {
        this.sessionObjectLimit = configuration.getAttributeAsInteger("sessionObjectLimit", 5);//TODO refactor this string-rbp
    }

    /**Get's the global default database type
     * for use in all database backed objects
     * @param configuration The application level configuration
     * @throws ConfigurationException
     */
    private void configureDefaultDatabaseType(Configuration configuration) throws ConfigurationException {
        Configuration defaultDatabaseConf = configuration.getChild(DatabaseBacked.ELEMENT_NAME_DATABASE, false);
        if (defaultDatabaseConf != null)
            props.put(DEFAULT_DATABASE_CONF, defaultDatabaseConf);
    }

    //TODO : javadoc -pb
    private void configureRepositories(Configuration configuration) {

        //at this point, we should have a <sdx:repositories> element containing a list of repositories
        //intializing the array
        /*adding the repositories to this documentbase, must be done before documentBases
        are configured as global repos need to be provided to documentbases*/

        Configuration[] repoConfList = new Configuration[configuration.getChild(Repository.ELEMENT_NAME_REPOSITORIES).getChildren(Repository.ELEMENT_NAME_REPOSITORY).length];
        //getting an array of configuration objects from each of the <sdx:repository> subElements of <sdx:repositories> element
        repoConfList = configuration.getChild(Repository.ELEMENT_NAME_REPOSITORIES, true).getChildren(Repository.ELEMENT_NAME_REPOSITORY);
        //testing if we have something
        if (repoConfList == null || repoConfList.length == 0) {
            String[] args = new String[1];
            args[0] = this.id;
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_NO_APP_REPOS_DEFINED, args, null);
            Utilities.logInfo(logger, sdxE.getMessage());
            return;
        }

        //ensuring we have a hashtable to populate
        if (repositories == null) repositories = new Hashtable(); //TODO : instantiate when declared ? -pb
        //iterating over the list and creating the repositories
        for (int i = 0; i < repoConfList.length; i++) {
            Repository repo = null;
            try {

                //creating the repository
                repo = Utilities.createRepository(repoConfList[i], manager, props, logger);
                //populating the hashtable
                repositories.put(repo.getId(), repo);

            } catch (SDXException e) {
                //the creation of the SDXException should log this message
                //we don't want all repositories configurations to fail, so we won't throw this farther out
            } catch (ConfigurationException e) {
                Utilities.logException(logger, e);
                //we don't want all repository configurations to fail, so we won't throw this farther out
            }
        }

        //adding the list of repositories to the properties object
        props.put(APPLICATION_REPOSITORIES, repositories);
    }

    //TODO : javadoc -pb
    private void configureDocumentBases(Configuration[] dbConfList) {
        //a pointer to the first document base, for defaults
        DocumentBase firstDocumentBase = null;
        //iterating over the list of configuration objects to create the DocumentBase objects
        for (int i = 0; i < dbConfList.length; i++) {
            try {
                //reading the document base type attribute, if not we use the default, currently "lucene"
                String dbType = dbConfList[i].getAttribute(DocumentBase.ATTRIBUTE_TYPE, DEFAULT_DOCUMENTBASE_TYPE);
                //building the fully qualified class name based on the type
                String dbClassName = DocumentBase.PACKAGE_QUALNAME + dbType.substring(0, 1).toUpperCase() + dbType.substring(1, dbType.length()).toLowerCase() + DocumentBase.CLASS_NAME_SUFFIX;
                //getting the class from the class name
                Class dbClass = null;
                try {
                    dbClass = Class.forName(dbClassName);
                } catch (ClassNotFoundException e) {
                    //logging the first failure
                    String[] args = new String[2];
                    args[0] = dbConfList[i].getAttribute(DocumentBase.ATTRIBUTE_ID, this.id);
                    args[1] = e.getMessage();
                    //logging first exception
                    SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_CONFIGURE_DOCUMENT_BASE, args, e);
                    Utilities.logWarn(logger, sdxE.getMessage(), null);
                    //trying the attribute value, hopefully the user is providing a fully qualified attribute name
                    dbClassName = dbType;
                    try {
                        dbClass = Class.forName(dbClassName);
                    } catch (ClassNotFoundException e1) {
                        String[] args2 = new String[2];
                        args2[0] = dbConfList[i].getAttribute(DocumentBase.ATTRIBUTE_ID, this.id);
                        args2[1] = e1.getMessage();
                        throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_DOCUMENT_BASE, args2, e1);
                    }
                }
                //building a new instance of a DB object
                Object obj = dbClass.newInstance();

                if (obj == null) {
                    String[] args = new String[1];
                    args[0] = dbClassName;
                    throw new SDXException(logger, SDXExceptionCode.ERROR_NEW_OBJECT_INSTANCE_NULL, args, null);
                }
                //testing to see if the object implements the fr.gouv.culture.sdx.documentbase.DocumentBase Interface
                if (!(obj instanceof DocumentBase)) {
                    //the object doesn't implement our interface
                    String[] args = new String[3];
                    args[0] = DocumentBase.CLASS_NAME_SUFFIX;
                    args[1] = dbClass.getName();
                    args[2] = dbType;
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLASS_NOT_INSTANCEOF_SDX_INTERFACE, args, null);
                }
                //the object does implement our interface
                //casting it into a DocumentBase object
                DocumentBase db = (DocumentBase) obj;
                //setting the DB's logger
                db.enableLogging(this.logger);
                //passing component manager (actually Cocoon's one) to the DB
                db.compose(manager);
                //passing the context
                db.contextualize(this.context);
                //passing the properties object to the DB
                db.setProperties(new Hashtable(this.props));
                //configuring the DB
                db.configure(dbConfList[i]);
                //initializing the DB
                db.init();
                //adding the DB to the hashtable
                documentBases.put(db.getId(), db);
                //setting the default document base for this application
                //TODO : not compliant with SDX 1 behaviour ; first default one out of many was assumed -pb
                if (db.isDefault()) defaultDocumentBase = db;
                //assigning a value to the pointer
                if (i == 0) firstDocumentBase = db;
            } catch (InstantiationException e) {
                String[] args = new String[2];
                args[0] = dbConfList[i].getLocation();
                args[1] = e.getMessage();
                new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_DOCUMENT_BASE, args, e);
                //we don't want all document base configurations to fail, so we won't throw this farther out
            } catch (IllegalAccessException e) {
                String[] args = new String[2];
                args[0] = dbConfList[i].getLocation();
                args[1] = e.getMessage();
                new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_DOCUMENT_BASE, args, e);
                //we don't want all document base configurations to fail, so we won't throw this farther out
            } catch (ComponentException e) {
                String[] args = new String[2];
                args[0] = dbConfList[i].getLocation();
                args[1] = e.getMessage();
                new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_DOCUMENT_BASE, args, e);
                //we don't want all document base configurations to fail, so we won't throw this farther out
            } catch (ConfigurationException e) {
                Utilities.logException(logger, e);
                //we don't want all document base configurations to fail so, we won't throw this farther out
            } catch (ContextException e) {
                Utilities.logException(logger, e);
                //we don't want all document base configurations to fail so, we won't throw this farther out
            } catch (SDXException e) {
                //when create, exception should have been logged
                //we don't want all document base configurations to fail so, we won't throw this farther out
            }

            //if there isn't a default document base we set the default to the first one
            if (defaultDocumentBase == null) defaultDocumentBase = firstDocumentBase;
        }

    }

    private void addEntityCatalogs(Configuration[] catalogConfList) throws SDXException, ConfigurationException {
        SDXResolver resolver = null;
        String src = "";
        File srcFile = null;
        try {
            //getting our entity resolver
            Component c = this.manager.lookup(EntityResolver.ROLE);
            if (c != null && c instanceof SDXResolver)
                resolver = (SDXResolver) c;

            //verifying we have a resolver
            if (resolver == null) {
                SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_ACQUIRE_RESOLVER, null, null);
                String[] args = new String[2];
                args[0] = src;
                args[1] = sdxE.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_ENTITY_CATALOG_ADD, args, sdxE);
            }
            //adding catalogs
            for (int i = 0; i < catalogConfList.length; i++) {
                Configuration conf = catalogConfList[i];
                if (conf != null) {
                    //getting the file path
                    src = conf.getAttribute(ATTRIBUTE_CATALOG_SRC);
                    //verifying we have a good attribute
                    Utilities.checkConfAttributeValue(ATTRIBUTE_CATALOG_SRC, src, conf.getLocation());
                    //creating a file from the path
                    srcFile = Utilities.resolveFile(null, conf.getLocation(), props, src, false);
                    //adding the file to the entity resolver
                    if (srcFile != null) resolver.addCatalog(srcFile.toURL());
                }
            }
        } catch (ComponentException e) {
            throw new SDXException(logger, SDXExceptionCode.ERROR_ACQUIRE_RESOLVER, null, e);
        } catch (MalformedURLException e) {
            String[] args = new String[2];
            args[0] = src;
            args[1] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_ENTITY_CATALOG_ADD, args, e);
        } finally {
            if (resolver != null) this.manager.release(resolver);
        }
    }

    private void configureThesauri(Configuration configuration) {
        //at this point, we COULD have a <sdx:thesauri> element containing a list of thesauri
        //intializing the array
        Configuration[] thConfList = new Configuration[configuration.getChild(SDXThesaurus.ELEMENT_NAME_THESAURI, true).getChildren(SDXThesaurus.ELEMENT_NAME_THESAURUS).length];
        //getting an array of configuration objects from each of the <sdx:thesaurus> subElements of <sdx:thesauri> element
        thConfList = configuration.getChild(SDXThesaurus.ELEMENT_NAME_THESAURI, true).getChildren(SDXThesaurus.ELEMENT_NAME_THESAURUS);
        //a pointer to the first thesaurus, for defaults
        //Thesaurus firstThesaurus = null;
        //iterating over the list of configuration objects to create the thesauri objects
        for (int i = 0; i < thConfList.length; i++) {
            try {
                //reading the thesaurus type attribute, if not we use the default, currently "lucene"
                String thType = thConfList[i].getAttribute(SDXThesaurus.ATTRIBUTE_TYPE, DEFAULT_THESAURUS_TYPE);
                //building the fully qualified class name based on the type
                String thClassName = SDXThesaurus.PACKAGE_QUALNAME + thType.substring(0, 1).toUpperCase() + thType.substring(1, thType.length()).toLowerCase() + SDXThesaurus.CLASS_NAME_SUFFIX;
                //getting the class from the class name
                Class thClass = null;
                try {
                    thClass = Class.forName(thClassName);
                } catch (ClassNotFoundException e) {
                    //logging the first failure
                    String[] args = new String[2];
                    args[0] = thConfList[i].getAttribute(SDXThesaurus.ATTRIBUTE_ID);
                    args[1] = e.getMessage();
                    //logging first exception
                    SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_CONFIGURE_THESAURUS, args, e);
                    Utilities.logWarn(logger, sdxE.getMessage(), null);
                    //trying the attribute value, hopefully the user is providing a fully qualified attribute name
                    thClassName = thType;
                    try {
                        thClass = Class.forName(thClassName);
                    } catch (ClassNotFoundException e1) {
                        String[] args2 = new String[2];
                        args2[0] = thConfList[i].getAttribute(SDXThesaurus.ATTRIBUTE_ID);
                        args2[1] = e1.getMessage();
                        throw new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_THESAURUS, args2, e1);
                    }
                }
                //building a new instance of a DB object
                Object obj = thClass.newInstance();

                if (obj == null) {
                    String[] args = new String[1];
                    args[0] = thClassName;
                    throw new SDXException(logger, SDXExceptionCode.ERROR_NEW_OBJECT_INSTANCE_NULL, args, null);
                }
                //testing to see if the object implements the fr.gouv.culture.sdx.thesaurus.Thesaurus Interface
                if (!(obj instanceof Thesaurus)) {
                    //the object doesn't implement our interface
                    String[] args = new String[3];
                    args[0] = SDXThesaurus.CLASS_NAME_SUFFIX;
                    args[1] = thClass.getName();
                    args[2] = thType;
                    throw new SDXException(logger, SDXExceptionCode.ERROR_CLASS_NOT_INSTANCEOF_SDX_INTERFACE, args, null);
                }
                //the object does implement our interface
                //casting it into a Thesaurus object
                SDXThesaurus th = (SDXThesaurus) obj;
                //setting the DB's logger
                th.enableLogging(this.logger);
                //passing component manager (actually Cocoon's one) to the thesaurus
                th.compose(manager);
                //passing the properties object to the thesaurus
                th.setProperties(new Hashtable(this.props));
                //configuring the thesaurus
                th.configure(thConfList[i]);
                //initializing the thesaurus
                th.init();
                //adding the thesaurus to the hashtable
                thesauri.put(th.getId(), th);
                //setting the default thesaurus for this application
                //if (th.isDefault()) defaultThesaurus = th;
                //assigning a value to the pointer
                //if (i == 0) firstThesaurus = th;
            } catch (InstantiationException e) {
                String[] args = new String[2];
                args[0] = thConfList[i].getLocation();
                args[1] = e.getMessage();
                new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_THESAURUS, args, e);
                //we don't want all thesaurus configurations to fail, so we won't throw this farther out
            } catch (IllegalAccessException e) {
                String[] args = new String[2];
                args[0] = thConfList[i].getLocation();
                args[1] = e.getMessage();
                new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_THESAURUS, args, e);
                //we don't want all thesaurus configurations to fail, so we won't throw this farther out
            } catch (ComponentException e) {
                String[] args = new String[2];
                args[0] = thConfList[i].getLocation();
                args[1] = e.getMessage();
                new SDXException(logger, SDXExceptionCode.ERROR_CONFIGURE_THESAURUS, args, e);
                //we don't want all thesaurus configurations to fail, so we won't throw this farther out
            } catch (ConfigurationException e) {
                Utilities.logException(logger, e);
                //we don't want all thesaurus configurations to fail so, we won't throw this farther out
            } catch (SDXException e) {
                //when create, exception should have been logged
                //we don't want all document base configurations to fail so, we won't throw this farther out
            }
            //if there isn't a default document base we set the default to the first one
            //if (defaultThesaurus == null) defaultThesaurus = firstThesaurus;
        }
    }

    /** Initializes the application and makes the necessary data structures available.
     *
     * @throws SDXException
     */
    public void init() throws SDXException, ConfigurationException, ComponentException {

        //testing the directory for the path for the base directory for the document bases, to ensure it is available and we have access to it
        Utilities.checkDirectory(Utilities.getStringFromHashtable(DOCUMENTBASES_DIR_PATH, props), logger);

        //testing the directory for the path for the users configuration information to ensure it is available and we have access to it
        //removed this as the subDirectories allow for the creation of the non-existent parent
        //Utilities.checkDirectory(props.getField(USERS_DIR_PATH), logger);

        //TODO:remove soon not needed anymore
        //testing the directory for the path for of the application's LuceneDataBase for user index, to ensure it is available and we have access to it
        //File userDbDir = Utilities.checkDirectory(Utilities.getStringFromHashtable(USERS_DATABASE_DIR_PATH, props), logger);

        //creating the user database
        //userDatabase = new UserDatabase(userDbDir);
        //TODO:but this in a constant
        userDatabase = new UserDatabase("sdxuserdatabase");
        //setting the logger
        userDatabase.enableLogging(logger);
        userDatabase.compose(this.manager);
        //setting the props object
        userDatabase.setProperties(new Hashtable(props));
        //hack configuration TODO: fix me
        userDatabase.configure(new DefaultConfiguration("hack", ""));
        //initializing the userDatabase
        userDatabase.init();
        //verifying the default admin group
        verifyDefaultAdminGroup();
    }

    private void loadClasses(String appConfPath) throws ConfigurationException {
        String libPath = appConfPath + LIB_DIR_NAME;
        File libDir = new File(libPath);
        String classesPath = appConfPath + CLASSES_DIR_NAME;
        File classesDir = new File(classesPath);
        String classpath = "";
        try {
            if (libDir.exists() || classesDir.exists()) {
                classpath = (String) context.get(Constants.CONTEXT_CLASSPATH);
                if (!classpath.endsWith(File.pathSeparator))
                    classpath += File.pathSeparator;
                if (Utilities.checkString(classpath)) {
                    if (libDir.exists()) {
                        File[] libs = libDir.listFiles();
                        for (int i = 0; i < libs.length; i++)

                            classpath += libs[i].getAbsolutePath() + File.pathSeparator;
                    }

                    if (classesDir.exists())
                        classpath += classesDir.getAbsolutePath() + File.pathSeparator;
                }
                context.put(Constants.CONTEXT_CLASSPATH, classpath);
            }

        } catch (ContextException e) {
            throw new ConfigurationException(e.getMessage(), e);
        }
    }

    /** Sets the properties object.
     *
     * @param props The properties.
     */
    public void setProperties(Hashtable props) {
        this.props = props;
    }

    /**
     * Gets the application's logger.
     *
     * @return  The logger.
     */
    public Logger getLogger() {
        return this.logger;
    }

    /** Gets the id of this application.
     * @return  The id. */
    public String getId() {
        return id;
    }

    /** Gets the name of the subdirectory in which this application's data resides.
     * @return  The path. */
    public String getPath() {
        return path;
    }

    /** Gets a repository in this document base.
     *
     * @param id The repository's id, null for getting default repository
     * @return The repository object
     */
    public Repository getRepository(String id) throws SDXException {
        if (!Utilities.checkString(id))
            return null;
        else {
            Repository repo = (Repository) repositories.get(id);
            if (repo == null) {
                String[] args = new String[2];
                args[0] = id;
                args[1] = this.getId();
                throw new SDXException(logger, SDXExceptionCode.ERROR_UNKNOWN_REPOSITORY, args, null);
            } else
                return repo;
        }
    }

    /** Gets a DocumentBase in this application.
     *
     * @param dbId	The documentBase's id.
     * @return	The documentBase object.
     * @throws SDXException
     */
    public DocumentBase getDocumentBase(String dbId) throws SDXException {
        if (!Utilities.checkString(dbId)) return null;
        DocumentBase db = null;
        db = (DocumentBase) documentBases.get(dbId);
        if (db == null) {
            String[] args = new String[2];
            args[0] = dbId;
            args[1] = this.getId();
            throw new SDXException(logger, SDXExceptionCode.ERROR_UNKNOWN_DOCUMENT_BASE, args, null);
        } else
            return db;
    }

    /** Gets an enumeration of DocumentBase ids in this application.
     *  used to display information and open functionality
     * for this application.
     *
     * @return	The Enumeration of documentBase object.
     */
    public Enumeration getDocumentBasesIds() {
        if (this.documentBases != null) {
            return this.documentBases.keys();
        } else
            return null;
    }

    /**  Gets the default DocumentBase for this application.
     *
     * @return  The default documentBase, null if there is no default document base.
     */
    public DocumentBase getDefaultDocumentBase() {
        return defaultDocumentBase;
    }

    /**Returns the xml:lang attibute value from the configuration*/
    public String getXmlLang() {
        return xmlLang;
    }

    /**Returns the locale for the application*/
    public Locale getLocale() {
        return locale;
    }

    public SDXThesaurus getThesaurus(String id) throws SDXException {
        if (!Utilities.checkString(id)) return null;
        SDXThesaurus th = null;
        th = (SDXThesaurus) thesauri.get(id);
        if (th == null) {
            String[] args = new String[2];
            args[0] = id;
            args[1] = this.getId();
            throw new SDXException(logger, SDXExceptionCode.ERROR_UNKNOWN_THESAURUS, args, null);
        } else
            return th;
    }

    public Searchable getSearchable(String id) {
        Searchable searchable = null;
        if (!Utilities.checkString(id)) return null;
        searchable = (Searchable) documentBases.get(id);
        if (searchable == null)
            searchable = (Searchable) thesauri.get(id);
        return searchable;
    }

    /**
     * Return userDatabase for special usages, should be unuseful
     */
    public UserDatabase getUserDatabase() {
        return this.userDatabase;
    }

    /**Returns the id of the default admin group.
     * The id will be <code>null</code if the group has been deleted
     * from the application after start-up
     */
    public String getDefaultAdminGroupId() {
        String gid = null;
        try {
            if (this.getUserInformation(defaultAdminGroupId) != null)
                gid = defaultAdminGroupId;
        } catch (SDXException e) {
            //the defaultAdminGroup has been deleted
            Utilities.logException(logger, e);
            gid = null;
        } finally {
            return gid;
        }
    }

    /**Returns the id of the default admin user.
     * The id will be <code>null</code> if the user has been deleted
     * from the application after start-up.
     */
    public String getDefaultAdminUserId() {
        String uid = null;
        try {
            if (this.getUserInformation(defaultAdminUserId) != null)
                uid = defaultAdminUserId;
        } catch (SDXException e) {
            //the defaultAdminUser has been deleted
            uid = null;
        } finally {
            return uid;
        }
    }

    private void verifyDefaultAdminGroup() throws SDXException, ConfigurationException {
//        <sdx:amdin groupId="admins" userId="admin" userPassword="somepassword may be empty"/>
        UserDatabase udb = getUserDatabase();
        if (udb != null) {
            /*TODO:move this configuration under the declaration of a user document base as if the user document base is not specified
            neither should this, as it is of no user*/
            if (defaultAdminGroupConf != null) {
                String adminGroupId = defaultAdminGroupConf.getAttribute(ATTRIBUTE_GROUP_ID);
                Utilities.checkConfAttributeValue(ATTRIBUTE_GROUP_ID, adminGroupId, defaultAdminGroupConf.getLocation());
                String adminUserId = defaultAdminGroupConf.getAttribute(ATTRIBUTE_USER_ID);
                Utilities.checkConfAttributeValue(ATTRIBUTE_USER_ID, adminUserId, defaultAdminGroupConf.getLocation());
                String passwd = defaultAdminGroupConf.getAttribute(ATTRIBUTE_USER_PASSWORD, null);
                //create the group with the provided id if it doesn't exist yet
                if (udb.getEntity(adminGroupId) == null) {
                    //for xml content
                    javax.xml.parsers.DocumentBuilderFactory factory = javax.xml.parsers.DocumentBuilderFactory.newInstance();
                    javax.xml.parsers.DocumentBuilder builder = null;
                    try {
                        builder = factory.newDocumentBuilder();

                        org.w3c.dom.Document doc = null;
                        org.w3c.dom.Element top = null;

                        //creating the default admin group
                        doc = builder.newDocument();
                        top = doc.createElementNS(Framework.SDXNamespaceURI, Framework.SDXNamespacePrefix + ":group");
                        doc.appendChild(top);
                        top.setAttribute("id", adminGroupId);

                        Group adminGroup = new Group();
                        adminGroup.enableLogging(logger);
                        adminGroup.setId(adminGroupId);
                        adminGroup.setPreferredFilename(adminGroupId + ".xml");
                        adminGroup.setContent(doc);
                        //adding the adminGroup
                        this.addIdentity(adminGroup, null, null, null, null);

                        //setting the class field
                        this.defaultAdminGroupId = adminGroupId;

                        //creating the user
                        doc = builder.newDocument();
                        top = doc.createElementNS(Framework.SDXNamespaceURI, Framework.SDXNamespacePrefix + ":user");
                        doc.appendChild(top);
                        top.setAttribute("id", adminUserId);

                        User adminUser = new User();
                        adminUser.enableLogging(logger);
                        adminUser.setId(adminUserId);
                        adminUser.setPreferredFilename(adminUserId + ".xml");
                        adminUser.setContent(doc);
                        //adding the adminUser
                        this.addIdentity(adminUser, passwd, null, null, null);
                        //setting the class field
                        this.defaultAdminUserId = adminUserId;

                        //adding the adminUser to the adminGroup
                        this.addMember(adminGroup, adminUserId);


                    } catch (ParserConfigurationException e) {
                        throw new ConfigurationException(e.getMessage(), e);
                    } catch (SAXException e) {
                        throw new ConfigurationException(e.getMessage(), e);
                    } catch (ProcessingException e) {
                        throw new ConfigurationException(e.getMessage(), e);
                    }

                    //for xml content


                    /*
                    String xml = "";
                    //creating the default admin group
                    Group adminGroup = new Group();
                    adminGroup.enableLogging(logger);
                    adminGroup.setId(adminGroupId);
                    xml = "<sdx:group xmlns:sdx=\"http://www.culture.gouv.fr/ns/sdx/sdx\" id=\"" + adminGroupId + "\"/>";
                    try {
                        adminGroup.setContent(xml.getBytes("UTF-8"));
                    } catch (UnsupportedEncodingException e) {
                        //doing nothing because we are sure we have a good encoding, UTF-8!
                    }
                    adminGroup.setPreferredFilename(adminGroupId + ".xml");
                    //adding the adminGroup
                    try {
                        this.addIdentity(adminGroup, null, null, null, null);
                    } catch (SAXException e) {
                        throw new ConfigurationException(e.getMessage(), e);
                    } catch (ProcessingException e) {
                        throw new ConfigurationException(e.getMessage(), e);
                    }
                    //setting the class field
                    this.defaultAdminGroupId = adminGroupId;

                    //creating the user
                    User adminUser = new User();
                    adminUser.enableLogging(logger);
                    adminUser.setId(adminUserId);
                    xml = "<sdx:user xmlns:sdx=\"http://www.culture.gouv.fr/ns/sdx/sdx\" id=\"" + adminUserId + "\"/>";
                    try {
                        adminUser.setContent(xml.getBytes("UTF-8"));
                    } catch (UnsupportedEncodingException e) {
                        //doing nothing because we are sure we have a good encoding, UTF-8!
                    }
                    adminUser.setPreferredFilename(adminUserId + ".xml");
                    //adding the adminUser
                    try {
                        this.addIdentity(adminUser, passwd, null, null, null);
                    } catch (SAXException e) {
                        throw new ConfigurationException(e.getMessage(), e);
                    } catch (ProcessingException e) {
                        throw new ConfigurationException(e.getMessage(), e);
                    }
                    */


                }
                //setting the class field (FG:even if the document already exists)
                if (udb.getEntity(adminGroupId) != null) this.defaultAdminGroupId = adminGroupId;
                //setting the class field (FG:even if the document already exists)
                if (udb.getEntity(adminUserId) != null) this.defaultAdminUserId = adminUserId;

            }
        }
    }

    /**
     * Adds a member to a group. The member should be in the database.
     *
     * @param   group       The group.
     * @param   id          Id of an identity to add as a member.
     * @throws SDXException  */
    public void addMember(Group group, String id) throws SDXException {
        if (group == null || group.getId() == null || id == null) return;
        userDatabase.addMember(group, id);
        resetUserInformation(id); // to have modification of groups in userInformation
    }

    /**
     * Adds members to a group. The members should be in the database.
     *
     * @param   group       The group.
     * @param   ids         Array of ids to add as a member.
     * @throws SDXException  */
    public void addMembers(Group group, String[] ids) throws SDXException {
        if (group == null || group.getId() == null || ids == null || ids.length == 0) return;
        for (int i = 0; i < ids.length; i++) addMember(group, ids[i]);
    }

    /**
     * Replace members of a group. All members are deleted before add the ids.
     *
     * @param   group       The group.
     * @param   ids         Array of ids to add as a member.
     * @throws SDXException  */
    public void replaceMembers(Group group, String[] ids) throws SDXException {
        String[] members = userDatabase.getMembers(group.getId());
        if (members != null && members.length != 0)
            for (int i = 0; i < members.length; i++)
                resetUserInformation(members[i]);
        userDatabase.deleteMembers(group);
        addMembers(group, ids);
    }

    /** Returns the membership of a user.
     *
     * @return A hashtable with a key for each group name the user belongs to ; the values are also the group name.
     * @param username The username.
     * @throws SDXException
     public Hashtable getMembership(String username) throws SDXException {
     if (username == null)
     return null;
     else {
     User usr = new User(username);
     usr.enableLogging(logger);
     return userDatabase.getMembership(usr);
     }
     }
     */

    /** Returns the parents of an identity.
     *
     * @return A Hashtable of group object by id.
     * @param  id  name of the user (or the group).
     * @throws SDXException  */

    public Hashtable getParents(String id) throws SDXException {
        if (id == null) return null;
        return userDatabase.getParents(id);
    }

    /** Returns the members of a group.
     *
     * @return An array of strings containing the names of the members (users or groups) belonging to the group; the values are also the group name.
     * @param groupname     The groupname.
     * @throws SDXException  */
    public String[] getMembers(String groupname) throws SDXException {
        if (groupname == null)
            return null;
        else {
            return userDatabase.getMembers(groupname);
        }
    }

    /**
     * Adds a user or group to this application.
     *
     * @param   identity        The user or group.
     * @param   password        The unencoded password given to this user (may be null).
     * @param   repository      The repository where the user document is stored (if null, default repository will be used).
     * @param   params          The parameters of this adding process (may be null).
     * @param   handler         A content handler where information on the adding process will be sent (may be null).
     */
    public void addIdentity(Identity identity, String password, Repository repository, IndexParameters params, ContentHandler handler) throws SDXException, SAXException, ProcessingException {
        if (identity == null) throw new SDXException(logger, SDXExceptionCode.ERROR_IDENTITY_NULL, null, null);

        DocumentBase userDB = (DocumentBase) documentBases.get(USER_DOCUMENT_BASE_ID);
        if (userDB == null) {
            String[] args = new String[2];
            args[0] = USER_DOCUMENT_BASE_ID;
            args[1] = this.getId();
            throw new SDXException(logger, SDXExceptionCode.ERROR_UNKNOWN_DOCUMENT_BASE, args, null);
        }
        // TODO what to do for identical users ? -> params
        userDB.index(identity, repository, params, handler);
        //ensuring we have a good document
        // Utilities.checkDocument(logger, (Document) identity);
        //System.out.println("identity : " + identity.getId() + " ; " +  password);
        userDatabase.add(identity, encode(password));

        // We must reset the user information object in case informations have changed
        if (identity.getDocType() == Document.DOCTYPE_USER) resetUserInformation(identity.getId());
    }

    public boolean changePassword(String username, String oldPass, String newPass) throws SDXException {
        if (this.userDatabase == null)
            return false;
        else
            return this.userDatabase.changePassword(username, oldPass, newPass);
    }

    /**
     * Removes an identity (user or group) from this application.
     *
     * @param   identity        The identity to remove (only it's name is needed).
     */
    public void deleteIdentity(Identity identity) throws SDXException {
        if (identity == null) throw new SDXException(logger, SDXExceptionCode.ERROR_IDENTITY_NULL, null, null);
        //ensuring we have a good document
        Utilities.checkDocument(logger, identity);

        DocumentBase userDB = (DocumentBase) documentBases.get(USER_DOCUMENT_BASE_ID);
        try {
            userDB.delete(identity, null);
        } catch (SAXException e) {
            //nothing here for the null handler provided
        } catch (ProcessingException e) {
            //nothing here for the null handler provided
        }
        userDatabase.delete(identity);
        // delete UserInformation
        if (identity.getDocType() == Document.DOCTYPE_USER) resetUserInformation(identity.getId());
    }

    /**
     * Checks if a user and a plain text password match.
     *
     * <p>
     * If the password is null, there is a match if no password has been given
     * to the user. If the password is an empty string, then an empty string must
     * have been given as a password for this user at creation time.
     * <p>
     * Otherwise, there is a match if the password exactly matches (including case)
     * the user password.
     *
     * @param   username    The username to check.
     * @param   password    The password to check (may be null).
     */
    public boolean validateUser(String username, String password) throws SDXException {
        if (username == null) return false;
        //System.out.println("username : " + username + " password : " + password + " valid :" + userDatabase.checkPassword(username, encode(password)));
        return userDatabase.checkPassword(username, encode(password));
    }

    /**
     * Checks if an identity (user or group) belongs to a group.
     *
     *
     * @param   identity    The username to check.
     * @param   groupName   The password to check (may be null).
     */
    public boolean isMember(Identity identity, String groupName) throws SDXException {
        return userDatabase.isMember(identity, groupName);
    }

    /**
     * Encodes a password with this application private key.
     *
     * @param   password    The plain text password to encodeURL.
     */
    private String encode(String password) {
        if (password == null)
            return null;
        else
            return password;   //TODO?: encodeURL the password-MS, assume it should be similar to superuser encoding scheme-rbp
    }

    /** Returns information about a user.
     *
     * @param username The username (if null, anonymous user information is returned).
     * @throws SDXException
     * @return  The UserInformation object. */
    public UserInformation getUserInformation(String username) throws SDXException {

        // Check to see if we need anonymous user
        if (username == null) username = UserInformation.ANONYMOUS_USERNAME;

        // If we have the object, return it.
        if (userInformations != null && userInformations.get(username) != null) return (UserInformation) userInformations.get(username);

        // If not, build it, store it and return it
        UserInformation userInfo = userDatabase.getUserInformation(this.id, username, this.locale, this.defaultAdminGroupId);
        if (userInfo == null) return null;
        // Sets a flag to see if this user is an admin
        // if (userInfo.isMember(defaultAdminGroupId)) userInfo.setAdmin(this.id, true);
        // only useful if needed for toSAX()

        if (userInformations == null) userInformations = new Hashtable();
        userInformations.put(username, userInfo);
        return userInfo;
    }

    /**
     * Returns information about an anonymous user.
     */
    public UserInformation getUserInformation() throws SDXException {
        return getUserInformation(null);
    }


    /**
     * Returns the document where the user information is stored.
     *
     * @param   username    The username (if <code>null</code>, anonymous user information will be sent).
     * @param   groups      The groups the user belongs to (may be <code>null</code>).
     * @param   consumer     The XMLconsumer to feed with the information.
     */
    public void getUserDocument(String username, Hashtable groups, org.apache.cocoon.xml.XMLConsumer consumer) throws SDXException {
        //verifying the consumer
        Utilities.checkXmlConsumer(logger, consumer);
        if (username == null || username.equals(UserInformation.ANONYMOUS_USERNAME)) {
            AnonymousUserInformation anon = (AnonymousUserInformation) userInformations.get(UserInformation.ANONYMOUS_USERNAME);
            try {
                if (anon != null) anon.toSAX(consumer);
            } catch (SAXException e) {
                //couldn't retrive the user document
                String[] args = new String[2];
                args[0] = UserInformation.ANONYMOUS_USERNAME;
                args[1] = e.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_USER_DOC_SAX_PARSE, args, e);
            }
        } else {
            // Builds a filter if necessary
            if (groups != null && groups.size() > 0) {
                GroupInformationInserter gii = new GroupInformationInserter(groups, consumer);
                gii.enableLogging(logger);
                Transformation pipe = gii;
                ((DocumentBase) documentBases.get(USER_DOCUMENT_BASE_ID)).getDocument(new XMLDocument(username), pipe);
            } else {
                ((DocumentBase) documentBases.get(USER_DOCUMENT_BASE_ID)).getDocument(new XMLDocument(username), consumer);
            }
        }
    }

    /**
     * Resets a user information object.
     *
     * @param username	The username.
     */
    private void resetUserInformation(String username) {

        // Make sure we have a valid username
        if (username == null) return; // Nothing to do

        //ensuring the availability of data structures
        if (userInformations == null)
            userInformations = new Hashtable(); //TODO : instantiate when declared ? -pb

        //removing the user
        else if (userInformations.containsKey(username)) userInformations.remove(username);
    }

    /**Could send an XML representation of something, but currently has no function.
     *
     * @param handler	A SAX content handler to feed with events.
     * @throws SAXException
     * @throws ProcessingException
     */
    public void toSAX(ContentHandler handler) throws SAXException, ProcessingException {
        // TODO: what XML structure and information here ?

    }

    public int getSessionObjectLimit() {
        return sessionObjectLimit;
    }

}
