/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.documentbase;

import fr.gouv.culture.oai.OAIHarvester;
import fr.gouv.culture.oai.OAIRepository;
import fr.gouv.culture.sdx.document.Document;
import fr.gouv.culture.sdx.document.IndexableDocument;
import fr.gouv.culture.sdx.document.ParsableDocument;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.pipeline.Pipeline;
import fr.gouv.culture.sdx.repository.Repository;
import fr.gouv.culture.sdx.search.Searchable;
import fr.gouv.culture.sdx.utils.constants.Node;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.xml.XMLConsumer;
import org.apache.cocoon.xml.XMLizable;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.Hashtable;

/**
 * A location where documents are indexed, stored, searched and retrieved.
 */
//TODORefactor2.2: the searchable interface extension ties us to Lucene

public interface DocumentBase extends XMLizable, Configurable, LogEnabled, Composable, Contextualizable, Searchable {

    /******************************************************************
     Element name of the configuration element in the configuration file
     ******************************************************************/

    /** Super-element name in the configuration file. */
    public static final String ELEMENT_NAME_DOCUMENT_BASES = "documentBases"; //TODO : move it to application level ? -pb
    /** Element name in the configuration file. */
    public static final String ELEMENT_NAME_DOCUMENT_BASE = "documentBase"; //TODO : move it to application level ? -pb

    /**********************************************************************
     Attribute names for the configuration element in the configuration file
     **********************************************************************/

    /** The attribute giving the id of this document base ; required unless for a userDocumentBase. */
    public static final String ATTRIBUTE_ID = Node.Name.ID;
    /** The required attribute giving the type of this document base. */
    public static final String ATTRIBUTE_TYPE = Node.Name.TYPE;

    /**************************************************
     Utilities for using concrete descendant class names
     **************************************************/

    /** The document base's package name (including trailing separator "."). */
    public static final String PACKAGE_QUALNAME = "fr.gouv.culture.sdx.documentbase.";
    /** The document base's class name suffix. */
    public static final String CLASS_NAME_SUFFIX = "DocumentBase"; //So  : "fr.gouv.culture.sdx.documentbase.WhateverYouLikeDocumentBase"

    /**
     * Initializes the document base.
     *
     * <p>
     * This method must be called after the logger has been set and the configuration
     * done.
     */
    public void init() throws SDXException, ConfigurationException;

    /**
     *Sets the properties object for the implementing class.
     *
     * @param props The desired properties object
     */
    public void setProperties(Hashtable props);

    /**
     * Sets the document base id.
     *
     * @param   id      The id (should not be <code>null</code>).
     */
    public void setId(String id) throws SDXException;

    /**
     * Returns the document base id.
     */
    public String getId();

    /**
     * Returns <code>true</code> if this document base is the default in the application.
     */
    public boolean isDefault();

    /**
     * Returns a repository which is owned by this document base using its id.
     *
     * @param   id      The requested repository's id.
     * @return          The requested repository, or the default repository is the id doesn't match any repository id. TODO : dangerous behaviour IMHO -pb
     */
    public Repository getRepository(String id) throws SDXException;

    /**
     * Returns the default repository for the DocumentBase.
     */
    public Repository getDefaultRepository();

    /** Returns the default indexation pipeline used for indexation in this document base. */
    public Pipeline getIndexationPipeline();

    /**
     * Indexes a document and stores it in a repository.
     *
     * @param   doc             The document to index and add.
     * @param   repository      The repository where the document have to be stored.
     * @param   params          Parameters governing the indexing process.
     * @param   handler         The SAX events consumer that receives informations about the indexing process.
     */
    public void index(IndexableDocument doc, Repository repository, IndexParameters params, ContentHandler handler) throws SDXException, SAXException, ProcessingException;

    /**
     * Indexes documents and stores them in a repository.
     *
     * @param   docs            The documents to index and add.
     * @param   repository      The repository where the documents have to be stored.
     * @param   params          Parameters governing the indexing process.
     * @param   handler         The SAX events consumer that receives informations about the indexing process.
     */
    public void index(IndexableDocument[] docs, Repository repository, IndexParameters params, ContentHandler handler) throws SDXException, SAXException, ProcessingException;

    /**
     * Deletes a document.
     *
     * @param   doc         The id of the document to delete.
     * @param   handler     The SAX events consumer that receives informations about the deletion process.
     */
    public void delete(Document doc, ContentHandler handler) throws SDXException, SAXException, ProcessingException;

    /**
     * Deletes documents.
     *
     * @param   docs        The documents to delete.
     * @param   handler     The SAX events consumer that receives informations about the deletion process.
     */
    public void delete(Document[] docs, ContentHandler handler) throws SDXException, SAXException, ProcessingException;

    /**
     * Sends a document as SAX events.
     *
     * @param   doc             A ParsableDocument, i.e. XMLDocument or HTMLDocument.
     * @param	consumer	A SAX content handler to feed with events.
     * The wrapped contentHandler for including events within an XSP page contentHandler should be created using
     * <code>IncludeXMLConsumer stripper = new IncludeXMLConsumer(xspContentHandler);</code> and then the "stripper" should be passed
     * to this method
     */
    public void getDocument(ParsableDocument doc, XMLConsumer consumer) throws SDXException;

    /**
     * Sends a document as SAX events.
     *
     * @param   doc             A ParsableDocument, i.e. XMLDocument or HTMLDocument.
     * @param	consumer	A SAX content handler to feed with events.
     *@param   docTypeKnown If set to <code>false</code> implementations should attempt to determine the doctyp(see LuceneDocumentBase)
     * The wrapped contentHandler for including events within an XSP page contentHandler should be created using
     * <code>IncludeXMLConsumer stripper = new IncludeXMLConsumer(xspContentHandler);</code> and then the "stripper" should be passed
     * to this method
     */
    public void getDocument(ParsableDocument doc, XMLConsumer consumer, boolean docTypeKnown) throws SDXException;


    /** Supplies the provided output stream with the requested document
     * @param doc	The document.
     * @param os	The output stream.
     * @throws SDXException
     */
    public void getDocument(Document doc, OutputStream os) throws SDXException;

    /** Provides the requested SDX document as an <code>InputStream</code>.
     * @param doc	The document.
     * @return	An input stream.
     * @throws SDXException
     */
    public InputStream getDocument(Document doc) throws SDXException;

    /** Supplies the mimeType for the the document if it exists.
     * otherwise the default mimetype should be returned TODO : dangerous behaviour IMHO -pb
     * @param doc	The document.
     */
    public String getMimeType(Document doc) throws SDXException;

    /**Gets the OAIRepository for the documentbase if one exists
     *
     * @return
     */
    OAIRepository getOAIRepository();

    /**Gets the AbstractDocumentBaseOAIHarvester for the documentbase if one exists
     *
     * @return
     */
    OAIHarvester getOAIHarvester();

    Date lastModificationDate();

    Date creationDate();

}
