/*
Copyright (C) 2000-2010  Ministere de la culture et de la communication (France), AJLSM
See LICENCE file
*/
package fr.gouv.culture.sdx.utils.database;

import fr.gouv.culture.sdx.utils.AbstractSdxObject;
import fr.gouv.culture.sdx.utils.Utilities;

import java.util.ArrayList;
import java.util.Iterator;

/**
 * An entity's property in SDX simple database abstraction.
 *
 * <p>
 * An SDX simplified database can contain entities. These entities contain properties,
 * which are name=value pairs, all strings. This class represents a property.
 * <p>
 * A property can have multiple values, but in an entity all properties have
 * different names.
 */
public class Property extends AbstractSdxObject {


    public static final String CLASS_NAME_SUFFIX = "Property";

    /** Property's name. */
    protected String name;

    /** Property's values. */
    protected ArrayList values = new ArrayList();

    protected String firstValue = null;


    /**Builds a property object
     *
     * The super.getLog() should be set after building this object.
     * Then there are methods to set the name of this property and its values.
     * @see #enableLogging
     * @see #setName
     * @see #setValues
     */
    public Property() {
    }

    /**
     * Builds a property from a name=value pair.
     *The super.getLog() should be set after building this object.
     * @param   name    The name for the property
     * @param   value   Its value
     * @see #enableLogging
     */
    public Property(String name, String value) {
        this.name = name;
        attemptToSetFirstValue(value);
        this.values.add(value);
        //System.out.println("Value added for property : " + this.getName() + " - " + value);
    }

    /**
     * Sets the property's name.
     *
     * @param   name    The property's name.
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Sets the values from an array.
     *
     * @param   values      The array containing the values.
     */
    public void setValues(String[] values) {
        if (values != null) {
            // Reset the values first
            resetValues();
            for (int i = 0; i < values.length; i++) {
                String value = values[i];
                this.addValue(value);
            }
        }
    }

    /**
     * Returns the property's name.
     */
    public String getName() {
        return this.name;
    }

    /**
     * Returns the values as an array.
     */
    public String[] getValues() {
        if (this.values != null) {
            Iterator keys = this.values.iterator();

            String[] vals = new String[this.values.size()];

            for (int i = 0; i < vals.length && keys.hasNext(); i++) {
                vals[i] = (String) keys.next();

            }

            return vals;

        } else
            return new String[0];
    }

    /**
     * Returns the first value.
     */
    public String getValue() {
        return this.firstValue;
    }


    /**Deletes the value from the list
     *
     * @param value The value to delete
     */
    public void deleteValue(String value) {
        if (values != null && value != null)
            this.values.remove(value);
    }

    public void addValue(String value) {
        if (this.values != null && value != null) {
            attemptToSetFirstValue(value);
            this.values.add(value);
        }
    }

    protected void attemptToSetFirstValue(String value) {
        if (Utilities.checkString(this.firstValue))
            return;
        else
            firstValue = value;
    }

    protected void resetValues() {
        if (this.values != null)
            this.values.clear();
        this.firstValue = null;
    }

    public boolean containsValue(String value) {
        return this.values.contains(value);
    }

    protected String getClassNameSuffix() {
        return Property.CLASS_NAME_SUFFIX;
    }

	/* (non-Javadoc)
	 * @see fr.gouv.culture.sdx.utils.AbstractSdxObject#initToSax()
	 */
	protected boolean initToSax() {
		return true;
	}

	/**Init the LinkedHashMap _xmlizable_volatile_objects with the objects in order to describ them in XML
	 * Some objects need to be refresh each time a toSAX is called*/
	protected void initVolatileObjectsToSax() {
	
	}


}
