/* Copyright (c) 2008 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: board_ravrf.h,v 1.3 2009/02/01 07:58:19 awachtler Exp $ */
/**
 * @file
 * @brief Definitions for platform @ref grpRZxxx.
 * @defgroup grpRZxxx RZ Raven
 * @brief Atmel Raven Development Kit, AT86RF230 Radio Adapter with Atmega1284.
 *
 * @ingroup grpBoard
 *
 * This board wiring fits the Atmel radio development kit hardware.
 *
 * The wiring of the radio and the Atmega1284 is shown below:
 *
<pre>
     AVR            RF230
     ---            -----
     PB3      -->  SLPTR
     XTAL1    <--  MCLK
     PD6/ICP  <--  IRQ
     PB1      -->  RSTN
     PB4      -->  SEL
     PB5      -->  MOSI
     PB6      <--  MISO
     PB7      -->  SCLK
     PB0      -->  TST

    LEDs: None
    KEYs: None

Fuses/Locks?????:
     LF: 0xe2 - 8MHz internal RC Osc.
     HF: 0x11 - without boot loader
     HF: 0x10 - with boot loader
     EF: 0xff
     LOCK: 0xef - protection of boot section

Original Fuses/Locks
     LF: 0xde
     HF: 0x91
     EF: 0xfb
     LOCK: 0xff

Bootloader:
    Start at byte=0x1e000, address=0xf000, size = 4096 instructions/ 8192 bytes

</pre>

 */

/**
 * @addtogroup grpRZxxx
 * @{
 */

/** ID String for this hardware */

#define BOARD_NAME_RAVRF "ravrf230"

#ifndef BOARD_RAVRF_H
#define BOARD_RAVRF_H

#define BOARD_TYPE (RAV_RF230)           /**< current board type (see const.h)*/
#define BOARD_NAME BOARD_NAME_RAVRF      /**< current board name */

/*=== Compile time parameters ========================================*/
#ifndef MAX_FRAME_SIZE
# define MAX_FRAME_SIZE (127) /**< maximum allowed frame size */
#endif

#ifndef DEFAULT_SPI_RATE
# define DEFAULT_SPI_RATE  (SPI_RATE_1_2)
#endif

/*=== Hardware Components ============================================*/
#define CPU_TYPE   (CPU_M1284)          /**< used CPU (see const.h)*/
#ifndef RADIO_TYPE
#define RADIO_TYPE (RADIO_AT86RF230)    /**< used radiio (see const.h)*/
#endif

/*=== TRX pin access macros ==========================================*/

#define DDR_TRX_RESET   DDRB            /**< DDR register for RESET pin */
#define PORT_TRX_RESET  PORTB           /**< PORT register for RESET pin */
#define MASK_TRX_RESET  (_BV(PB1))      /**< PIN mask for RESET pin */
/* check in board.h if standard definitions of TRX_RESET_INIT,
   TRX_RESET_HIGH, TRX_RESET_LOW are Ok, otherwise define here */

#define PORT_TRX_SLPTR  PORTB           /**< DDR register for SLEEP_TR pin */
#define DDR_TRX_SLPTR   DDRB            /**< PORT register for SLEEP_TR pin */
#define MASK_TRX_SLPTR  (_BV(PB3))      /**< PIN mask for SLEEP_TR pin */
/* check in board.h if standard definitions of TRX_SLPTR_INIT,
   TRX_SLPTR_HIGH, TRX_SLPTR_LOW are Ok, otherwise define here */

/*=== IRQ access macros ==============================================*/
# define TRX_IRQ         _BV(ICIE1)     /**< interrupt mask for GICR */
# define TRX_IRQ_vect    TIMER1_CAPT1_vect    /**< interrupt vector name */

/** configuration of interrupt handling */
# define TRX_IRQ_INIT()  do{\
                            TIMSK1 |= _BV(ICIE1);\
                          } while(0) /** rising edge triggers INT... */

/** disable TRX interrupt */
#define DI_TRX_IRQ() {TIMSK1 &= (~(TRX_IRQ));}
/** enable TRX interrupt */
#define EI_TRX_IRQ() {TIMSK1 |= (TRX_IRQ);}

/** timestamp register for RX_START event */
#define TRX_TSTAMP_REG ICR1

/*=== SPI access macros ==============================================*/
#define SPI_TYPE  SPI_TYPE_SPI
#define DDR_SPI  (DDRB)    /**< DDR register for SPI port */
#define PORT_SPI (PORTB)   /**< PORT register for SPI port */

#define SPI_MOSI _BV(PB5)  /**< PIN mask for MOSI pin */
#define SPI_MISO _BV(PB6)  /**< PIN mask for MISO pin */
#define SPI_SCK  _BV(PB7)  /**< PIN mask for SCK pin */
#define SPI_SS   _BV(PB4)  /**< PIN mask for SS pin */

#define SPI_DATA_REG SPDR  /**< abstraction for SPI data register */


/**
 * @brief inline function for SPI initialization
 */
static inline void SPI_INIT(uint8_t spirate)
{
    /* first configure SPI Port, then SPCR */
    DDR_SPI  |= SPI_MOSI | SPI_SCK | SPI_SS;
    DDR_SPI  &= ~SPI_MISO;
    PORT_SPI |= SPI_SCK | SPI_SS;

    SPCR = (_BV(SPE) | _BV(MSTR));

    SPCR &= ~(_BV(SPR1) | _BV(SPR0) );
    SPSR &= ~_BV(SPI2X);

    SPCR |= (spirate & 0x03);
    SPSR |= ((spirate >> 2) & 0x01);

}

/** set SS line to low level */
#define SPI_SELN_LOW()       uint8_t sreg = SREG; cli(); PORT_SPI &=~SPI_SS
/** set SS line to high level */
#define SPI_SELN_HIGH()      PORT_SPI |= SPI_SS; SREG = sreg
/** wait until SPI transfer is ready */
#define SPI_WAITFOR()        do { while((SPSR & _BV(SPIF)) == 0);} while(0)

/*=== LED access macros ==============================================*/
#define NO_LEDS       (1)        /**< if defined, no LEDs are connected */
#if 0
#define LED_PORT      PORT.      /**< PORT register for LEDs */
#define LED_DDR       DDR.       /**< DDR register for LEDs */
#define LED_MASK      (0x..)     /**< MASK value for LEDs (msb aligned)*/
#define LED_SHIFT     (.)        /**< SHIFT value for LEDs */
#define LEDS_INVERSE  (.)        /**< = 1, if low level at port
                                      means LED on */
#define LED_NUMBER    (.)        /**< number of LEDs for this board */
#endif

/*=== KEY access macros ==============================================*/
#define NO_KEYS       (1)        /**< if defined, no KEYS are connected */
#if 0

#define PORT_KEY      PORT.      /**< PORT register for keys */
#define PIN_KEY       PIN.       /**< PIN register for keys */
#define DDR_KEY       DDR.       /**< DDR register for keys */
#define MASK_KEY      (0x..)     /**< MASK value for keys (msb aligned) */
#define SHIFT_KEY     (.)        /**< SHIFT value for keys */
#define INVERSE_KEYS  (.)        /**< if 1, low level at port means that
                                      KEY is pressed */
#define PULLUP_KEYS   (.)        /**< if 1, port bits are set to enable
                                      internal pullup resistors of the MCU */
#endif
/*=== Host Interface ================================================*/
/** Type of the host interface. */
#define HIF_TYPE    HIF_UART_0

/*=== TIMER Interface ===============================================*/
#define HWTMR_PRESCALE  (1)
#define HWTIMER_TICK    ((1.0*HWTMR_PRESCALE)/F_CPU) /**< hardware timer clock period in us
                                      (usually: prescaler / F_CPU) */
#define HWTIMER_TICK_NB (0xFFFFUL) /**< number of hardware timer ticks,
                                        when IRQ routine is called */
#define HWTIMER_REG (TCNT1) /**< name of the register where the clock
                                 ticks can be read */
#define TIMER_TICK (HWTIMER_TICK * HWTIMER_TICK_NB)
                            /**< period in us, when the timer interrupt
                                 routine is called */

#define TIMER_POOL_SIZE (0) /**< number of software timers running at
                                 a time */

#define TIMER_INIT() \
        do{\
            TCCR1B |= (_BV(CS10)); \
            TIMSK1 |= _BV(TOIE1); \
        }while(0) /**< macro that initializes the
                                       hardware timer */

/** symbolic name of the timer interrupt routine that is called */
#define TIMER_IRQ_vect   TIMER1_OVF1_vect


/*=== OSCCAL tuning =================================================*/
#ifndef TUNED_OSCCAL
# define TUNED_OSCCAL (0xbf)  /* default is 0xb1, but @2.9V 0xbf is better */
#endif
#endif /** @} BOARD_RAVRF_H*/
