/* Copyright (c) 2013 Daniel Thiele, Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id$ */
/** Example for use of Lightweight Mesh, third node (receiving) */

/* === includes ============================================================ */
/* uracoli inclusions */
#include "board.h"
#include "ioutil.h"
#include "transceiver.h"
#include "lw_mesh.h"

#include "tlogger.h"
#include "sensordata.h"

/* === macros ============================================================== */

/* === types =============================================================== */

/* === globals ============================================================= */

/* IEEE802.15.4 parameters for communication */
const node_config_t PROGMEM nc_flash =
{
	.short_addr = 0x0000,
	.pan_id = 0x3412,
	.channel = 17
};

static node_config_t NodeConfig;

/* === prototypes ========================================================== */

/* === functions =========================================================== */
static bool rx_endpoint_data_ind(NWK_DataInd_t *ind)
{
	sensormask_t sensor;
	uint8_t idx = 0;
	char *s;

	PRINTF("ADDR=0x%04X, RSSI=%u", ind->srcAddr, ind->rssi);
	do{
		sensor = ind->data[idx++];
		switch(sensor){
		case SENSORDATA_NONE:
			break;
		case SENSORDATA_TEMPERATURE_RFA:
			PRINTF(", Temperature[degC]=%.2f", *((float*)&ind->data[idx]));
			idx += sizeof(float);
			break;
		case SENSORDATA_VOLTAGE_AVR:
			PRINTF(", VMCU[mV]=%.2f", *((float*)&ind->data[idx]));
			idx += sizeof(float);
			break;
		case SENSORDATA_CAPTION:
			s=(char*)&ind->data[idx];
			PRINTF(", CAPT=%s", s);
			idx += strlen(s) + 1; /* plus NULL terminator */
			break;
		default:
			PRINTF(", unknown(%02X)", sensor);
		}
	}while( (sensor != SENSORDATA_NONE) && (idx < ind->size) );

	PRINTF(EOL, 0);

    /* do not send acknowledgment frames except underlying layers needs some */
    return false;
}

void app_task_handler()
{
    /* do some application specific things */
}

/*
 * \brief Initialize Application
 *
 */
void app_init(void)
{
	char cfg_location = '?';

	/* === read node configuration data ===================================== */
	/* 1st trial: read from EEPROM */
	if (get_node_config_eeprom(&NodeConfig, 0) == 0)
	{
		/* using EEPROM config */;
		cfg_location = 'E';
	}
	/* 2nd trial: read from FLASHEND */
	else if (get_node_config(&NodeConfig) == 0)
	{
		/* using FLASHEND config */;
		cfg_location = 'F';
	}
	/* 3rd trial: read default values compiled into the application */
	else
	{
		/* using application default config */;
		memcpy_P(&NodeConfig, (PGM_VOID_P) & nc_flash, sizeof(node_config_t));
		cfg_location = 'D';
	}

	lw_mesh_init(NodeConfig.pan_id, NodeConfig.short_addr, NodeConfig.channel);

#if defined(SR_RND_VALUE)
	/* TODO: if the random value is not available in HW, e.g. AT86RF230A/B,
	 * we need to find another method of seeding CSMA  */
	trx_reg_write(RG_CSMA_SEED_0, trx_bit_read(SR_RND_VALUE) | (trx_bit_read(SR_RND_VALUE) << 4));
	trx_reg_write(RG_CSMA_SEED_1, trx_bit_read(SR_RND_VALUE) | (trx_bit_read(SR_RND_VALUE) << 4));
#endif

	/* Welcome Blink */
	LED_SET(0);
	LED_SET(1);
	_delay_ms(20);
	LED_CLR(0);
	LED_CLR(1);
}

int main(void)
{
    /* init LEDs */
    LED_INIT();
    LED_SET_VALUE(LED_MAX_VALUE);
    LED_SET_VALUE(0);

	app_init();
	hif_init(HIF_DEFAULT_BAUDRATE);

    sei();

    /* register endpoint */
    lw_mesh_open_endpoint(RX_ENDPOINT_ID, rx_endpoint_data_ind);

	
	PRINTF("LWM_Tlogger Host Application PAN=%04X, ADDR=%04X, CHN=%u"EOL,
			NodeConfig.pan_id, NodeConfig.short_addr, NodeConfig.channel
			);
	
    while(1)
    {
        lw_mesh_task_handler();
        app_task_handler();
    }
}
/* EOF */
