# The `eigo` Package

**Enhanced Go diagrams for LaTeX**

Version: 1  
Date: 2025-09-06  
Author: Marc Levivier (with AI assistance)  
License: LaTeX Project Public License v1.3c or later  

## Description

The `eigo` package provides comprehensive tools for creating Go (Weiqi/Baduk) game diagrams in LaTeX documents. It offers enhanced features including:

- **Full and partial board displays** with automatic zone detection and enhanced borders
- **Multiple stone colors** (black, white, red, blue, green) with full RGB customization
- **Two custom colors** (customone, customtwo) with dedicated commands and default styling
- **Automatic numbering systems** with alternating colors or monochrome sequences
- **Symbol placement** (triangles, squares, circles, crosses, custom text)
- **Geometric transformations** (rotations, mirrors, color swapping)
- **Flexible sizing** with extended predefined sizes and validation
- **Coordinate display** with customizable formatting
- **Multiple board support** for variations and analysis
- **LuaLaTeX compatibility** with enhanced UTF-8 handling
- **borders** for publication-quality diagrams

## Key Features

### Stone Placement and Customization
- Basic commands: `\blackstones`, `\whitestones`, `\redstones`, `\bluestones`, `\greenstones`
- Custom colors: `\customonestones`, `\customtwostones` 
- Full color customization: `\eigoblackcolor{R,G,B}`, `\eigowhitecolor{R,G,B}`, etc.
- Automatic numbering: `\autoblackfirst`, `\autowhitefirst`, `\blackn`, `\whiten`
- Position format: letter+number (e.g., `d4`, `k10`, `q16`)

### Enhanced Board Display
- Full board: `\showfullgoban` with **enhanced 2pt borders**
- Zones: `\showgoban[a1,s19]` or `\showgoban[k10]` (auto 5×5)
- Frame control: `\eigoframe`, `\eigonoframe`
- Extended sizing: `\tinygoban`, `\smallgoban`, `\normalgoban`, `\largegoban`, `\hugegoban`, `\massivegoban`
- Custom size with validation: `\eigounitsize{10mm}`

### Transformations
- Rotations: `\eigorotate{90}`, `\eigorotate{180}`, `\eigorotate{270}`
- Mirrors: `\eigomirrorh`, `\eigomirrorv`, `\eigomirrordiag`
- Color swap: `\eigoswapcolors`

### Symbols and Markup
- Geometric symbols: `\gobansymbol{d4,k10}{\eigotriangle}`
- Custom text: `\gobansymbol{d4}{A}`
- Available symbols: `\eigotriangle`, `\eigosquare`, `\eigocircle`, `\eigocross`

## Quick Start

```latex
\documentclass{article}
\usepackage[frame,coords]{eigo}  % Enable enhanced borders and coordinates

\begin{document}
% Customize colors if desired
\eigocustomonecolor{255,0,255}   % Bright magenta
\eigocustomtwocolor{0,255,255}   % Cyan

% Set up a position
\blackstones{d4,k10,q16}
\whitestones{d16,k4,q4}
\customonestones{d10}            % Use custom color

% Show the full board with enhanced borders
\showfullgoban

% Show just a corner
\showgoban[a1,j10]

% Add numbered sequence with color alternation
\autoblackfirst[1]{e4,f5,g6}

% Show the result
\showgoban[e4]
\end{document}
```

## Color Customization

**NEW FEATURE**: Full control over all stone colors:

```latex
% Customize standard colors
\eigoblackcolor{50,50,50}        % Dark gray instead of black
\eigowhitecolor{240,240,240}     % Off-white instead of pure white
\eigoredcolor{180,0,0}           % Deep red
\eigobluecolor{0,0,180}          % Deep blue  
\eigogreencolor{0,120,0}         % Dark green

% Customize additional colors
\eigocustomonecolor{128,0,128}   % Purple (default)
\eigocustomtwocolor{255,165,0}   % Orange (default)

% Use all colors in your diagrams
\blackstones{d4}
\whitestones{d5}
\redstones{e4}
\bluestones{e5}
\greenstones{f4}
\customonestones{f5}
\customtwostones{g4}
```

## Package Options

- `color` / `bw`: Enable/disable colored stones and backgrounds
- `frame` / `noframe`: Show/hide enhanced board frame and background
- `coords` / `nocoords`: Show/hide coordinate labels
- `debug`: Enable debug output with detailed logging

Example: `\usepackage[frame,coords,color]{eigo}`

## Board Sizes

Supports boards from 2×2 to 25×25 with **extended size options**:
- `\gobansize{19}` - Standard 19×19 board
- `\gobansize{13}` - 13×13 board
- `\gobansize{9}` - 9×9 board

**Extended size presets with validation**:
- `\tinygoban` - 4mm per intersection (for overviews)
- `\smallgoban` - 6mm per intersection
- `\normalgoban` - 8mm per intersection (default)
- `\largegoban` - 12mm per intersection
- `\hugegoban` - 16mm per intersection
- `\massivegoban` - 20mm per intersection (maximum practical)

The package automatically validates sizes and issues warnings for very small (<2mm) or very large (>20mm) unit sizes.

## Enhanced Features

### Border Display
When using `\eigoframe` with full board display:
- **Thick 2pt borders** around the complete board
- Colored background in color mode
- Appearance suitable for publication
- Automatic detection of full vs partial board display

### Advanced Automatic Numbering
- **Skip positions**: Use `-` in position lists to skip numbering while maintaining sequence
- **Alternating colors**: `\autoblackfirst[1]{d4,d5,-,e5}` (skips d5, continues with correct numbers)
- **Monochrome sequences**: `\blackn[1]{d4,e4,f4}` creates sequence 1,3,5
- **Flexible starting numbers**: Begin sequences at any number

### Size Information Commands
```latex
\theeigountisize    % Get current unit size
\eigosizeinfo       % Display current size information
```

## Multiple Boards

Use `\usegoban{n}` to work with board number n, each with independent color settings:
```latex
\usegoban{1}
\eigoblackcolor{0,0,0}           % Standard black for main line
\blackstones{d4,k10}

\usegoban{2}
\eigoblackcolor{100,100,100}     % Gray for variation
\blackstones{d4,k10,q16}
```

## Requirements

- LaTeX with TikZ and xcolor packages
- Optional: LuaLaTeX for enhanced UTF-8 support
- Recommended: Modern TeX distribution (TeX Live 2020+)

## Installation

### From CTAN (when available)
```bash
tlmgr install eigo
```

### Manual Installation
1. Download `eigo.sty`
2. Place in your local texmf tree or document directory
3. Run `texhash` if installed system-wide

## Documentation

See `eigo.pdf` for complete documentation with examples, advanced usage, and troubleshooting guide.

## Advanced Examples

### Publication-Quality Problem Diagram
```latex
\usepackage[frame,coords]{eigo}

\cleargoban
\eigoframe                       % Enable borders
\blackstones{f3,g3,h3,g4,h4,g5}
\whitestones{f4,f5,g6,h5,h6}
\gobansymbol{f6}{A}
\gobansymbol{h7}{B}
\showgoban[e2,j8]               % Show with enhanced border
```

### Multi-Color Game Analysis
```latex
\eigoblackcolor{0,0,0}          % Black for actual moves
\eigoredcolor{200,0,0}          % Red for good moves  
\eigobluecolor{0,0,200}         % Blue for questionable moves
\eigocustomonecolor{0,150,0}    % Green for best moves

\blackstones{d4,k10}            % Actual game
\redstones{d10}                 % Good alternative
\bluestones{q4}                 % Questionable move
\customonestones{k4}            % Best move
```

### Variation Tree with Color Coding
```latex
% Main line (Board 1)
\usegoban{1}
\eigoblackcolor{0,0,0}
\blackstones{d4,q16,d16,q4}

% Variation A (Board 2)  
\usegoban{2}
\eigoblackcolor{100,0,0}        % Slightly red tint
\blackstones{d4,q16,d16}
\customonestones[1]{q3}         % Variation move in custom color

% Variation B (Board 3)
\usegoban{3}  
\eigoblackcolor{0,0,100}        % Slightly blue tint
\blackstones{d4,q16,d16}
\customtwostones[1]{o17}        % Different variation
```

## Troubleshooting

### Color Issues
- **Colors not changing**: Ensure color commands are called before stone placement
- **Custom colors reverting**: Color changes affect only subsequent stone placements
- **Colors too similar**: Use high contrast RGB values for clarity

### Display Issues  
- **Borders not showing**: Enable frame mode with `\eigoframe` and use full board display
- **Board too small/large**: Use size validation - package warns about extreme sizes
- **Coordinates misaligned**: Adjust `\eigocoorddistance{0.7}` as needed

### Debug Mode
Enable comprehensive debugging:
```latex
\usepackage[debug]{eigo}
```

Provides detailed console output about position parsing, color changes, size validation, and rendering decisions.

## License

This package is released under the LaTeX Project Public License v1.3c or later.
See http://www.latex-project.org/lppl.txt

## Support

For bug reports and feature requests, please contact the maintainer or visit the package homepage.

---

*The `eigo` package - Go diagrams for LaTeX with full customization*