/****************************************************************************
 *
 * Copyright (c) 1992-1996 John Forkosh Associates.  All rights reserved.
 * --------------------------------------------------------------------------
 *
 * Functions:	--- WordPerfect-to-TeX functions ---
 *		wp2tex(tpfcb,inbuff,outbuff) Xlates WP inbuff to TeX outbuff.
 *		tokenmode(tpfcb,nwhite,tokmem,funcmode) set TeX mode of token.
 *		modifytok(tpfcb,tokmem,tokpunc,tokmode,funcmode) applies TeX
 *				syntax modifications to token.
 *		modifykwd(tpfcb,tokmem,funcmode) converts WP
 *				equation editor keywords to TeX.
 *		--- Ancillary functions ---
 *		texhdr(tpfcb,idisplay) displays TeX header or trailer.
 *
 * Notes:     o	See individual function headers.
 *
 * Source:	WP2TEX.C
 *
 * --------------------------------------------------------------------------
 * Revision History:
 * 01/05/93	J.Forkosh	Installation.
 * 08/08/96	J.Forkosh	Modified and streamlined for copyleft release
 *
 ***************************************************************************/

/* --------------------------------------------------------------------------
Headers and Static Data
-------------------------------------------------------------------------- */
/* --- standard headers --- */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/* --- application headers --- */
#include "texperf.h"
#include "eqnkwds.h"

/****************************************************************************
 * Function:	wp2tex ( tpfcb, inbuff, outbuff )
 * Purpose:	Process next token from WordPerfect file,
 *		adjusting mode as necessary.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	File control block for WordPerfect document.
 *		inbuff (I)	address of MEMBUFF containing input to be
 *				processed, or NULL to read from input file.
 *		outbuff (O)	address of MEMBUFF returning processed
 *				input, or NULL to write to output file.
 * Returns:	(int)		new mode (PARMODE, MATHMODE, or DISPMATH),
 *				or EOF at end-of-file or for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */	
int	wp2tex ( tpfcb, inbuff, outbuff )
TPFCB	*tpfcb;
MEMBUFF	*inbuff;
MEMBUFF	*outbuff;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
/* --- processing buffers --- */
static	MEMBUFF  *tokmem0 = (MEMBUFF *)NULL; /* standard token membuff */
static	MEMBUFF *funcmem0 = (MEMBUFF *)NULL; /* standard function buffer */
static	unsigned char tokpunc[64],prevpunc[64]=""; /* trailing punctuation */
static	int nprevpunc=0;		/* trailing punc carried over */
MEMBUFF	*tokmem = (MEMBUFF *)NULL;	/* token membuff for this call */
MEMBUFF	*funcmem = (MEMBUFF *)NULL;	/* function buffer for this call */
unsigned char *tokbuff = (unsigned char *)NULL; /*token from WordPerfect doc*/
unsigned char *funcbuff = (unsigned char *)NULL; /* function after token */
int	isinbuff=0, isoutbuff=0, isbuff=0; /*true if caller provides buffer*/
int	wpchar = 0;			/* char from WP file */
int	nwhite=0, ntoken=0,npunc=0;	/* #chars in buffers */
int	modestatus = tpfcb->texmode,	/* init to current mode */
	tokmode=modestatus, funcmode = 0; /* mode of token, function */
int	nhardcr = tpfcb->nhardcr;	/* # consecutive hard cr's so far */
int	msglevel = tpfcb->msglevel;	/* output message level */
/* --- debugging info --- */
static	int ncalls=0, ndebug=0,idebug=0; /* #calls to wp2tex(), #debugs */
static	char *debugtok = (char *)NULL;	/* token for debugging */

/* --------------------------------------------------------------------------
Initialization
-------------------------------------------------------------------------- */
/* --- first-call initialization --- */
if ( ++ncalls == 1 )			/* first call to wptoken() */
  {
  /* --- get debugging token from args[0] --- */
  /* int  nargs = getvarnval(tpfcb->tpvars,"ARGS"); */ /* #args supplied */
  /* char **argptr = (char **)(getvarptr(tpfcb->tpvars,"ARGS",char)); */
  int nargs=0; char **argptr=(char **)NULL;
  if ( nargs >= 1 )			/* we only want the first arg */
    if ( argptr != (char **)NULL )	/* address check ok */
      {	debugtok = argptr[0];		/* first string */
        ndebug = (nargs<2)?1:atoi(argptr[1]); /* #strings to debug */
	printf("wptoken> debug token = \"%s\"\n",debugtok); }
  /* --- initialize standard buffers --- */
  tokmem0 = tpmalloc(-1);		/* standard token membuff */
  funcmem0 = tpmalloc(-1);		/* standard function buffer */
  } /* --- end-of-if(++ncalls==1) --- */

/* --- initialization for every entry - establish buffers --- */
if ( inbuff != (MEMBUFF *)NULL ) isinbuff=1; /* caller provided inbuff */
if ( outbuff != (MEMBUFF *)NULL ) isoutbuff=1; /* caller provided outbuff */
isbuff = (isinbuff || isoutbuff);	/* caller provided any buffer */
tokmem = tokmem0;			/* assume standard token membuff */
funcmem = funcmem0;			/* assume standard function buffer */
if ( isbuff )				/* allocate temporary buffers */
  { tokmem = tpmalloc(-1);		/* temporary token membuff */
    funcmem = tpmalloc(-1); }		/* temporary function membuff */
if (  tokmem == (MEMBUFF *)NULL		/* check buffer allocation */
||   funcmem == (MEMBUFF *)NULL )
  { if ( msglevel >= 1 )		/* report error */
      printf("wp2tex> failed to allocate memory buffer\n");
    wpchar = EOF; goto end_of_job; }	/* terminate job */
tokbuff = tokmem->buffer;		/* token from WordPerfect doc */
funcbuff = funcmem->buffer;		/* function after token */

/* --------------------------------------------------------------------------
Start next token
-------------------------------------------------------------------------- */
next_token:
*funcbuff = *tokbuff = *tokpunc = '\000'; /* nothing in buffers yet */
tokmem->bufftail = funcmem->bufftail = 0;
nwhite = ntoken = npunc = 0;
funcmode = 0;

/* --------------------------------------------------------------------------
Accumulate leading whitespace, and accumulate token till next non-ascii char.
-------------------------------------------------------------------------- */
/* --- accumulate token one character at a time --- */
while ( (wpchar = wpgetc(tpfcb,inbuff)) != EOF ) /* until EOF */
  {
  int chartype = 0;			/* character type */
  if ( wpchar<0 || wpchar>tpfcb->wpcodetblsz ) /* char out-of-range */
    { modestatus=EOF; goto end_of_job; } /* quit with error */
  chartype = tpfcb->wpcodetbl[wpchar];	/* look up character type */
  switch ( chartype )			/* process character by type */
    {
    default:				/* found trailing delimiter */
    tokbuff[ntoken] = '\000';		/* null-terminate completed token */
    tokmem->bufftail = ntoken;		/* #chars in token */
    tpfcb->tokenlen = ntoken;		/* #chars in token */
    if ( msglevel >= 99 )
      printf("wp2tex#%d> nleadwhite=%d, token=\"%s\", function=%X\n",
      ncalls,nwhite,tokbuff,(int)(wpchar));
    switch ( chartype )			/* process delimiter by type */
      {
      default: /* --- unrecognized character type --- */
	goto end_of_job;			/* can't process */
      case FUNC: /* --- single-byte function --- */
	funcmode = wpfunc(tpfcb,inbuff,wpchar,funcmem);
	break;
      case FIXLEN: /* --- fixed-length, multi-byte function --- */
	funcmode = wpfixlen(tpfcb,inbuff,wpchar,funcmem);
	break;
      case VARLEN: /* --- variable-length, multi-byte function --- */
	funcmode = wpvarlen(tpfcb,inbuff,wpchar,funcmem);
	if ( *funcbuff != SOFTCR )	/* check for converted soft page */
	  break;			/*don't have one so process usually*/
	wpchar = *funcbuff;		/* treat softpage as softcr */
      case CTRL: /* --- may need to trap SOFTCR as prevwhite --- */
	if ( wpchar == HARDCR )		/* token terminates on HARDCR */
	  (tpfcb->nhardcr)++;		/* increment hardcr count */
	if ( wpchar == SOFTCR )		/* token terminates on SOFTCR */
	 if ( ntoken > 0 )		/* token + SOFTCR */
	  { wpungetc(tpfcb,inbuff,wpchar); /*reset lead white for next token*/
	    break; }			/* and process this token */
	 else				/* leading SOFTCR before token */
	  { nwhite++;			/* just interpret SOFTCR as space */
	    goto end_of_wpchar; }	/* and process next char */
	funcmode = PARMODE;		/* reset to paragraph mode */
	*funcbuff = (unsigned char)(wpchar); /* just use control char */
	funcbuff[1] = '\000';		/* null-terminated */
	funcmem->bufftail = 1;		/* one control char in func buffer */
	break;
      } /* --- end-of-switch(chartype) --- */
      goto end_of_token;		/* token and delimiter completed */
    case WHITE: /* --- whitespace ascii chars --- */
      if ( ntoken < 1 )			/* leading white before token */
	{ nwhite++; break; }		/* accumulate white */
      else				/* trailing white after token */
	{ wpungetc(tpfcb,inbuff,wpchar); /* trailing char pushed back */
	  goto end_of_token; }		/* token accumulation completed */
    case ASCII: /* --- ascii chars accumulated as token --- */
      tokbuff[ntoken++] = (unsigned char)(wpchar); /* accumulate char */
      break;				/* read next char */
    } /* --- end-of-switch(chartype) --- */
  end_of_wpchar: ;
  } /* --- end-of-while(wpchar!=EOF) --- */
end_of_token:
  tokbuff[ntoken] = '\000';		/* null-terminate completed token */
  tokmem->bufftail = ntoken;		/* #chars in token */
  if ( !isbuff )
   { tokmode = tokenmode(tpfcb,nwhite,tokmem,funcmode); /*check token mode*/
     npunc = modifytok(tpfcb,tokmem,tokpunc,tokmode,funcmode); /*apply fixes*/
     ntoken = tokmem->bufftail; }	/* token may have been modified */
  if ( isbuff )
   { ntoken = modifykwd(tpfcb,tokmem,funcmode); } /* apply keyword fixes */
  if ( tpfcb->isdblspace )		/* check if we need second cr */
   if ( nhardcr == tpfcb->nhardcr );	/* */

/* --------------------------------------------------------------------------
Debugging
-------------------------------------------------------------------------- */
if ( debugtok != (char *)NULL )
 if ( !memcmp(debugtok,tokbuff,strlen(debugtok)) )
   idebug = ndebug;
if ( msglevel>=99 || idebug>0 )
 {printf("wp2tex#%d>nwhite=%d,token[%d/%d]=\"%s\",func[%d/%d]=\"%s\"\n",
  ncalls,nwhite, ntoken,tokmode,tokbuff, funcmem->bufftail,funcmode,funcbuff);
  idebug--;}

/* --------------------------------------------------------------------------
Process the token
-------------------------------------------------------------------------- */
if ( nwhite>0 || ntoken>0 || funcmode>0 )
  {
  if ( !(tpfcb->isheader) )		/* need header and preamble */
    if ( ntoken > 0 )			/* catch tabs,margins before header */
      texhdr(tpfcb,HEADER);		/* header before first ascii token */
  if ( nwhite>0 || (!isbuff && nprevpunc>0) )
      {	if ( !isbuff )
          { if ( tokmode == PARMODE ) modestatus = setmode(tpfcb,tokmode);
	    if ( nprevpunc > 0 ) tpputs(tpfcb,outbuff,prevpunc,-1);
	      *prevpunc = '\000'; nprevpunc=0; } /* reset prevpunc buffer */
        while ( --nwhite >= 0 )		/* blanks or \;'s */
	  if ( !isbuff && modestatus==MATHMODE && tokmode==MATHMODE )
	    tpputs(tpfcb,outbuff,"\\; ",-1); /* \;'s in mathmode */
	  else tpoutc(tpfcb,outbuff,(int)(' ')); } /* blanks in parmode */
  if ( ntoken > 0 )
      {	if ( !isbuff ) modestatus = setmode(tpfcb,tokmode);
	tpputs(tpfcb,outbuff,tokbuff,-1);
	if ( funcmode<=0 && npunc>0 )
	  {strcpy(prevpunc,tokpunc); nprevpunc=strlen(prevpunc); npunc=0;} }
  if ( funcmode > 0 )
      {	
        if ( !isbuff )
         if ( funcmode != TOKENMODE )	/* keep token mode (par or math) */
	  { if ( funcmode == DISPMATH ) tpfcb->isdblspace = 0;
	    modestatus = setmode(tpfcb,funcmode); } /* set function mode */
	if ( npunc > 0 ) tpputs(tpfcb,outbuff,tokpunc,-1);
	if ( funcmem->bufftail > 0 )
	    tpputs(tpfcb,outbuff,funcbuff,funcmem->bufftail);
	if ( funcmode == DISPMATH )	/* display math completed */
	  { tpfcb->isdblspace=1;
	    modestatus = setmode(tpfcb,PARMODE); } /* reset function mode */
      } /* --- end-of-if(funcmode>0) --- */
  } /* --- end-of-if(ntoken>0||etc) --- */

/* --------------------------------------------------------------------------
Back to caller with texmode or EOF
-------------------------------------------------------------------------- */
end_of_job:
  if ( isbuff && wpchar!=EOF ) goto next_token;
  if ( tokmem != tokmem0 ) tpfree(tokmem); /* free temporary tokmem */
  if ( funcmem != funcmem0 ) tpfree(funcmem); /* free temporary funcmem */
  return ( (wpchar==EOF)?EOF:modestatus ); /* mode, or EOF=end-of-job */
} /* --- end-of-function wp2tex() --- */


/****************************************************************************
 * Function:	tokenmode ( tpfcb, nwhite, tokmem, funcmode )
 * Purpose:	Determines mode of token string.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	File control block for WordPerfect document.
 *		nwhite (I)	int containing #whitespace chars before token.
 *		tokmem (I)	address of MEMBUFF containing token string.
 *		funcmode (I)	int containing 0 or mode of func after token.
 * Returns:	(int)		token mode (PARMODE, MATHMODE, or DISPMATH)
 *				if successful, or 0 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	tokenmode ( tpfcb, nwhite, tokmem, funcmode )
TPFCB	*tpfcb;
int	nwhite;
MEMBUFF	*tokmem;
int	funcmode;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
static	char texpar[] = "AIO.,:;()[]0123456789"; /*single-char parmode token*/
static	int ncalls = 0;			/* to detect initialization call */
static	char **mathvars = (char **)NULL; /* tokens to be treated as math... */
static	int nmathvars = 0;		/* ...set during initialization */
unsigned char *token = tokmem->buffer;	/* token buffer in tokmem struct */
int	ntoken = tokmem->bufftail;	/* #chars in token */
char	*delim, *strpbrk(), *precat(), lastchar; /* reserved chars in token */
int	is1letwd=0, is1punc=0,ispunc=0;	/*is 1st char 1letwd, 1st,last punc*/
int	inotdigit /*, strspn()*/;	/* offset of 1st char not in set */
unsigned char *tokptr = token;		/* ptr to token buffer */
int	modestatus = tpfcb->texmode,	/* init to current mode */
	tokmode = modestatus;		/* default mode of token = current */
int	msglevel = tpfcb->msglevel;	/* output message level */

/* --------------------------------------------------------------------------
Determine mode of token
-------------------------------------------------------------------------- */
/* --- get address of mathvars array during initialization --- */
ncalls++;				/* bump call counter */
#if 0
if ( ncalls == 1 )			/* first call */
  { nmathvars = getvarnval(tpfcb->tpvars,"MATHVARS"); /*#mathvars from user*/
    mathvars = (char **)(getvarptr(tpfcb->tpvars,"MATHVARS",char)); }
#endif
if ( msglevel >= 99 )
  printf("tokenmode#%d> nleadwhite=%d, token[%d]=\"%s\", funcmode=%d\n",
  ncalls,nwhite,ntoken,token,funcmode);
/* --- default=current mode, or determine mode of token string ---  */
if ( ntoken > 0 )			/* we have a token to check */
  {
  tokmode = 0;				/* reset token mode */
  if ( ntoken > 1 )			/* two-or-more chars in token */
   if( strspn(token,LETTERS) == ntoken ) /*token entirely lower-case letters*/
    goto got_mode;			/* so just set paragraph mode */
  lastchar = token[ntoken-1];		/* last char of token */
  is1punc = ismychar(TEXPUNC,*token);	/* is first char a punc */
  ispunc = ismychar(TEXPUNC,lastchar);	/* or token followed by punctuation */
  is1letwd = (is1punc)?			/* check 2nd letter if 1st quote */
  	ismychar(TEX1LETWDS,token[1]):	/* 2nd char=1-letter wd */
  	ismychar(TEX1LETWDS,*token);	/* else 1st char=1-letter wd */
  inotdigit = strspn(token,DIGITS);	/* index/offset to first non-digit */
  /* --- check for explicit math mode char --- */
  if ( (delim = strpbrk(token,TEXMATH))	/* look for explicit math mode char */
  != (char *)NULL )			/* found one */
      { tokmode = MATHMODE; goto got_mode; } /* set mode and stop looking */
  /* --- check for char followed by digit --- */
  if ( inotdigit == 0 )			/* first char non-digit */
    if ( (delim = strpbrk(token,DIGITS)) /* find first digit */
    !=    (char *)NULL )		/* digit after 1st char */
      { tokmode = MATHMODE; goto got_mode; } /* set mode and stop looking */
  /* --- check for number (possibly followed by punctuation) --- */
  if ( inotdigit==ntoken		/* string is all digits */
  ||   (ispunc && inotdigit==(ntoken-1)) ) /* or digits+punc */
    { if ( modestatus != PARMODE ) tokmode = modestatus; /* previous mode */
      else if ( funcmode!=0 && funcmode!=PARMODE
	     && funcmode!=TOKENMODE ) tokmode = funcmode;
      else tokmode = PARMODE;
      goto got_mode; }			/* mode determined */
  /* --- check for single char --- */
  if ( ntoken == 1 )			/* single char */
    { tokmode = (ismychar(texpar,toupper(*token)))? PARMODE : MATHMODE;
      goto got_mode; }			/* mode determined */
  /* --- check for single char followed by period,comma,prime,etc --- */
  if ( ispunc )				/* got a punctuation delimiter */
    if ( !is1letwd )			/* not after 1-letter word */
      if ( ntoken == 2			/* single char followed by punc */
      ||   (ntoken==3 && is1punc) )	/* or surrounded by puncs */
        { tokmode = MATHMODE; goto got_mode; } /* must be mathmode */
  /* --- check for single-letter function followed by ( --- */
  if ( ntoken >= 2 )			/* single char followed by ( */
    if ( !is1letwd )			/* not after 1-letter word */
      if ( token[1]=='(' || token[2]=='(' ) /* got a function */
        if ( token[1] == '('		/* got a function */
        ||   (token[2]=='(' && ismychar(TEXPUNC,token[1])) ) /* primed func */
          { tokmode = MATHMODE; goto got_mode; } /* must be mathmode */
  } /* --- end-of-if(ntoken>0) --- */
/* --- check for subscripts, etc --- */
got_mode:				/* token mode determined */
if ( tokmode == 0 )			/* oops...mode not determined */
    tokmode = PARMODE;			/* must be just a word */
if ( tokmode == PARMODE )		/* we think token is just a word */
  { if ( funcmode!=0 && funcmode!=PARMODE /* but it's followed by math mode */
      && funcmode!=TOKENMODE ) tokmode = funcmode; /*must be part of exprsn*/
    if ( modestatus!=PARMODE && nwhite==0 ) /* immediately preceded by math */
      tokmode = modestatus; }		/* so it must be part of expression */
/* --- check for special mathvars --- */
if ( tokmode == PARMODE )		/* only necessary if in parmode */
  if ( nmathvars > 0 )			/* and if we have some mathvars */
    {
    int	imathvar=0, mathlen=0;		/* mathvars index, string length */
    char *amathvar = (char *)NULL;	/* mathvar string itself */
    for ( imathvar=0; imathvar<nmathvars; imathvar++ )
      {	amathvar = mathvars[imathvar];	/* dereference mathvars array */
	mathlen = strlen(amathvar);	/* length of this var */
	if (!ispunc && (ntoken!=mathlen) ) continue; /* check lengths */
	if (ispunc && ((ntoken-1)!=mathlen) ) continue; /* token plus punc */
	if ( !memcmp(token,amathvar,mathlen) ) /*lengths match, check token*/
	  { tokmode = MATHMODE; break; } /* found a mathmode token */
      } /* --- end-of-for(imathvar) --- */
    } /* --- end-of-if(mathvars>0) --- */
if ( msglevel >= 99 )
  printf("tokmode#%d> tokmode=%d\n",
  ncalls,tokmode);
return ( tokmode );
} /* --- end-of-function tokenmode() --- */


/****************************************************************************
 * Function:	modifytok ( tpfcb, tokmem, punc, tokmode, funcmode )
 * Purpose:	Makes a variety of required TeX "fixes" to token.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	File control block for WordPerfect document.
 *		tokmem (I/O)	address of MEMBUFF containing token string.
 *		punc (O)	address of char returning trailing punc.
 *		tokmode (I)	int containing 0 or mode of token.
 *		funcmode (I)	int containing 0 or mode of func after token.
 * Returns:	(int)		length of trailing punctuation if successful,
 *				or -1 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	modifytok ( tpfcb, tokmem, punc, tokmode, funcmode )
TPFCB	*tpfcb;
MEMBUFF	*tokmem;
char	*punc;
int	tokmode;
int	funcmode;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
static	int ncalls = 0;			/* call entry number */
int	npunc = 0;			/* #punc chars returned to caller */
unsigned char *token = tokmem->buffer;	/* token string within membuff */
int	ntoken = tokmem->bufftail;	/* #chars initially in token */
int	isrquote=0;			/*don't strip trailing right quote*/
char	*delim, *strpbrk(), *precat(), lastchar; /* reserved chars in token */
int	ispunc = 0;			/* true if lastchar is punctuation */
unsigned char *tokptr = token;		/* ptr to token buffer */
int	msglevel = tpfcb->msglevel;	/* output message level */

/* --------------------------------------------------------------------------
Special token preprocessing
-------------------------------------------------------------------------- */
ncalls++;				/* bump call number */
if ( msglevel >= 99 )
  printf("modifytok#%d> token[%d]=\"%s\", tokmode=%d, funcmode=%d\n",
  ncalls,ntoken,token,tokmode,funcmode);
*punc = '\000';				/* no trailing puntuation yet */
if ( ntoken > 0 )			/* we have a token to check */
  {
  /* ------------------------------------------------------------------------
  convert & to \&, etc (i.e., for all chars in texbackslash[])
  ------------------------------------------------------------------------ */
  tokptr = token;			/* start at beginning of token */
  lastchar = token[ntoken-1];		/* last char of token */
  ispunc = ismychar(TEXPUNC,lastchar);	/* token followed by punctuation */
  while ( (delim = strpbrk(tokptr,TEXBACKSLASH)) /* find chars needing \ */
  !=      (char *)NULL )		/* until there are no more */
    { precat(delim,"\\");		/* pre-concat backslash */
      tokptr = delim+2; }		/* move past \ and past char */
  if ( tokptr != token )		/* added some backslashes */
    { ntoken = strlen(token);		/* set new token length */
      lastchar = (ntoken>0)?token[ntoken-1]:'\000'; } /*last char of token*/
  /* ------------------------------------------------------------------------
  convert leading/trailing " quotes to `` and ''
  ------------------------------------------------------------------------ */
  if ( ntoken > 1 )			/* check for leading/trailing " */
    {
    if ( *token == '\"' )		/* got an opening quote */
      {	strcpy(token,token+1);		/* squeeze out opening quote */
	precat(token,TEXLQUOTE);	/* replace by tex opening quotes */
	ntoken = strlen(token); }	/* new token length */
    if ( lastchar == '\"' )		/* got a closing quote */
      {	if ( tokmode == PARMODE )	/* leave math mode closing " alone */
	  strcpy(token+ntoken-1,TEXRQUOTE); /* replace by tex quotes */
	isrquote = 1; }			/* set flag for right quote */
      else
	if ( ispunc && token[ntoken-2]=='\"' ) /*closing quote before punc*/
	  { strcpy(token+ntoken-2,TEXRQUOTE); /* replace by tex quotes */
	    ntoken = strlen(token);	/* new length */
	    token[ntoken++] = lastchar; /* replace last character */
	    token[ntoken] = '\000'; }	/* and null */
    ntoken = strlen(token);		/* set new token length */
    lastchar = (ntoken>0)?token[ntoken-1]:'\000'; /* last char of token */
    if ( lastchar == '\'' ) isrquote=1;	/* leave closing ' with token */
    } /* --- end-of-if(tokmode==parmode) --- */
  /* ------------------------------------------------------------------------
  change to parmode before trailing punctuation
  ------------------------------------------------------------------------ */
  if ( ispunc && !isrquote )		/* got trailing punc, but not quote */
   if ( tokmode != PARMODE )		/* may want to strip trailing punc */
    if ( funcmode==0 || funcmode==PARMODE ) /* and we may gain something */
      {	punc[0]=lastchar; punc[1]='\000'; npunc=1; /* set punc buffer */
	token[--ntoken] = '\000'; }	/* and strip punc from token */
  tokmem->bufftail = ntoken;		/* #chars in modified token */
  } /* --- end-of-if(ntoken>0) --- */
end_of_job:
  if ( msglevel >= 99 )
    printf("modifytok#%d> token[%d]=\"%s\", punc[%d]=%s\n",
    ncalls,ntoken,token,npunc,punc);
  return ( npunc );			/* punc length or -1 to caller */
} /* --- end-of-function modifytok() --- */


/****************************************************************************
 * Function:	modifykwd ( tpfcb, tokmem, funcmode )
 * Purpose:	Translates WordPerfect Equation Editor keywords to TeX
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	File control block for WordPerfect document.
 *		tokmem (I/O)	address of MEMBUFF containing token string.
 *		funcmode (I)	int containing 0 or mode of func after token.
 * Returns:	(int)		length of translated token if successful,
 *				or -1 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	modifykwd ( tpfcb, tokmem, funcmode )
TPFCB	*tpfcb;
MEMBUFF	*tokmem;
int	funcmode;
{
/* --------------------------------------------------------------------------
Replace Equation Editor keywords with TeX keyowrds
-------------------------------------------------------------------------- */
int	ntoken = tokmem->bufftail;	/* current token length */
(tokmem->buffer)[ntoken] = '\000';	/*make sure token is null-terminated*/
streplace(tokmem->buffer,eqnkwds,texkwds,3); /* make replacements */
tokmem->bufftail = ntoken = strlen(tokmem->buffer); /* new token length */
return ( ntoken );			/* back to caller */
} /* --- end-of-function modifykwd() --- */


/****************************************************************************
 * Function:	texhdr ( tpfcb, idisplay )
 * Purpose:	Display the TeX header and Preamble, or the Trailer.
 * --------------------------------------------------------------------------
 * Arguments:	tpfcb (I)	File control block for WordPerfect document.
 *		idisplay (I)	int containing HEADER or TRAILER.
 * Returns:	(int)		1 if successful, or EOF for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	texhdr ( tpfcb, idisplay )
TPFCB	*tpfcb;
int	idisplay;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	status = EOF;			/* status retruned to caller */
FILE	*fopen();			/* to open user's header file */
char	*fgets();			/* to read user's header file */
/*char	*varlambda();*/			/* replace args in hdr with tpvars */
/*GETVARPTR(tpfcb->tpvars,"HDRFILE",char,hdrfile);*/ /*header file name*/
char	*hdrfile = "\000";		/* no non-default header file */
/* --------------------------------------------------------------------------
Write header,preamble to output file at first ascii char
-------------------------------------------------------------------------- */
if ( idisplay == HEADER ) tpfcb->isheader = 1; /* set header flag */
tpfcb->isdblspace = 0;			/* dblspace off for header/trailer */
if ( idisplay == HEADER			/* user file for header only */
&&   strlen(hdrfile) > 0 )		/* user-specified ascii header file */
	{ /* --- copy user header to output TeX file --- */
	FILE *hdrptr=fopen(hdrfile,"r"); /* open user's ascii header file */
	FILE *texptr=tpfcb->tpfile.fp;	/* TeX output file already opened */
	unsigned char hdrline[256] = "\000"; /* buffer for header line */
	if ( texptr == (FILE *)NULL ) texptr = stdout; /* write to stdout */
	if ( hdrptr == (FILE *)NULL )	/* non-existent user file */
	 tpfcb->isheader = 999;		/* signals no header, no trailer */
	else				/* copy header, substituting args */
	 { while( fgets(hdrline,255,hdrptr) != (char *)NULL )/*read till EOF*/
	    /*if ( fputs(varlambda(tpfcb->tpvars,hdrline),texptr)*/
	    if ( fputs(hdrline,texptr)	/*write hdr*/
	    !=   0 ) goto end_of_job;	/* quit on error */
	   fclose(hdrptr); }		/* close ehader file */
	} /* --- end-of-copy-user-header-file --- */
else
	{ /* --- generate default header/trailer for TeX type --- */
	int iheader = 0;		/* index for header/preamble lines */
	char *cmdstr = (char *)NULL;	/*tex header/Preamble command string*/
	while ( (cmdstr = texcommand(tpfcb,idisplay,iheader)) /* get line */
	!= (char *)NULL )		/* check for end-of-header/trailer */
	  { tpputs(tpfcb,NULL,cmdstr,-1); /* write the line */
	    tpputc(tpfcb,HARDCR); iheader++; } /* and a carriage return */
	iheader = 0;			/* reset for preamble */
	if ( idisplay == HEADER )	/* no preamble for trailer */
	 while ( (cmdstr = texcommand(tpfcb,PREAMBLE,iheader)) /* preamble */
	 != (char *)NULL )		/* check for end-of-preamble */
	  { tpputs(tpfcb,NULL,cmdstr,-1); /* write the line */
	    tpputc(tpfcb,HARDCR); iheader++; } /* and a carriage return */
	} /* --- end-of-if/else(strlen(hdrfile)>0) --- */
tpfcb->isdblspace = 1;			/* turn on paragraph double space */
status = 1;				/* set ok status before returning */
end_of_job: return ( status );		/* 1, or EOF for any error */
} /* --- end-of-function texhdr() --- */


/****************************************************************************
 * Function:	precat ( target, source )
 * Purpose:	Pre-concatanates source at the beginning of target.
 * --------------------------------------------------------------------------
 * Arguments:	target (I)	address of char string at whose beginning
 *				source will be pre-concatanated.
 *		source (I)	address of char string to be placed at the
 *				beginning of target.
 * Returns:	(char *)	target, as supplied by caller,
 *				or NULL for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
char	*precat( target, source )
char	*target;
char	*source;
{
/* --------------------------------------------------------------------------
Pre-concatanate source at the beginning of target
-------------------------------------------------------------------------- */
static	unsigned char buffer[1024] = "\000"; /* buffer to save target */
strcpy(buffer,target);			/* save target */
strcpy(target,source);			/* begin with source */
strcat(target,buffer);			/* follow with target */
return(target);				/* back to caller */
} /* --- end-of-function precat() --- */


/****************************************************************************
 * Function:	vindex ( source, targets )
 * Purpose:	Searches a source string for (the first occurrence of) any
 *		one of several target strings.
 * --------------------------------------------------------------------------
 * Arguments:	source (I/O)	addr(addr of char string) containing
 *				the string to be searched, and returning
 *				a char string ptr to the first byte past
 *				the occurrence of target in source.
 *		targets (I)	addr of array of char strings, for any of
 *				which source will be searched.
 * Returns:	(int)		index+1 into targets array for matched target
 *				string found in source, or 0 if no match.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	 vindex ( source, targets )
char	 **source;
char	 **targets;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
#define MAXTARGET 128			/* max number of target strings */
#define	tocase(x) ((iscase)?(x):(toupper((x)))) /* case-sensitivity check */
int	iscase = 0;			/* true for case-sensitivity */
static	 char *tptr[MAXTARGET]; 	/* ptrs to current target chars */
char	 *sptr; 			/* ptr to current source char */
int	 itptr; 			/* index of current target */
char	 **atptr;			/* addr of tptr[itptr] */

/* --------------------------------------------------------------------------
Initialization - start compares at beginning of each target string.
-------------------------------------------------------------------------- */
for ( itptr=0, atptr=tptr; itptr<MAXTARGET; itptr++, atptr++ )
    { /* --- init local ptrs to beginning of each target --- */
    if ( (*atptr = targets[itptr])	/* local copy of itptr-th ptr */
    ==   (char *)NULL ) break;		/* null ptr is one trailer */
    if ( **atptr == '\0' )		/* null string is the second, */
	 { *atptr = (char *)NULL; break; } /* so set null ptr and break */
    } tptr[MAXTARGET-1] = (char *)NULL;	/* always guarantee a trailer */

/* --------------------------------------------------------------------------
Search for any target string somewheres in source.
-------------------------------------------------------------------------- */
for (sptr = *source; *sptr != '\0'; sptr++) /* for each char from source */
    { /* ---compare current char of every target to current source char--- */
    for ( itptr=0, atptr=tptr; *atptr != (char *)NULL; itptr++, atptr++ )
	 { /* --- match current target char against source --- */
	 if ( tocase(**atptr) != tocase(*sptr) ) /* usually fail to match */
	      { /* --- so start at beginning of target again --- */
	      *atptr = targets[itptr];	/* reset the pointer */
	      if(tocase(**atptr)!=tocase(*sptr))continue; } /*check 1st char*/
	 /* ---matched current char of current target to current source--- */
	 if ( *(++(*atptr))		/* bump target one char */
	 ==   '\0' )			/* matched complete target */
	       { *source = sptr+1; return (itptr+1); } /* so go home */
	 } /* --- try the next target against current source --- */
    } /* --- try the next source char against all targets --- */
return ( 0 );				    /* failed to find any target */
} /* --- end-of-function vindex() --- */


/****************************************************************************
 * Function:	streplace ( source, targets, replacements, blanks )
 * Purpose:	Replaces all occurrences of targets in source
 *		with the corresponding replacement strings.
 * --------------------------------------------------------------------------
 * Arguments:	source (I/O)	addr of char string containing targets
 *				to be replaced with replacements.
 *		targets (I)	addr of array of char strings, for any of
 *				which source will be searched and replaced
 *				with the corresponding replacement.
 *		replacements(I)	addr of array of char strings, to replace
 *				any of the corresponding targets in source.
 *		blanks (I)	int containing 1 to remove trailing blanks,
 *				2 to remove leading blanks, 3 to remove both,
 *				0 to leave alone.
 * Returns:	(int)		number of replacements made, 0 if none
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	 streplace ( source, targets, replacements, blanks )
char	 *source;
char	 **targets;
char	 **replacements;
int	blanks;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
char	*psource = source;		/* pointer to source string */
int	nreplace = 0;			/* number of replacements */
int	itarget = 0;			/* target[] index found in source */
int	tarlen=0, replen=0;		/* length of target, replacement */
int	islead = (blanks==2 || blanks==3), /* remove leading blanks */
	istrail= (blanks==1 || blanks==3); /* remove trailing blanks */
/* --------------------------------------------------------------------------
Replace one at a time
-------------------------------------------------------------------------- */
while ( (itarget=vindex(&psource,targets)) > 0 ) /* still finding targets */
  {
  itarget--;				/* change to array index */
  tarlen = strlen(targets[itarget]);	/* length of target string matched */
  replen = strlen(replacements[itarget]); /* length of replacement string */
  psource -= tarlen;			/* ptr to 1st char of target */
  if ( islead )				/* leading blanks not wanted */
    while ( psource>source && *(psource-1)==' ' ) /* got a leading blank */
      { psource--; tarlen++; }		/* get rid of it */
  if ( istrail )			/* trailing blanks not wanted */
    while ( *(psource+tarlen) == ' ' ) tarlen++;
  strcpy(psource,psource+tarlen);	/* squeeze out target */
  precat(psource,replacements[itarget]); /* insert replacement */
  psource += replen;			/* pick up search after replacement */
  } /* --- end-of-while(itarget>0) --- */
return ( nreplace );			/* #replacements back to caller */
} /* --- end-of-function streplace() --- */
/* ------------------------- END-OF-FILE WP2TEX.C ------------------------ */
