#include <errno.h>
#include <sys/doscalls.h>

#define N_DOS_ERRORS 0x5A

#define RESERVED  EIO

static unsigned char dos_errors[N_DOS_ERRORS + 1] =
{
    0,			    /* 0x00  no error								 */
    EINVAL,		    /* 0x01  function number invalid				 */
    ENOENT,		    /* 0x02  file not found						 */
    ENOENT,		    /* 0x03  path not found						 */
    EMFILE,		    /* 0x04  too many open files					 */
    EACCES,		    /* 0x05  access denied						 */
    EBADF,		    /* 0x06  invalid handle						 */
    EIO,		    /* 0x07  memory control block destroyed		 */
    ENOMEM,		    /* 0x08  insufficient memory					 */
    EIO,		    /* 0x09  memory block address invalid			 */
    EINVAL,		    /* 0x0A  environment invalid					 */
    ENOEXEC,		    /* 0x0B  format invalid						 */
    EINVAL,		    /* 0x0C  access code invalid					 */
    EINVAL,		    /* 0x0D  data invalid							 */
    RESERVED,		    /* 0x0E  reserved								 */
    EACCES,		    /* 0x0F  invalid drive						 */
    EACCES,		    /* 0x10  attempted to remove current directory   */
    EXDEV,		    /* 0x11  not same device						 */
    ENOENT,		    /* 0x12  no more files						 */
    EIO,		    /* 0x13  disk write-protected					 */
    EIO,		    /* 0x14  unknown unit							 */
    EIO,		    /* 0x15  drive not ready						 */
    EIO,		    /* 0x16  unknown command						 */
    EIO,		    /* 0x17  data error 							 */
    EIO,		    /* 0x18  bad request structure length			 */
    EIO,		    /* 0x19  seek error 							 */
    EIO,		    /* 0x1A  unknown media type 					 */
    EIO,		    /* 0x1B  sector not found						 */
    ENOSPC,		    /* 0x1C  printer out of paper					 */
    EIO,		    /* 0x1D  write fault							 */
    EIO,		    /* 0x1E  read fault 							 */
    EIO,		    /* 0x1F  general failure						 */
    EACCES,		    /* 0x20  sharing violation						 */
    EACCES,		    /* 0x21  lock violation						 */
    EIO,		    /* 0x22  disk change invalid					 */
    EIO,		    /* 0x23  FCB unavailable						 */
    EIO,		    /* 0x24  sharing buffer overflow				 */
    EIO,		    /* 0x25  code page mismatch 					 */
    EIO,		    /* 0x26  cannot complete file operation		 */
    ENOSPC,		    /* 0x27  insufficient disk space				 */
    RESERVED,		    /* 0x28  reserved								 */
    RESERVED,		    /* 0x29  reserved								 */
    RESERVED,		    /* 0x2A  reserved								 */
    RESERVED,		    /* 0x2B  reserved								 */
    RESERVED,		    /* 0x2C  reserved								 */
    RESERVED,		    /* 0x2D  reserved								 */
    RESERVED,		    /* 0x2E  reserved								 */
    RESERVED,		    /* 0x2F  reserved								 */
    RESERVED,		    /* 0x30  reserved								 */
    RESERVED,		    /* 0x31  reserved								 */
    EIO,		    /* 0x32  network request not supported		 */
    EIO,		    /* 0x33  remote computer not listening		 */
    EIO,		    /* 0x34  duplicate name on network				 */
    EIO,		    /* 0x35  network name not found				 */
    EIO,		    /* 0x36  network busy							 */
    EIO,		    /* 0x37  network device no longer exists		 */
    EIO,		    /* 0x38  network BIOS command limit exceeded	 */
    EIO,		    /* 0x39  network adapter hardware error		 */
    EIO,		    /* 0x3A  incorrect response from network		 */
    EIO,		    /* 0x3B  unexpected network error				 */
    EIO,		    /* 0x3C  incompatible remote adapter			 */
    ENOSPC,		    /* 0x3D  print queue full						 */
    ENOSPC,		    /* 0x3E  queue not full						 */
    EIO,		    /* 0x3F  not enough space to print file		 */
    EIO,		    /* 0x40  network name was deleted				 */
    EACCES,		    /* 0x41  network: Access denied				 */
    EIO,		    /* 0x42  network device type incorrect		 */
    EIO,		    /* 0x43  network name not found				 */
    EIO,		    /* 0x44  network name limit exceeded			 */
    EIO,		    /* 0x45  network BIOS session limit exceeded	 */
    EIO,		    /* 0x46  temporarily paused 					 */
    EIO,		    /* 0x47  network request not accepted			 */
    EIO,		    /* 0x48  network print/disk redirection paused   */
    EACCES,		    /* 0x49  invalid network version				 */
    EIO,		    /* 0x4A  account expired						 */
    EIO,		    /* 0x4B  password expired						 */
    EIO,		    /* 0x4C  login attempt invalid at this time 	 */
    EIO,		    /* 0x4D  disk limit exceeded on network node	 */
    EIO,		    /* 0x4E  not logged in to network node		 */
    RESERVED,		    /* 0x4F  reserved								 */
    EEXIST,		    /* 0x50  file exists							 */
    RESERVED,		    /* 0x51  reserved								 */
    ENOENT,		    /* 0x52  cannot make directory				 */
    EIO,		    /* 0x53  fail on INT 24h						 */
    EIO,		    /* 0x54  too many redirections				 */
    EIO,		    /* 0x55  duplicate redirection				 */
    EIO,		    /* 0x56  invalid password						 */
    EINVAL,		    /* 0x57  invalid parameter						 */
    EIO,		    /* 0x58  network write fault					 */
    EIO,		    /* 0x59  function not supported on network		 */
    EIO 		    /* 0x5A  required system component not installed */
};

void _sys_doserror2errno(unsigned short error_dos)
{
    if (error_dos >= N_DOS_ERRORS)
	errno = EIO;
    else
	errno = dos_errors[error_dos];
}

static unsigned short GetDS(void)
{
    register unsigned short v;
    __asm__("movw %%ds, %0 \n\t"
	    : "=r" ((unsigned short) v) );
    return v;
}
static unsigned short GetES(void)
{
    register unsigned short v;
    __asm__("movw %%es, %0 \n\t"
	    : "=r" ((unsigned short) v) );
    return v;
}

/* ********** DISK OPERATIONS ********** */

/*
** AH = 0x0E
** DL = drive (a=0,b=1,c=2)
** return: AL max drive
*/
int dos_setdrive(unsigned char drive)
{
    struct REGPACK r;

    r.eax = 0x0E00;
    r.edx = drive;

    _intr(0x21, &r);

    return (r.eax & 0xFF);
}

/*
** AH = 0x19
** return: AL this drive (a=0,b=1,c=2)
*/
unsigned char dos_getdrive(void)
{
    struct REGPACK r;

    r.eax = 0x1900;

    _intr(0x21, &r);

    return (r.eax & 0xFF);
}

/* ********** DATE/TIME OPERATIONS ********** */

/*
** AH = 0x2A
** return: CX=year DX=month/day
*/
void dos_getdate(struct dos_date * dd)
{
    struct REGPACK r;

    r.eax = 0x2A00;

    _intr(0x21, &r);

    dd->ddate_year = (unsigned short) r.ecx;
    dd->ddate_month = (unsigned char) ((unsigned short) r.edx >> 8);
    dd->ddate_day = (unsigned char) r.edx;
    dd->ddate_dayofweek = (unsigned char) r.eax;
}

/*
** AH = 0x2B
** CX = year
** DX = month/day
*/
int dos_setdate(struct dos_date * dd)
{
    struct REGPACK r;

    r.eax = 0x2B00;
    r.ecx = dd->ddate_year;
    r.edx = ((unsigned short) dd->ddate_month << 8) | dd->ddate_day;

    _intr(0x21, &r);

    if ((r.eax & 0xFF) == 0xFF)
	return -1;
    else
	return 0;
}

/*
** AH = 0x2C
** return CX=hour/min DX=sec/hsec
*/
void dos_gettime(struct dos_time * dt)
{
    struct REGPACK r;

    r.eax = 0x2C00;

    _intr(0x21, &r);

    dt->dtime_hour = (unsigned char) ((unsigned short) r.ecx >> 8);
    dt->dtime_minutes = (unsigned char) r.ecx;
    dt->dtime_seconds = (unsigned char) ((unsigned short) r.edx >> 8);
    dt->dtime_hsec = (unsigned char) r.edx;
}

/*
** AH = 0x2D
** CX = hour/min
** DX = sec/hsec
*/
int dos_settime(struct dos_time * dt)
{
    struct REGPACK r;

    r.eax = 0x2D00;
    r.ecx = ((unsigned short) dt->dtime_hour << 8) | dt->dtime_minutes;
    r.edx = ((unsigned short) dt->dtime_seconds << 8) | dt->dtime_hsec;

    _intr(0x21, &r);

    if ((r.eax & 0xFF) == 0xFF)
	return -1;
    else
	return 0;
}

/* ********** DIRECTORY OPERATIONS ********** */

/*
** AH = 0x39
** DS:DX name (64 bytes)
*/
int dos_mkdir(const char *name)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x3900;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (0);
}

/*
** AH = 0x3A
** DS:DX name (64 bytes)
*/
int dos_rmdir(const char *name)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x3A00;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (0);
}

/*
** AH = 0x3B
** DS:DX name (64 bytes)
*/
int dos_chdir(const char *name)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x3B00;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (0);
}

/*
** AH = 0x41
** DS:DX name (64 bytes)
*/
int dos_remove(const char *name)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x4100;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (0);
}

/*
** AH = 0x43 ; AL = 0
** DS:DX name
** return CX: attr
*/
int dos_getfattr(const char *name, int * attr)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x4300;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else {
	*attr = r.ecx & 0xFFFF;
	return 0;
    }
}

int dos_access(const char *name, int mode)
{
    int attr;

    if (dos_getfattr(name, &attr) == -1)
	return -1;
    if ((attr & 1) && (mode & 2))	/* RDONLY and try WRITE access */
	return -1;
    else
	return 0;
}

/*
** AH = 0x43 ; AL = 1
** CX = attr
** DS:DX name
*/
int dos_setfattr(const char *name, int attr)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x4301;
    r.ecx = attr;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.eax & 0xFFFF);
}

/*
** AH = 0x47
** DS:SI name
*/
int dos_getcwd(unsigned char drive, char *name)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x4700;
    r.edx = drive;

    SET_SEG_OFF(name, sr.ds, r.esi);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (0);
}

/*
** AH = 0x4E
** CX = attr
** DS:DX name
*/
int dos_findfirst(const char *name, int attr)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x4E00;
    r.ecx = attr;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (0);
}

/*
** AH = 0x4F
*/
int dos_findnext(void)
{
    struct REGPACK r;

    r.eax = 0x4F00;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (0);
}

/*
** AH = 0x1A
** DS:DX dta address
*/
void dos_setdta(void *buf)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x1A00;
    SET_SEG_OFF(buf, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);
}

/*
** AH = 0x56
** DS:DX name (64 bytes)
** ES:DI name (64 bytes)
*/
int dos_rename(const char *oldname, const char *newname)
{
    struct REGPACK r;
    struct SEGPACK sr;

    r.eax = 0x5600;
    SET_SEG_OFF(oldname, sr.ds, r.edx);
    SET_SEG_OFF(newname, sr.es, r.edi);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (0);
}

/* ********** FILE HANDLE OPERATIONS ********** */

/*
** AH = 0x3C
** CX = attr
** DS:DX = name
** return: AX fileno
*/
int dos_creat(const char *name, int attr)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x3C00;
    r.ecx = attr;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.eax & 0xFFFF);
}

/*
** AH = 0x3D
** AL = mode
** DS:DX = name
** return: AX fileno
*/
int dos_open(const char *name, int modes)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x3D00 | (char) modes;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.eax & 0xFFFF);
}

/*
** AH = 0x3E
** BX = file handle
** return: -1 on error
*/
int dos_close(int handle)
{
    struct REGPACK r;

    r.eax = 0x3E00;
    r.ebx = handle;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return 0;
}

/*
** AH = 0x3F
** BX = handle
** CX = bytes
** DS:DX = buf
** return: AX bytes
*/
int dos_read(int handle, void *buf, unsigned bytes)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x3F00;
    r.ebx = handle;
    r.ecx = bytes;
    SET_SEG_OFF(buf, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.eax & 0xFFFF);
}

/*
** AH = 0x40
** BX = handle
** CX = bytes
** DS:DX = buf
** return: AX bytes
*/
int dos_write(int handle, const void *buf, unsigned bytes)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x4000;
    r.ebx = handle;
    r.ecx = bytes;
    SET_SEG_OFF(buf, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.eax & 0xFFFF);
}

/*
** AH = 0x42
** AL = orgin
** BX = handle
** CX:DX = pos
** return: new pos
*/
long dos_lseek(int handle, long offset, int orgin)
{
    struct REGPACK r;

    r.eax = 0x4200 | (char) orgin;
    r.ebx = handle;
    r.ecx = offset >> 16;
    r.edx = offset & 0xFFFF;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1L);
    }
    else
	return (r.edx << 16) | (r.eax & 0xFFFF);
}

/*
** AH = 0x45
** BX = handle
** return: AX new handle
*/
int dos_dup(int handle)
{
    struct REGPACK r;

    r.eax = 0x4500;
    r.ebx = handle;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.eax & 0xFFFF);
}

/*
** AH = 0x46
** BX = handle
** CX = new handle
** return: -1 on error
*/
int dos_dup2(int handle, int newhandle)
{
    struct REGPACK r;

    r.eax = 0x4600;
    r.ebx = handle;
    r.ecx = newhandle;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (newhandle);
}

/*
** AX = 0x5700
** BX = handle
** return: CX:DX
*/
int dos_getftime(int handle, unsigned short * date, unsigned short * time)
{
    struct REGPACK r;

    r.eax = 0x5700;
    r.ebx = handle;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else {
	*date = r.edx & 0xFFFF;
	*time = r.ecx & 0xFFFF;
	return 0;
    }
}

/*
** AX = 0x5701
** BX = handle
** CX:DX = time/date
*/
int dos_setftime(int handle, unsigned short date, unsigned short time)
{
    struct REGPACK r;

    r.eax = 0x5701;
    r.ebx = handle;
    r.ecx = time;
    r.edx = date;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.eax & 0xFFFF);
}

/*
** AH = 0x5B
** CX = attr
** DS:DX = name
** return: AX fileno
*/
int dos_creatnew(const char *name, int attr)
{
    struct REGPACK r;
    struct SEGPACK sr = { GetDS(), GetES()};

    r.eax = 0x5B00;
    r.ecx = attr;
    SET_SEG_OFF(name, sr.ds, r.edx);

    _intxr(0x21, &r, &sr);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.eax & 0xFFFF);
}


/* ********** IOCTL OPERATIONS ********** */

int dos_ioctl(int handle, int code)
{
    struct REGPACK r;

    r.eax = 0x4400 | (code & 0xFF);
    r.ebx = handle;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.edx & 0xFFFF);
}

/*
** AH = 0x44 ; AL = 0
** BX = handle
*/
int dos_ioctl_getattr(int handle)
{
    struct REGPACK r;

    r.eax = 0x4400;
    r.ebx = handle;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (r.edx & 0xFFFF);
}

int dos_isatty(int handle)
{
    int i = dos_ioctl_getattr(handle);

    if (i == -1)
	return 0;
    else
	return i & 128;
}

/*
** AH = 0x44 ; AL = 01
** BX = handle
** DX = mode
*/
int dos_ioctl_setattr(int handle, int mode)
{
    struct REGPACK r;

    r.eax = 0x4401;
    r.ebx = handle;
    r.edx = mode;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return 0;
}

/*
** AH = 0x44 ; AL = 06
** BX = handle
*/
int dos_ioctl_select_in(int handle)
{
    struct REGPACK r;

    r.eax = 0x4406;
    r.ebx = handle;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return ((unsigned char)r.eax == 0xFF) ? 1 : 0;
}

/*
** AH = 0x44 ; AL = 09
** BL = drive
** ret: 1/0 remote or -1
*/
int dos_ioctl_remotedrive(unsigned char drive)
{
    struct REGPACK r;

    r.eax = 0x4409;
    r.ebx = drive;

    _intr(0x21, &r);

    if (r.eflags & 1) {
	_sys_doserror2errno( r.eax & 0xFFFF);
	return (-1);
    }
    else
	return (int) (r.edx & 0x1000) >> 12;
}
